package org.hildan.chrome.devtools.domains.layertree

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.dom.Rect

/**
 * Unique Layer identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#type-LayerId)
 */
public typealias LayerId = String

/**
 * Unique snapshot identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#type-SnapshotId)
 */
public typealias SnapshotId = String

/**
 * Rectangle where scrolling happens on the main thread.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#type-ScrollRect)
 */
@Serializable
public data class ScrollRect(
  /**
   * Rectangle itself.
   */
  public val rect: Rect,
  /**
   * Reason for rectangle to force scrolling on the main thread
   */
  public val type: String
)

/**
 * Sticky position constraints.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#type-StickyPositionConstraint)
 */
@Serializable
public data class StickyPositionConstraint(
  /**
   * Layout rectangle of the sticky element before being shifted
   */
  public val stickyBoxRect: Rect,
  /**
   * Layout rectangle of the containing block of the sticky element
   */
  public val containingBlockRect: Rect,
  /**
   * The nearest sticky layer that shifts the sticky box
   */
  public val nearestLayerShiftingStickyBox: LayerId? = null,
  /**
   * The nearest sticky layer that shifts the containing block
   */
  public val nearestLayerShiftingContainingBlock: LayerId? = null
)

/**
 * Serialized fragment of layer picture along with its offset within the layer.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#type-PictureTile)
 */
@Serializable
public data class PictureTile(
  /**
   * Offset from owning layer left boundary
   */
  public val x: Double,
  /**
   * Offset from owning layer top boundary
   */
  public val y: Double,
  /**
   * Base64-encoded snapshot data. (Encoded as a base64 string when passed over JSON)
   */
  public val picture: String
)

/**
 * Information about a compositing layer.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#type-Layer)
 */
@Serializable
public data class Layer(
  /**
   * The unique id for this layer.
   */
  public val layerId: LayerId,
  /**
   * The id of parent (not present for root).
   */
  public val parentLayerId: LayerId? = null,
  /**
   * The backend id for the node associated with this layer.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * Offset from parent layer, X coordinate.
   */
  public val offsetX: Double,
  /**
   * Offset from parent layer, Y coordinate.
   */
  public val offsetY: Double,
  /**
   * Layer width.
   */
  public val width: Double,
  /**
   * Layer height.
   */
  public val height: Double,
  /**
   * Transformation matrix for layer, default is identity matrix
   */
  public val transform: List<Double>? = null,
  /**
   * Transform anchor point X, absent if no transform specified
   */
  public val anchorX: Double? = null,
  /**
   * Transform anchor point Y, absent if no transform specified
   */
  public val anchorY: Double? = null,
  /**
   * Transform anchor point Z, absent if no transform specified
   */
  public val anchorZ: Double? = null,
  /**
   * Indicates how many time this layer has painted.
   */
  public val paintCount: Int,
  /**
   * Indicates whether this layer hosts any content, rather than being used for
   * transform/scrolling purposes only.
   */
  public val drawsContent: Boolean,
  /**
   * Set if layer is not visible.
   */
  public val invisible: Boolean? = null,
  /**
   * Rectangles scrolling on main thread only.
   */
  public val scrollRects: List<ScrollRect>? = null,
  /**
   * Sticky position constraint information
   */
  public val stickyPositionConstraint: StickyPositionConstraint? = null
)

/**
 * Array of timings, one per paint step.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#type-PaintProfile)
 */
public typealias PaintProfile = List<Double>
