package org.hildan.chrome.devtools.domains.performancetimeline

import kotlin.Boolean
import kotlin.Double
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.dom.Rect
import org.hildan.chrome.devtools.domains.network.TimeSinceEpoch
import org.hildan.chrome.devtools.domains.page.FrameId

/**
 * See https://github.com/WICG/LargestContentfulPaint and largest_contentful_paint.idl
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/PerformanceTimeline/#type-LargestContentfulPaint)
 */
@Serializable
public data class LargestContentfulPaint(
  public val renderTime: TimeSinceEpoch,
  public val loadTime: TimeSinceEpoch,
  /**
   * The number of pixels being painted.
   */
  public val size: Double,
  /**
   * The id attribute of the element, if available.
   */
  public val elementId: String? = null,
  /**
   * The URL of the image (may be trimmed).
   */
  public val url: String? = null,
  public val nodeId: BackendNodeId? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/PerformanceTimeline/#type-LayoutShiftAttribution)
 */
@Serializable
public data class LayoutShiftAttribution(
  public val previousRect: Rect,
  public val currentRect: Rect,
  public val nodeId: BackendNodeId? = null
)

/**
 * See https://wicg.github.io/layout-instability/#sec-layout-shift and layout_shift.idl
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/PerformanceTimeline/#type-LayoutShift)
 */
@Serializable
public data class LayoutShift(
  /**
   * Score increment produced by this event.
   */
  public val value: Double,
  public val hadRecentInput: Boolean,
  public val lastInputTime: TimeSinceEpoch,
  public val sources: List<LayoutShiftAttribution>
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/PerformanceTimeline/#type-TimelineEvent)
 */
@Serializable
public data class TimelineEvent(
  /**
   * Identifies the frame that this event is related to. Empty for non-frame targets.
   */
  public val frameId: FrameId,
  /**
   * The event type, as specified in
   * https://w3c.github.io/performance-timeline/#dom-performanceentry-entrytype
   * This determines which of the optional "details" fiedls is present.
   */
  public val type: String,
  /**
   * Name may be empty depending on the type.
   */
  public val name: String,
  /**
   * Time in seconds since Epoch, monotonically increasing within document lifetime.
   */
  public val time: TimeSinceEpoch,
  /**
   * Event duration, if applicable.
   */
  public val duration: Double? = null,
  public val lcpDetails: LargestContentfulPaint? = null,
  public val layoutShiftDetails: LayoutShift? = null
)
