package org.hildan.chrome.devtools.domains.profiler

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.profiler.events.ProfilerEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Response type for the [ProfilerDomain.getBestEffortCoverage] command.
 */
@Serializable
public data class GetBestEffortCoverageResponse(
  /**
   * Coverage data for the current isolate.
   */
  public val result: List<ScriptCoverage>
)

/**
 * Request object containing input parameters for the [ProfilerDomain.setSamplingInterval] command.
 */
@Serializable
public data class SetSamplingIntervalRequest(
  /**
   * New sampling interval in microseconds.
   */
  public val interval: Int
)

/**
 * Request object containing input parameters for the [ProfilerDomain.startPreciseCoverage] command.
 */
@Serializable
public data class StartPreciseCoverageRequest(
  /**
   * Collect accurate call counts beyond simple 'covered' or 'not covered'.
   */
  public val callCount: Boolean? = null,
  /**
   * Collect block-based coverage.
   */
  public val detailed: Boolean? = null,
  /**
   * Allow the backend to send updates on its own initiative
   */
  public val allowTriggeredUpdates: Boolean? = null
)

/**
 * Response type for the [ProfilerDomain.startPreciseCoverage] command.
 */
@Serializable
public data class StartPreciseCoverageResponse(
  /**
   * Monotonically increasing time (in seconds) when the coverage update was taken in the backend.
   */
  public val timestamp: Double
)

/**
 * Response type for the [ProfilerDomain.stop] command.
 */
@Serializable
public data class StopResponse(
  /**
   * Recorded profile.
   */
  public val profile: Profile
)

/**
 * Response type for the [ProfilerDomain.takePreciseCoverage] command.
 */
@Serializable
public data class TakePreciseCoverageResponse(
  /**
   * Coverage data for the current isolate.
   */
  public val result: List<ScriptCoverage>,
  /**
   * Monotonically increasing time (in seconds) when the coverage update was taken in the backend.
   */
  public val timestamp: Double
)

/**
 * Response type for the [ProfilerDomain.takeTypeProfile] command.
 */
@Serializable
@ExperimentalChromeApi
public data class TakeTypeProfileResponse(
  /**
   * Type profile for all scripts since startTypeProfile() was turned on.
   */
  public val result: List<ScriptTypeProfile>
)

/**
 * Response type for the [ProfilerDomain.getCounters] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetCountersResponse(
  /**
   * Collected counters information.
   */
  public val result: List<CounterInfo>
)

/**
 * Response type for the [ProfilerDomain.getRuntimeCallStats] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetRuntimeCallStatsResponse(
  /**
   * Collected runtime call counter information.
   */
  public val result: List<RuntimeCallCounterInfo>
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler)
 */
public class ProfilerDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out ProfilerEvent>> =
      mapOf(
      "Profiler.consoleProfileFinished" to serializer<ProfilerEvent.ConsoleProfileFinishedEvent>(),
      "Profiler.consoleProfileStarted" to serializer<ProfilerEvent.ConsoleProfileStartedEvent>(),
      "Profiler.preciseCoverageDeltaUpdate" to
          serializer<ProfilerEvent.PreciseCoverageDeltaUpdateEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<ProfilerEvent> = session.events(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#event-consoleProfileFinished)
   */
  public fun consoleProfileFinished(): Flow<ProfilerEvent.ConsoleProfileFinishedEvent> =
      session.events("Profiler.consoleProfileFinished")

  /**
   * Sent when new profile recording is started using console.profile() call.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#event-consoleProfileStarted)
   */
  public fun consoleProfileStarted(): Flow<ProfilerEvent.ConsoleProfileStartedEvent> =
      session.events("Profiler.consoleProfileStarted")

  /**
   * Reports coverage delta since the last poll (either from an event like this, or from
   * `takePreciseCoverage` for the current isolate. May only be sent if precise code
   * coverage has been started. This event can be trigged by the embedder to, for example,
   * trigger collection of coverage data immediatelly at a certain point in time.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#event-preciseCoverageDeltaUpdate)
   */
  @ExperimentalChromeApi
  public fun preciseCoverageDeltaUpdate(): Flow<ProfilerEvent.PreciseCoverageDeltaUpdateEvent> =
      session.events("Profiler.preciseCoverageDeltaUpdate")

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Profiler.disable", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("Profiler.enable", Unit)

  /**
   * Collect coverage data for the current isolate. The coverage data may be incomplete due to
   * garbage collection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-getBestEffortCoverage)
   */
  public suspend fun getBestEffortCoverage(): GetBestEffortCoverageResponse =
      session.request("Profiler.getBestEffortCoverage", Unit)

  /**
   * Changes CPU profiler sampling interval. Must be called before CPU profiles recording started.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-setSamplingInterval)
   */
  public suspend fun setSamplingInterval(input: SetSamplingIntervalRequest): Unit =
      session.request("Profiler.setSamplingInterval", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-start)
   */
  public suspend fun start(): Unit = session.request("Profiler.start", Unit)

  /**
   * Enable precise code coverage. Coverage data for JavaScript executed before enabling precise
   * code
   * coverage may be incomplete. Enabling prevents running optimized code and resets execution
   * counters.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-startPreciseCoverage)
   */
  public suspend fun startPreciseCoverage(input: StartPreciseCoverageRequest):
      StartPreciseCoverageResponse = session.request("Profiler.startPreciseCoverage", input)

  /**
   * Enable type profile.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-startTypeProfile)
   */
  @ExperimentalChromeApi
  public suspend fun startTypeProfile(): Unit = session.request("Profiler.startTypeProfile", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-stop)
   */
  public suspend fun stop(): StopResponse = session.request("Profiler.stop", Unit)

  /**
   * Disable precise code coverage. Disabling releases unnecessary execution count records and
   * allows
   * executing optimized code.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-stopPreciseCoverage)
   */
  public suspend fun stopPreciseCoverage(): Unit = session.request("Profiler.stopPreciseCoverage",
      Unit)

  /**
   * Disable type profile. Disabling releases type profile data collected so far.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-stopTypeProfile)
   */
  @ExperimentalChromeApi
  public suspend fun stopTypeProfile(): Unit = session.request("Profiler.stopTypeProfile", Unit)

  /**
   * Collect coverage data for the current isolate, and resets execution counters. Precise code
   * coverage needs to have started.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-takePreciseCoverage)
   */
  public suspend fun takePreciseCoverage(): TakePreciseCoverageResponse =
      session.request("Profiler.takePreciseCoverage", Unit)

  /**
   * Collect type profile.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-takeTypeProfile)
   */
  @ExperimentalChromeApi
  public suspend fun takeTypeProfile(): TakeTypeProfileResponse =
      session.request("Profiler.takeTypeProfile", Unit)

  /**
   * Enable counters collection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-enableCounters)
   */
  @ExperimentalChromeApi
  public suspend fun enableCounters(): Unit = session.request("Profiler.enableCounters", Unit)

  /**
   * Disable counters collection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-disableCounters)
   */
  @ExperimentalChromeApi
  public suspend fun disableCounters(): Unit = session.request("Profiler.disableCounters", Unit)

  /**
   * Retrieve counters.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-getCounters)
   */
  @ExperimentalChromeApi
  public suspend fun getCounters(): GetCountersResponse = session.request("Profiler.getCounters",
      Unit)

  /**
   * Enable run time call stats collection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-enableRuntimeCallStats)
   */
  @ExperimentalChromeApi
  public suspend fun enableRuntimeCallStats(): Unit =
      session.request("Profiler.enableRuntimeCallStats", Unit)

  /**
   * Disable run time call stats collection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-disableRuntimeCallStats)
   */
  @ExperimentalChromeApi
  public suspend fun disableRuntimeCallStats(): Unit =
      session.request("Profiler.disableRuntimeCallStats", Unit)

  /**
   * Retrieve run time call stats.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Profiler/#method-getRuntimeCallStats)
   */
  @ExperimentalChromeApi
  public suspend fun getRuntimeCallStats(): GetRuntimeCallStatsResponse =
      session.request("Profiler.getRuntimeCallStats", Unit)
}
