package org.hildan.chrome.devtools.domains.systeminfo

import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement

/**
 * Describes a single graphics processor (GPU).
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/SystemInfo/#type-GPUDevice)
 */
@Serializable
public data class GPUDevice(
  /**
   * PCI ID of the GPU vendor, if available; 0 otherwise.
   */
  public val vendorId: Double,
  /**
   * PCI ID of the GPU device, if available; 0 otherwise.
   */
  public val deviceId: Double,
  /**
   * Sub sys ID of the GPU, only available on Windows.
   */
  public val subSysId: Double? = null,
  /**
   * Revision of the GPU, only available on Windows.
   */
  public val revision: Double? = null,
  /**
   * String description of the GPU vendor, if the PCI ID is not available.
   */
  public val vendorString: String,
  /**
   * String description of the GPU device, if the PCI ID is not available.
   */
  public val deviceString: String,
  /**
   * String description of the GPU driver vendor.
   */
  public val driverVendor: String,
  /**
   * String description of the GPU driver version.
   */
  public val driverVersion: String
)

/**
 * Describes the width and height dimensions of an entity.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/SystemInfo/#type-Size)
 */
@Serializable
public data class Size(
  /**
   * Width in pixels.
   */
  public val width: Int,
  /**
   * Height in pixels.
   */
  public val height: Int
)

/**
 * Describes a supported video decoding profile with its associated minimum and
 * maximum resolutions.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/SystemInfo/#type-VideoDecodeAcceleratorCapability)
 */
@Serializable
public data class VideoDecodeAcceleratorCapability(
  /**
   * Video codec profile that is supported, e.g. VP9 Profile 2.
   */
  public val profile: String,
  /**
   * Maximum video dimensions in pixels supported for this |profile|.
   */
  public val maxResolution: Size,
  /**
   * Minimum video dimensions in pixels supported for this |profile|.
   */
  public val minResolution: Size
)

/**
 * Describes a supported video encoding profile with its associated maximum
 * resolution and maximum framerate.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/SystemInfo/#type-VideoEncodeAcceleratorCapability)
 */
@Serializable
public data class VideoEncodeAcceleratorCapability(
  /**
   * Video codec profile that is supported, e.g H264 Main.
   */
  public val profile: String,
  /**
   * Maximum video dimensions in pixels supported for this |profile|.
   */
  public val maxResolution: Size,
  /**
   * Maximum encoding framerate in frames per second supported for this
   * |profile|, as fraction's numerator and denominator, e.g. 24/1 fps,
   * 24000/1001 fps, etc.
   */
  public val maxFramerateNumerator: Int,
  public val maxFramerateDenominator: Int
)

/**
 * YUV subsampling type of the pixels of a given image.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/SystemInfo/#type-SubsamplingFormat)
 */
@Serializable
public enum class SubsamplingFormat {
  yuv420,
  yuv422,
  yuv444,
}

/**
 * Image format of a given image.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/SystemInfo/#type-ImageType)
 */
@Serializable
public enum class ImageType {
  jpeg,
  webp,
  unknown,
}

/**
 * Describes a supported image decoding profile with its associated minimum and
 * maximum resolutions and subsampling.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/SystemInfo/#type-ImageDecodeAcceleratorCapability)
 */
@Serializable
public data class ImageDecodeAcceleratorCapability(
  /**
   * Image coded, e.g. Jpeg.
   */
  public val imageType: ImageType,
  /**
   * Maximum supported dimensions of the image in pixels.
   */
  public val maxDimensions: Size,
  /**
   * Minimum supported dimensions of the image in pixels.
   */
  public val minDimensions: Size,
  /**
   * Optional array of supported subsampling formats, e.g. 4:2:0, if known.
   */
  public val subsamplings: List<SubsamplingFormat>
)

/**
 * Provides information about the GPU(s) on the system.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/SystemInfo/#type-GPUInfo)
 */
@Serializable
public data class GPUInfo(
  /**
   * The graphics devices on the system. Element 0 is the primary GPU.
   */
  public val devices: List<GPUDevice>,
  /**
   * An optional dictionary of additional GPU related attributes.
   */
  public val auxAttributes: JsonElement? = null,
  /**
   * An optional dictionary of graphics features and their status.
   */
  public val featureStatus: JsonElement? = null,
  /**
   * An optional array of GPU driver bug workarounds.
   */
  public val driverBugWorkarounds: List<String>,
  /**
   * Supported accelerated video decoding capabilities.
   */
  public val videoDecoding: List<VideoDecodeAcceleratorCapability>,
  /**
   * Supported accelerated video encoding capabilities.
   */
  public val videoEncoding: List<VideoEncodeAcceleratorCapability>,
  /**
   * Supported accelerated image decoding capabilities.
   */
  public val imageDecoding: List<ImageDecodeAcceleratorCapability>
)

/**
 * Represents process info.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/SystemInfo/#type-ProcessInfo)
 */
@Serializable
public data class ProcessInfo(
  /**
   * Specifies process type.
   */
  public val type: String,
  /**
   * Specifies process id.
   */
  public val id: Int,
  /**
   * Specifies cumulative CPU usage in seconds across all threads of the
   * process since the process start.
   */
  public val cpuTime: Double
)
