package org.hildan.chrome.devtools.domains.target

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.browser.BrowserContextID
import org.hildan.chrome.devtools.domains.target.events.TargetEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [TargetDomain.activateTarget] command.
 */
@Serializable
public data class ActivateTargetRequest(
  public val targetId: TargetID
)

/**
 * Request object containing input parameters for the [TargetDomain.attachToTarget] command.
 */
@Serializable
public data class AttachToTargetRequest(
  public val targetId: TargetID,
  /**
   * Enables "flat" access to the session via specifying sessionId attribute in the commands.
   * We plan to make this the default, deprecate non-flattened mode,
   * and eventually retire it. See crbug.com/991325.
   */
  public val flatten: Boolean? = null
)

/**
 * Response type for the [TargetDomain.attachToTarget] command.
 */
@Serializable
public data class AttachToTargetResponse(
  /**
   * Id assigned to the session.
   */
  public val sessionId: SessionID
)

/**
 * Response type for the [TargetDomain.attachToBrowserTarget] command.
 */
@Serializable
@ExperimentalChromeApi
public data class AttachToBrowserTargetResponse(
  /**
   * Id assigned to the session.
   */
  public val sessionId: SessionID
)

/**
 * Request object containing input parameters for the [TargetDomain.closeTarget] command.
 */
@Serializable
public data class CloseTargetRequest(
  public val targetId: TargetID
)

/**
 * Response type for the [TargetDomain.closeTarget] command.
 */
@Serializable
public data class CloseTargetResponse(
  /**
   * Always set to true. If an error occurs, the response indicates protocol error.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val success: Boolean
)

/**
 * Request object containing input parameters for the [TargetDomain.exposeDevToolsProtocol] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ExposeDevToolsProtocolRequest(
  public val targetId: TargetID,
  /**
   * Binding name, 'cdp' if not specified.
   */
  public val bindingName: String? = null
)

/**
 * Request object containing input parameters for the [TargetDomain.createBrowserContext] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CreateBrowserContextRequest(
  /**
   * If specified, disposes this context when debugging session disconnects.
   */
  public val disposeOnDetach: Boolean? = null,
  /**
   * Proxy server, similar to the one passed to --proxy-server
   */
  public val proxyServer: String? = null,
  /**
   * Proxy bypass list, similar to the one passed to --proxy-bypass-list
   */
  public val proxyBypassList: String? = null
)

/**
 * Response type for the [TargetDomain.createBrowserContext] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CreateBrowserContextResponse(
  /**
   * The id of the context created.
   */
  public val browserContextId: BrowserContextID
)

/**
 * Response type for the [TargetDomain.getBrowserContexts] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetBrowserContextsResponse(
  /**
   * An array of browser context ids.
   */
  public val browserContextIds: List<BrowserContextID>
)

/**
 * Request object containing input parameters for the [TargetDomain.createTarget] command.
 */
@Serializable
public data class CreateTargetRequest(
  /**
   * The initial URL the page will be navigated to.
   */
  public val url: String,
  /**
   * Frame width in DIP (headless chrome only).
   */
  public val width: Int? = null,
  /**
   * Frame height in DIP (headless chrome only).
   */
  public val height: Int? = null,
  /**
   * The browser context to create the page in.
   */
  public val browserContextId: BrowserContextID? = null,
  /**
   * Whether BeginFrames for this target will be controlled via DevTools (headless chrome only,
   * not supported on MacOS yet, false by default).
   */
  @ExperimentalChromeApi
  public val enableBeginFrameControl: Boolean? = null,
  /**
   * Whether to create a new Window or Tab (chrome-only, false by default).
   */
  public val newWindow: Boolean? = null,
  /**
   * Whether to create the target in background or foreground (chrome-only,
   * false by default).
   */
  public val background: Boolean? = null
)

/**
 * Response type for the [TargetDomain.createTarget] command.
 */
@Serializable
public data class CreateTargetResponse(
  /**
   * The id of the page opened.
   */
  public val targetId: TargetID
)

/**
 * Request object containing input parameters for the [TargetDomain.detachFromTarget] command.
 */
@Serializable
public data class DetachFromTargetRequest(
  /**
   * Session to detach.
   */
  public val sessionId: SessionID? = null,
  /**
   * Deprecated.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val targetId: TargetID? = null
)

/**
 * Request object containing input parameters for the [TargetDomain.disposeBrowserContext] command.
 */
@Serializable
@ExperimentalChromeApi
public data class DisposeBrowserContextRequest(
  public val browserContextId: BrowserContextID
)

/**
 * Request object containing input parameters for the [TargetDomain.getTargetInfo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetTargetInfoRequest(
  public val targetId: TargetID? = null
)

/**
 * Response type for the [TargetDomain.getTargetInfo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetTargetInfoResponse(
  public val targetInfo: TargetInfo
)

/**
 * Response type for the [TargetDomain.getTargets] command.
 */
@Serializable
public data class GetTargetsResponse(
  /**
   * The list of targets.
   */
  public val targetInfos: List<TargetInfo>
)

/**
 * Request object containing input parameters for the [TargetDomain.sendMessageToTarget] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class SendMessageToTargetRequest(
  public val message: String,
  /**
   * Identifier of the session.
   */
  public val sessionId: SessionID? = null,
  /**
   * Deprecated.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val targetId: TargetID? = null
)

/**
 * Request object containing input parameters for the [TargetDomain.setAutoAttach] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetAutoAttachRequest(
  /**
   * Whether to auto-attach to related targets.
   */
  public val autoAttach: Boolean,
  /**
   * Whether to pause new targets when attaching to them. Use `Runtime.runIfWaitingForDebugger`
   * to run paused targets.
   */
  public val waitForDebuggerOnStart: Boolean,
  /**
   * Enables "flat" access to the session via specifying sessionId attribute in the commands.
   * We plan to make this the default, deprecate non-flattened mode,
   * and eventually retire it. See crbug.com/991325.
   */
  public val flatten: Boolean? = null
)

/**
 * Request object containing input parameters for the [TargetDomain.setDiscoverTargets] command.
 */
@Serializable
public data class SetDiscoverTargetsRequest(
  /**
   * Whether to discover available targets.
   */
  public val discover: Boolean
)

/**
 * Request object containing input parameters for the [TargetDomain.setRemoteLocations] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetRemoteLocationsRequest(
  /**
   * List of remote locations.
   */
  public val locations: List<RemoteLocation>
)

/**
 * Supports additional targets discovery and allows to attach to them.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target)
 */
public class TargetDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out TargetEvent>> =
      mapOf(
      "Target.attachedToTarget" to serializer<TargetEvent.AttachedToTargetEvent>(),
      "Target.detachedFromTarget" to serializer<TargetEvent.DetachedFromTargetEvent>(),
      "Target.receivedMessageFromTarget" to
          serializer<TargetEvent.ReceivedMessageFromTargetEvent>(),
      "Target.targetCreated" to serializer<TargetEvent.TargetCreatedEvent>(),
      "Target.targetDestroyed" to serializer<TargetEvent.TargetDestroyedEvent>(),
      "Target.targetCrashed" to serializer<TargetEvent.TargetCrashedEvent>(),
      "Target.targetInfoChanged" to serializer<TargetEvent.TargetInfoChangedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<TargetEvent> = session.events(deserializersByEventName)

  /**
   * Issued when attached to target because of auto-attach or `attachToTarget` command.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-attachedToTarget)
   */
  @ExperimentalChromeApi
  public fun attachedToTarget(): Flow<TargetEvent.AttachedToTargetEvent> =
      session.events("Target.attachedToTarget")

  /**
   * Issued when detached from target for any reason (including `detachFromTarget` command). Can be
   * issued multiple times per target if multiple sessions have been attached to it.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-detachedFromTarget)
   */
  @ExperimentalChromeApi
  public fun detachedFromTarget(): Flow<TargetEvent.DetachedFromTargetEvent> =
      session.events("Target.detachedFromTarget")

  /**
   * Notifies about a new protocol message received from the session (as reported in
   * `attachedToTarget` event).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-receivedMessageFromTarget)
   */
  public fun receivedMessageFromTarget(): Flow<TargetEvent.ReceivedMessageFromTargetEvent> =
      session.events("Target.receivedMessageFromTarget")

  /**
   * Issued when a possible inspection target is created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetCreated)
   */
  public fun targetCreated(): Flow<TargetEvent.TargetCreatedEvent> =
      session.events("Target.targetCreated")

  /**
   * Issued when a target is destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetDestroyed)
   */
  public fun targetDestroyed(): Flow<TargetEvent.TargetDestroyedEvent> =
      session.events("Target.targetDestroyed")

  /**
   * Issued when a target has crashed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetCrashed)
   */
  public fun targetCrashed(): Flow<TargetEvent.TargetCrashedEvent> =
      session.events("Target.targetCrashed")

  /**
   * Issued when some information about a target has changed. This only happens between
   * `targetCreated` and `targetDestroyed`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetInfoChanged)
   */
  public fun targetInfoChanged(): Flow<TargetEvent.TargetInfoChangedEvent> =
      session.events("Target.targetInfoChanged")

  /**
   * Activates (focuses) the target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-activateTarget)
   */
  public suspend fun activateTarget(input: ActivateTargetRequest): Unit =
      session.request("Target.activateTarget", input)

  /**
   * Attaches to the target with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-attachToTarget)
   */
  public suspend fun attachToTarget(input: AttachToTargetRequest): AttachToTargetResponse =
      session.request("Target.attachToTarget", input)

  /**
   * Attaches to the browser target, only uses flat sessionId mode.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-attachToBrowserTarget)
   */
  @ExperimentalChromeApi
  public suspend fun attachToBrowserTarget(): AttachToBrowserTargetResponse =
      session.request("Target.attachToBrowserTarget", Unit)

  /**
   * Closes the target. If the target is a page that gets closed too.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-closeTarget)
   */
  public suspend fun closeTarget(input: CloseTargetRequest): CloseTargetResponse =
      session.request("Target.closeTarget", input)

  /**
   * Inject object to the target's main frame that provides a communication
   * channel with browser target.
   *
   * Injected object will be available as `window[bindingName]`.
   *
   * The object has the follwing API:
   * - `binding.send(json)` - a method to send messages over the remote debugging protocol
   * - `binding.onmessage = json => handleMessage(json)` - a callback that will be called for the
   * protocol notifications and command responses.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-exposeDevToolsProtocol)
   */
  @ExperimentalChromeApi
  public suspend fun exposeDevToolsProtocol(input: ExposeDevToolsProtocolRequest): Unit =
      session.request("Target.exposeDevToolsProtocol", input)

  /**
   * Creates a new empty BrowserContext. Similar to an incognito profile but you can have more than
   * one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-createBrowserContext)
   */
  @ExperimentalChromeApi
  public suspend fun createBrowserContext(input: CreateBrowserContextRequest):
      CreateBrowserContextResponse = session.request("Target.createBrowserContext", input)

  /**
   * Returns all browser contexts created with `Target.createBrowserContext` method.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-getBrowserContexts)
   */
  @ExperimentalChromeApi
  public suspend fun getBrowserContexts(): GetBrowserContextsResponse =
      session.request("Target.getBrowserContexts", Unit)

  /**
   * Creates a new page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-createTarget)
   */
  public suspend fun createTarget(input: CreateTargetRequest): CreateTargetResponse =
      session.request("Target.createTarget", input)

  /**
   * Detaches session with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-detachFromTarget)
   */
  public suspend fun detachFromTarget(input: DetachFromTargetRequest): Unit =
      session.request("Target.detachFromTarget", input)

  /**
   * Deletes a BrowserContext. All the belonging pages will be closed without calling their
   * beforeunload hooks.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-disposeBrowserContext)
   */
  @ExperimentalChromeApi
  public suspend fun disposeBrowserContext(input: DisposeBrowserContextRequest): Unit =
      session.request("Target.disposeBrowserContext", input)

  /**
   * Returns information about a target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-getTargetInfo)
   */
  @ExperimentalChromeApi
  public suspend fun getTargetInfo(input: GetTargetInfoRequest): GetTargetInfoResponse =
      session.request("Target.getTargetInfo", input)

  /**
   * Retrieves a list of available targets.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-getTargets)
   */
  public suspend fun getTargets(): GetTargetsResponse = session.request("Target.getTargets", Unit)

  /**
   * Sends protocol message over session with given id.
   * Consider using flat mode instead; see commands attachToTarget, setAutoAttach,
   * and crbug.com/991325.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-sendMessageToTarget)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun sendMessageToTarget(input: SendMessageToTargetRequest): Unit =
      session.request("Target.sendMessageToTarget", input)

  /**
   * Controls whether to automatically attach to new targets which are considered to be related to
   * this one. When turned on, attaches to all existing related targets as well. When turned off,
   * automatically detaches from all currently attached targets.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-setAutoAttach)
   */
  @ExperimentalChromeApi
  public suspend fun setAutoAttach(input: SetAutoAttachRequest): Unit =
      session.request("Target.setAutoAttach", input)

  /**
   * Controls whether to discover available targets and notify via
   * `targetCreated/targetInfoChanged/targetDestroyed` events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-setDiscoverTargets)
   */
  public suspend fun setDiscoverTargets(input: SetDiscoverTargetsRequest): Unit =
      session.request("Target.setDiscoverTargets", input)

  /**
   * Enables target discovery for the specified locations, when `setDiscoverTargets` was set to
   * `true`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-setRemoteLocations)
   */
  @ExperimentalChromeApi
  public suspend fun setRemoteLocations(input: SetRemoteLocationsRequest): Unit =
      session.request("Target.setRemoteLocations", input)
}
