package org.hildan.chrome.devtools.domains.audits

import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.network.RequestId
import org.hildan.chrome.devtools.domains.page.FrameId

/**
 * Information about a cookie that is affected by an inspector issue.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-AffectedCookie)
 */
@Serializable
public data class AffectedCookie(
  /**
   * The following three properties uniquely identify a cookie
   */
  public val name: String,
  public val path: String,
  public val domain: String
)

/**
 * Information about a request that is affected by an inspector issue.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-AffectedRequest)
 */
@Serializable
public data class AffectedRequest(
  /**
   * The unique request id.
   */
  public val requestId: RequestId,
  public val url: String? = null
)

/**
 * Information about the frame affected by an inspector issue.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-AffectedFrame)
 */
@Serializable
public data class AffectedFrame(
  public val frameId: FrameId
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-SameSiteCookieExclusionReason)
 */
@Serializable
public enum class SameSiteCookieExclusionReason {
  ExcludeSameSiteUnspecifiedTreatedAsLax,
  ExcludeSameSiteNoneInsecure,
  ExcludeSameSiteLax,
  ExcludeSameSiteStrict,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-SameSiteCookieWarningReason)
 */
@Serializable
public enum class SameSiteCookieWarningReason {
  WarnSameSiteUnspecifiedCrossSiteContext,
  WarnSameSiteNoneInsecure,
  WarnSameSiteUnspecifiedLaxAllowUnsafe,
  WarnSameSiteStrictLaxDowngradeStrict,
  WarnSameSiteStrictCrossDowngradeStrict,
  WarnSameSiteStrictCrossDowngradeLax,
  WarnSameSiteLaxCrossDowngradeStrict,
  WarnSameSiteLaxCrossDowngradeLax,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-SameSiteCookieOperation)
 */
@Serializable
public enum class SameSiteCookieOperation {
  SetCookie,
  ReadCookie,
}

/**
 * This information is currently necessary, as the front-end has a difficult
 * time finding a specific cookie. With this, we can convey specific error
 * information without the cookie.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-SameSiteCookieIssueDetails)
 */
@Serializable
public data class SameSiteCookieIssueDetails(
  public val cookie: AffectedCookie,
  public val cookieWarningReasons: List<SameSiteCookieWarningReason>,
  public val cookieExclusionReasons: List<SameSiteCookieExclusionReason>,
  /**
   * Optionally identifies the site-for-cookies and the cookie url, which
   * may be used by the front-end as additional context.
   */
  public val operation: SameSiteCookieOperation,
  public val siteForCookies: String? = null,
  public val cookieUrl: String? = null,
  public val request: AffectedRequest? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-MixedContentResolutionStatus)
 */
@Serializable
public enum class MixedContentResolutionStatus {
  MixedContentBlocked,
  MixedContentAutomaticallyUpgraded,
  MixedContentWarning,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-MixedContentResourceType)
 */
@Serializable
public enum class MixedContentResourceType {
  Audio,
  Beacon,
  CSPReport,
  Download,
  EventSource,
  Favicon,
  Font,
  Form,
  Frame,
  Image,
  Import,
  Manifest,
  Ping,
  PluginData,
  PluginResource,
  Prefetch,
  Resource,
  Script,
  ServiceWorker,
  SharedWorker,
  Stylesheet,
  Track,
  Video,
  Worker,
  XMLHttpRequest,
  XSLT,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-MixedContentIssueDetails)
 */
@Serializable
public data class MixedContentIssueDetails(
  /**
   * The type of resource causing the mixed content issue (css, js, iframe,
   * form,...). Marked as optional because it is mapped to from
   * blink::mojom::RequestContextType, which will be replaced
   * by network::mojom::RequestDestination
   */
  public val resourceType: MixedContentResourceType? = null,
  /**
   * The way the mixed content issue is being resolved.
   */
  public val resolutionStatus: MixedContentResolutionStatus,
  /**
   * The unsafe http url causing the mixed content issue.
   */
  public val insecureURL: String,
  /**
   * The url responsible for the call to an unsafe url.
   */
  public val mainResourceURL: String,
  /**
   * The mixed content request.
   * Does not always exist (e.g. for unsafe form submission urls).
   */
  public val request: AffectedRequest? = null,
  /**
   * Optional because not every mixed content issue is necessarily linked to a frame.
   */
  public val frame: AffectedFrame? = null
)

/**
 * Enum indicating the reason a response has been blocked. These reasons are
 * refinements of the net error BLOCKED_BY_RESPONSE.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-BlockedByResponseReason)
 */
@Serializable
public enum class BlockedByResponseReason {
  CoepFrameResourceNeedsCoepHeader,
  CoopSandboxedIFrameCannotNavigateToCoopPage,
  CorpNotSameOrigin,
  CorpNotSameOriginAfterDefaultedToSameOriginByCoep,
  CorpNotSameSite,
}

/**
 * Details for a request that has been blocked with the BLOCKED_BY_RESPONSE
 * code. Currently only used for COEP/COOP, but may be extended to include
 * some CSP errors in the future.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-BlockedByResponseIssueDetails)
 */
@Serializable
public data class BlockedByResponseIssueDetails(
  public val request: AffectedRequest,
  public val parentFrame: AffectedFrame? = null,
  public val blockedFrame: AffectedFrame? = null,
  public val reason: BlockedByResponseReason
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-HeavyAdResolutionStatus)
 */
@Serializable
public enum class HeavyAdResolutionStatus {
  HeavyAdBlocked,
  HeavyAdWarning,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-HeavyAdReason)
 */
@Serializable
public enum class HeavyAdReason {
  NetworkTotalLimit,
  CpuTotalLimit,
  CpuPeakLimit,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-HeavyAdIssueDetails)
 */
@Serializable
public data class HeavyAdIssueDetails(
  /**
   * The resolution status, either blocking the content or warning.
   */
  public val resolution: HeavyAdResolutionStatus,
  /**
   * The reason the ad was blocked, total network or cpu or peak cpu.
   */
  public val reason: HeavyAdReason,
  /**
   * The frame that was blocked.
   */
  public val frame: AffectedFrame
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-ContentSecurityPolicyViolationType)
 */
@Serializable
public enum class ContentSecurityPolicyViolationType {
  kInlineViolation,
  kEvalViolation,
  kURLViolation,
  kTrustedTypesSinkViolation,
  kTrustedTypesPolicyViolation,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-SourceCodeLocation)
 */
@Serializable
public data class SourceCodeLocation(
  public val url: String,
  public val lineNumber: Int,
  public val columnNumber: Int
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-ContentSecurityPolicyIssueDetails)
 */
@Serializable
public data class ContentSecurityPolicyIssueDetails(
  /**
   * The url not included in allowed sources.
   */
  public val blockedURL: String? = null,
  /**
   * Specific directive that is violated, causing the CSP issue.
   */
  public val violatedDirective: String,
  public val isReportOnly: Boolean,
  public val contentSecurityPolicyViolationType: ContentSecurityPolicyViolationType,
  public val frameAncestor: AffectedFrame? = null,
  public val sourceCodeLocation: SourceCodeLocation? = null,
  public val violatingNodeId: BackendNodeId? = null
)

/**
 * Details for a request that has been blocked with the BLOCKED_BY_RESPONSE
 * code. Currently only used for COEP/COOP, but may be extended to include
 * some CSP errors in the future.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-SharedArrayBufferTransferIssueDetails)
 */
@Serializable
public data class SharedArrayBufferTransferIssueDetails(
  public val sourceCodeLocation: SourceCodeLocation,
  public val isWarning: Boolean
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-TwaQualityEnforcementViolationType)
 */
@Serializable
public enum class TwaQualityEnforcementViolationType {
  kHttpError,
  kUnavailableOffline,
  kDigitalAssetLinks,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-TrustedWebActivityIssueDetails)
 */
@Serializable
public data class TrustedWebActivityIssueDetails(
  /**
   * The url that triggers the violation.
   */
  public val url: String,
  public val violationType: TwaQualityEnforcementViolationType,
  public val httpStatusCode: Int? = null,
  /**
   * The package name of the Trusted Web Activity client app. This field is
   * only used when violation type is kDigitalAssetLinks.
   */
  public val packageName: String? = null,
  /**
   * The signature of the Trusted Web Activity client app. This field is only
   * used when violation type is kDigitalAssetLinks.
   */
  public val signature: String? = null
)

/**
 * A unique identifier for the type of issue. Each type may use one of the
 * optional fields in InspectorIssueDetails to convey more specific
 * information about the kind of issue.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-InspectorIssueCode)
 */
@Serializable
public enum class InspectorIssueCode {
  SameSiteCookieIssue,
  MixedContentIssue,
  BlockedByResponseIssue,
  HeavyAdIssue,
  ContentSecurityPolicyIssue,
  SharedArrayBufferTransferIssue,
  TrustedWebActivityIssue,
}

/**
 * This struct holds a list of optional fields with additional information
 * specific to the kind of issue. When adding a new issue code, please also
 * add a new optional field to this type.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-InspectorIssueDetails)
 */
@Serializable
public data class InspectorIssueDetails(
  public val sameSiteCookieIssueDetails: SameSiteCookieIssueDetails? = null,
  public val mixedContentIssueDetails: MixedContentIssueDetails? = null,
  public val blockedByResponseIssueDetails: BlockedByResponseIssueDetails? = null,
  public val heavyAdIssueDetails: HeavyAdIssueDetails? = null,
  public val contentSecurityPolicyIssueDetails: ContentSecurityPolicyIssueDetails? = null,
  public val sharedArrayBufferTransferIssueDetails: SharedArrayBufferTransferIssueDetails? = null,
  public val twaQualityEnforcementDetails: TrustedWebActivityIssueDetails? = null
)

/**
 * An inspector issue reported from the back-end.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#type-InspectorIssue)
 */
@Serializable
public data class InspectorIssue(
  public val code: InspectorIssueCode,
  public val details: InspectorIssueDetails
)
