package org.hildan.chrome.devtools.domains.overlay

import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.dom.NodeId
import org.hildan.chrome.devtools.domains.dom.Quad
import org.hildan.chrome.devtools.domains.dom.RGBA
import org.hildan.chrome.devtools.domains.overlay.events.OverlayEvent
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [OverlayDomain.getHighlightObjectForTest]
 * command.
 */
@Serializable
public data class GetHighlightObjectForTestRequest(
  /**
   * Id of the node to get highlight object for.
   */
  public val nodeId: NodeId,
  /**
   * Whether to include distance info.
   */
  public val includeDistance: Boolean? = null,
  /**
   * Whether to include style info.
   */
  public val includeStyle: Boolean? = null,
  /**
   * The color format to get config with (default: hex).
   */
  public val colorFormat: ColorFormat? = null,
  /**
   * Whether to show accessibility info (default: true).
   */
  public val showAccessibilityInfo: Boolean? = null
)

/**
 * Response type for the [OverlayDomain.getHighlightObjectForTest] command.
 */
@Serializable
public data class GetHighlightObjectForTestResponse(
  /**
   * Highlight data for the node.
   */
  public val highlight: JsonElement
)

/**
 * Request object containing input parameters for the [OverlayDomain.getGridHighlightObjectsForTest]
 * command.
 */
@Serializable
public data class GetGridHighlightObjectsForTestRequest(
  /**
   * Ids of the node to get highlight object for.
   */
  public val nodeIds: List<NodeId>
)

/**
 * Response type for the [OverlayDomain.getGridHighlightObjectsForTest] command.
 */
@Serializable
public data class GetGridHighlightObjectsForTestResponse(
  /**
   * Grid Highlight data for the node ids provided.
   */
  public val highlights: JsonElement
)

/**
 * Request object containing input parameters for the
 * [OverlayDomain.getSourceOrderHighlightObjectForTest] command.
 */
@Serializable
public data class GetSourceOrderHighlightObjectForTestRequest(
  /**
   * Id of the node to highlight.
   */
  public val nodeId: NodeId
)

/**
 * Response type for the [OverlayDomain.getSourceOrderHighlightObjectForTest] command.
 */
@Serializable
public data class GetSourceOrderHighlightObjectForTestResponse(
  /**
   * Source order highlight data for the node id provided.
   */
  public val highlight: JsonElement
)

/**
 * Request object containing input parameters for the [OverlayDomain.highlightFrame] command.
 */
@Serializable
public data class HighlightFrameRequest(
  /**
   * Identifier of the frame to highlight.
   */
  public val frameId: FrameId,
  /**
   * The content box highlight fill color (default: transparent).
   */
  public val contentColor: RGBA? = null,
  /**
   * The content box highlight outline color (default: transparent).
   */
  public val contentOutlineColor: RGBA? = null
)

/**
 * Request object containing input parameters for the [OverlayDomain.highlightNode] command.
 */
@Serializable
public data class HighlightNodeRequest(
  /**
   * A descriptor for the highlight appearance.
   */
  public val highlightConfig: HighlightConfig,
  /**
   * Identifier of the node to highlight.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node to highlight.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node to be highlighted.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * Selectors to highlight relevant nodes.
   */
  public val selector: String? = null
)

/**
 * Request object containing input parameters for the [OverlayDomain.highlightQuad] command.
 */
@Serializable
public data class HighlightQuadRequest(
  /**
   * Quad to highlight
   */
  public val quad: Quad,
  /**
   * The highlight fill color (default: transparent).
   */
  public val color: RGBA? = null,
  /**
   * The highlight outline color (default: transparent).
   */
  public val outlineColor: RGBA? = null
)

/**
 * Request object containing input parameters for the [OverlayDomain.highlightRect] command.
 */
@Serializable
public data class HighlightRectRequest(
  /**
   * X coordinate
   */
  public val x: Int,
  /**
   * Y coordinate
   */
  public val y: Int,
  /**
   * Rectangle width
   */
  public val width: Int,
  /**
   * Rectangle height
   */
  public val height: Int,
  /**
   * The highlight fill color (default: transparent).
   */
  public val color: RGBA? = null,
  /**
   * The highlight outline color (default: transparent).
   */
  public val outlineColor: RGBA? = null
)

/**
 * Request object containing input parameters for the [OverlayDomain.highlightSourceOrder] command.
 */
@Serializable
public data class HighlightSourceOrderRequest(
  /**
   * A descriptor for the appearance of the overlay drawing.
   */
  public val sourceOrderConfig: SourceOrderConfig,
  /**
   * Identifier of the node to highlight.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node to highlight.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node to be highlighted.
   */
  public val objectId: RemoteObjectId? = null
)

/**
 * Request object containing input parameters for the [OverlayDomain.setInspectMode] command.
 */
@Serializable
public data class SetInspectModeRequest(
  /**
   * Set an inspection mode.
   */
  public val mode: InspectMode,
  /**
   * A descriptor for the highlight appearance of hovered-over nodes. May be omitted if `enabled
   * == false`.
   */
  public val highlightConfig: HighlightConfig? = null
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowAdHighlights] command.
 */
@Serializable
public data class SetShowAdHighlightsRequest(
  /**
   * True for showing ad highlights
   */
  public val show: Boolean
)

/**
 * Request object containing input parameters for the [OverlayDomain.setPausedInDebuggerMessage]
 * command.
 */
@Serializable
public data class SetPausedInDebuggerMessageRequest(
  /**
   * The message to display, also triggers resume and step over controls.
   */
  public val message: String? = null
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowDebugBorders] command.
 */
@Serializable
public data class SetShowDebugBordersRequest(
  /**
   * True for showing debug borders
   */
  public val show: Boolean
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowFPSCounter] command.
 */
@Serializable
public data class SetShowFPSCounterRequest(
  /**
   * True for showing the FPS counter
   */
  public val show: Boolean
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowGridOverlays] command.
 */
@Serializable
public data class SetShowGridOverlaysRequest(
  /**
   * An array of node identifiers and descriptors for the highlight appearance.
   */
  public val gridNodeHighlightConfigs: List<GridNodeHighlightConfig>
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowFlexOverlays] command.
 */
@Serializable
public data class SetShowFlexOverlaysRequest(
  /**
   * An array of node identifiers and descriptors for the highlight appearance.
   */
  public val flexNodeHighlightConfigs: List<FlexNodeHighlightConfig>
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowPaintRects] command.
 */
@Serializable
public data class SetShowPaintRectsRequest(
  /**
   * True for showing paint rectangles
   */
  public val result: Boolean
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowLayoutShiftRegions]
 * command.
 */
@Serializable
public data class SetShowLayoutShiftRegionsRequest(
  /**
   * True for showing layout shift regions
   */
  public val result: Boolean
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowScrollBottleneckRects]
 * command.
 */
@Serializable
public data class SetShowScrollBottleneckRectsRequest(
  /**
   * True for showing scroll bottleneck rects
   */
  public val show: Boolean
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowHitTestBorders] command.
 */
@Serializable
public data class SetShowHitTestBordersRequest(
  /**
   * True for showing hit-test borders
   */
  public val show: Boolean
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowWebVitals] command.
 */
@Serializable
public data class SetShowWebVitalsRequest(
  public val show: Boolean
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowViewportSizeOnResize]
 * command.
 */
@Serializable
public data class SetShowViewportSizeOnResizeRequest(
  /**
   * Whether to paint size or not.
   */
  public val show: Boolean
)

/**
 * Request object containing input parameters for the [OverlayDomain.setShowHinge] command.
 */
@Serializable
public data class SetShowHingeRequest(
  /**
   * hinge data, null means hideHinge
   */
  public val hingeConfig: HingeConfig? = null
)

/**
 * This domain provides various functionality related to drawing atop the inspected page.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay)
 */
@ExperimentalChromeApi
public class OverlayDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out OverlayEvent>> =
      mapOf(
      "Overlay.inspectNodeRequested" to serializer<OverlayEvent.InspectNodeRequestedEvent>(),
      "Overlay.nodeHighlightRequested" to serializer<OverlayEvent.NodeHighlightRequestedEvent>(),
      "Overlay.screenshotRequested" to serializer<OverlayEvent.ScreenshotRequestedEvent>(),
      "Overlay.inspectModeCanceled" to serializer<OverlayEvent.InspectModeCanceledEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<OverlayEvent> = session.events(deserializersByEventName)

  /**
   * Fired when the node should be inspected. This happens after call to `setInspectMode` or when
   * user manually inspects an element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#event-inspectNodeRequested)
   */
  public fun inspectNodeRequested(): Flow<OverlayEvent.InspectNodeRequestedEvent> =
      session.events("Overlay.inspectNodeRequested")

  /**
   * Fired when the node should be highlighted. This happens after call to `setInspectMode`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#event-nodeHighlightRequested)
   */
  public fun nodeHighlightRequested(): Flow<OverlayEvent.NodeHighlightRequestedEvent> =
      session.events("Overlay.nodeHighlightRequested")

  /**
   * Fired when user asks to capture screenshot of some area on the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#event-screenshotRequested)
   */
  public fun screenshotRequested(): Flow<OverlayEvent.ScreenshotRequestedEvent> =
      session.events("Overlay.screenshotRequested")

  /**
   * Fired when user cancels the inspect mode.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#event-inspectModeCanceled)
   */
  public fun inspectModeCanceled(): Flow<OverlayEvent.InspectModeCanceledEvent> =
      session.events("Overlay.inspectModeCanceled")

  /**
   * Disables domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Overlay.disable", Unit)

  /**
   * Enables domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("Overlay.enable", Unit)

  /**
   * For testing.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-getHighlightObjectForTest)
   */
  public suspend fun getHighlightObjectForTest(input: GetHighlightObjectForTestRequest):
      GetHighlightObjectForTestResponse = session.request("Overlay.getHighlightObjectForTest",
      input)

  /**
   * For Persistent Grid testing.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-getGridHighlightObjectsForTest)
   */
  public suspend fun getGridHighlightObjectsForTest(input: GetGridHighlightObjectsForTestRequest):
      GetGridHighlightObjectsForTestResponse =
      session.request("Overlay.getGridHighlightObjectsForTest", input)

  /**
   * For Source Order Viewer testing.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-getSourceOrderHighlightObjectForTest)
   */
  public suspend
      fun getSourceOrderHighlightObjectForTest(input: GetSourceOrderHighlightObjectForTestRequest):
      GetSourceOrderHighlightObjectForTestResponse =
      session.request("Overlay.getSourceOrderHighlightObjectForTest", input)

  /**
   * Hides any highlight.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-hideHighlight)
   */
  public suspend fun hideHighlight(): Unit = session.request("Overlay.hideHighlight", Unit)

  /**
   * Highlights owner element of the frame with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightFrame)
   */
  public suspend fun highlightFrame(input: HighlightFrameRequest): Unit =
      session.request("Overlay.highlightFrame", input)

  /**
   * Highlights DOM node with given id or with the given JavaScript object wrapper. Either nodeId or
   * objectId must be specified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightNode)
   */
  public suspend fun highlightNode(input: HighlightNodeRequest): Unit =
      session.request("Overlay.highlightNode", input)

  /**
   * Highlights given quad. Coordinates are absolute with respect to the main frame viewport.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightQuad)
   */
  public suspend fun highlightQuad(input: HighlightQuadRequest): Unit =
      session.request("Overlay.highlightQuad", input)

  /**
   * Highlights given rectangle. Coordinates are absolute with respect to the main frame viewport.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightRect)
   */
  public suspend fun highlightRect(input: HighlightRectRequest): Unit =
      session.request("Overlay.highlightRect", input)

  /**
   * Highlights the source order of the children of the DOM node with given id or with the given
   * JavaScript object wrapper. Either nodeId or objectId must be specified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-highlightSourceOrder)
   */
  public suspend fun highlightSourceOrder(input: HighlightSourceOrderRequest): Unit =
      session.request("Overlay.highlightSourceOrder", input)

  /**
   * Enters the 'inspect' mode. In this mode, elements that user is hovering over are highlighted.
   * Backend then generates 'inspectNodeRequested' event upon element selection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setInspectMode)
   */
  public suspend fun setInspectMode(input: SetInspectModeRequest): Unit =
      session.request("Overlay.setInspectMode", input)

  /**
   * Highlights owner element of all frames detected to be ads.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowAdHighlights)
   */
  public suspend fun setShowAdHighlights(input: SetShowAdHighlightsRequest): Unit =
      session.request("Overlay.setShowAdHighlights", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setPausedInDebuggerMessage)
   */
  public suspend fun setPausedInDebuggerMessage(input: SetPausedInDebuggerMessageRequest): Unit =
      session.request("Overlay.setPausedInDebuggerMessage", input)

  /**
   * Requests that backend shows debug borders on layers
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowDebugBorders)
   */
  public suspend fun setShowDebugBorders(input: SetShowDebugBordersRequest): Unit =
      session.request("Overlay.setShowDebugBorders", input)

  /**
   * Requests that backend shows the FPS counter
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowFPSCounter)
   */
  public suspend fun setShowFPSCounter(input: SetShowFPSCounterRequest): Unit =
      session.request("Overlay.setShowFPSCounter", input)

  /**
   * Highlight multiple elements with the CSS Grid overlay.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowGridOverlays)
   */
  public suspend fun setShowGridOverlays(input: SetShowGridOverlaysRequest): Unit =
      session.request("Overlay.setShowGridOverlays", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowFlexOverlays)
   */
  public suspend fun setShowFlexOverlays(input: SetShowFlexOverlaysRequest): Unit =
      session.request("Overlay.setShowFlexOverlays", input)

  /**
   * Requests that backend shows paint rectangles
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowPaintRects)
   */
  public suspend fun setShowPaintRects(input: SetShowPaintRectsRequest): Unit =
      session.request("Overlay.setShowPaintRects", input)

  /**
   * Requests that backend shows layout shift regions
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowLayoutShiftRegions)
   */
  public suspend fun setShowLayoutShiftRegions(input: SetShowLayoutShiftRegionsRequest): Unit =
      session.request("Overlay.setShowLayoutShiftRegions", input)

  /**
   * Requests that backend shows scroll bottleneck rects
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowScrollBottleneckRects)
   */
  public suspend fun setShowScrollBottleneckRects(input: SetShowScrollBottleneckRectsRequest): Unit
      = session.request("Overlay.setShowScrollBottleneckRects", input)

  /**
   * Requests that backend shows hit-test borders on layers
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowHitTestBorders)
   */
  public suspend fun setShowHitTestBorders(input: SetShowHitTestBordersRequest): Unit =
      session.request("Overlay.setShowHitTestBorders", input)

  /**
   * Request that backend shows an overlay with web vital metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowWebVitals)
   */
  public suspend fun setShowWebVitals(input: SetShowWebVitalsRequest): Unit =
      session.request("Overlay.setShowWebVitals", input)

  /**
   * Paints viewport size upon main frame resize.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowViewportSizeOnResize)
   */
  public suspend fun setShowViewportSizeOnResize(input: SetShowViewportSizeOnResizeRequest): Unit =
      session.request("Overlay.setShowViewportSizeOnResize", input)

  /**
   * Add a dual screen device hinge
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Overlay/#method-setShowHinge)
   */
  public suspend fun setShowHinge(input: SetShowHingeRequest): Unit =
      session.request("Overlay.setShowHinge", input)
}
