@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.audits

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.audits.events.AuditsEvent
import org.hildan.chrome.devtools.domains.network.RequestId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [AuditsDomain.getEncodedResponse] command.
 */
@Serializable
public data class GetEncodedResponseRequest(
  /**
   * Identifier of the network request to get content for.
   */
  public val requestId: RequestId,
  /**
   * The encoding to use.
   */
  public val encoding: String,
  /**
   * The quality of the encoding (0-1). (defaults to 1)
   */
  public val quality: Double? = null,
  /**
   * Whether to only return the size information (defaults to false).
   */
  public val sizeOnly: Boolean? = null
)

/**
 * Response type for the [AuditsDomain.getEncodedResponse] command.
 */
@Serializable
public data class GetEncodedResponseResponse(
  /**
   * The encoded body as a base64 string. Omitted if sizeOnly is true. (Encoded as a base64 string
   * when passed over JSON)
   */
  public val body: String? = null,
  /**
   * Size before re-encoding.
   */
  public val originalSize: Int,
  /**
   * Size after re-encoding.
   */
  public val encodedSize: Int
)

/**
 * Audits domain allows investigation of page violations and possible improvements.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits)
 */
@ExperimentalChromeApi
public class AuditsDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out AuditsEvent>> =
      mapOf(
      "Audits.issueAdded" to serializer<AuditsEvent.IssueAddedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<AuditsEvent> = session.events(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#event-issueAdded)
   */
  public fun issueAdded(): Flow<AuditsEvent.IssueAddedEvent> = session.events("Audits.issueAdded")

  /**
   * Returns the response body and size if it were re-encoded with the specified settings. Only
   * applies to images.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#method-getEncodedResponse)
   */
  public suspend fun getEncodedResponse(input: GetEncodedResponseRequest):
      GetEncodedResponseResponse = session.request("Audits.getEncodedResponse", input)

  /**
   * Disables issues domain, prevents further issues from being reported to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Audits.disable", Unit)

  /**
   * Enables issues domain, sends the issues collected so far to the client by means of the
   * `issueAdded` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Audits/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("Audits.enable", Unit)
}
