@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.emulation

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.dom.RGBA
import org.hildan.chrome.devtools.domains.emulation.events.EmulationEvent
import org.hildan.chrome.devtools.domains.network.TimeSinceEpoch
import org.hildan.chrome.devtools.domains.page.Viewport
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Response type for the [EmulationDomain.canEmulate] command.
 */
@Serializable
public data class CanEmulateResponse(
  /**
   * True if emulation is supported.
   */
  public val result: Boolean
)

/**
 * Request object containing input parameters for the [EmulationDomain.setFocusEmulationEnabled]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetFocusEmulationEnabledRequest(
  /**
   * Whether to enable to disable focus emulation.
   */
  public val enabled: Boolean
)

/**
 * Request object containing input parameters for the [EmulationDomain.setCPUThrottlingRate]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetCPUThrottlingRateRequest(
  /**
   * Throttling rate as a slowdown factor (1 is no throttle, 2 is 2x slowdown, etc).
   */
  public val rate: Double
)

/**
 * Request object containing input parameters for the
 * [EmulationDomain.setDefaultBackgroundColorOverride] command.
 */
@Serializable
public data class SetDefaultBackgroundColorOverrideRequest(
  /**
   * RGBA of the default background color. If not specified, any existing override will be
   * cleared.
   */
  public val color: RGBA? = null
)

/**
 * Request object containing input parameters for the [EmulationDomain.setDeviceMetricsOverride]
 * command.
 */
@Serializable
public data class SetDeviceMetricsOverrideRequest(
  /**
   * Overriding width value in pixels (minimum 0, maximum 10000000). 0 disables the override.
   */
  public val width: Int,
  /**
   * Overriding height value in pixels (minimum 0, maximum 10000000). 0 disables the override.
   */
  public val height: Int,
  /**
   * Overriding device scale factor value. 0 disables the override.
   */
  public val deviceScaleFactor: Double,
  /**
   * Whether to emulate mobile device. This includes viewport meta tag, overlay scrollbars, text
   * autosizing and more.
   */
  public val mobile: Boolean,
  /**
   * Scale to apply to resulting view image.
   */
  @ExperimentalChromeApi
  public val scale: Double? = null,
  /**
   * Overriding screen width value in pixels (minimum 0, maximum 10000000).
   */
  @ExperimentalChromeApi
  public val screenWidth: Int? = null,
  /**
   * Overriding screen height value in pixels (minimum 0, maximum 10000000).
   */
  @ExperimentalChromeApi
  public val screenHeight: Int? = null,
  /**
   * Overriding view X position on screen in pixels (minimum 0, maximum 10000000).
   */
  @ExperimentalChromeApi
  public val positionX: Int? = null,
  /**
   * Overriding view Y position on screen in pixels (minimum 0, maximum 10000000).
   */
  @ExperimentalChromeApi
  public val positionY: Int? = null,
  /**
   * Do not set visible view size, rely upon explicit setVisibleSize call.
   */
  @ExperimentalChromeApi
  public val dontSetVisibleSize: Boolean? = null,
  /**
   * Screen orientation override.
   */
  public val screenOrientation: ScreenOrientation? = null,
  /**
   * If set, the visible area of the page will be overridden to this viewport. This viewport
   * change is not observed by the page, e.g. viewport-relative elements do not change positions.
   */
  @ExperimentalChromeApi
  public val viewport: Viewport? = null,
  /**
   * If set, the display feature of a multi-segment screen. If not set, multi-segment support
   * is turned-off.
   */
  @ExperimentalChromeApi
  public val displayFeature: DisplayFeature? = null
)

/**
 * Request object containing input parameters for the [EmulationDomain.setScrollbarsHidden] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetScrollbarsHiddenRequest(
  /**
   * Whether scrollbars should be always hidden.
   */
  public val hidden: Boolean
)

/**
 * Request object containing input parameters for the [EmulationDomain.setDocumentCookieDisabled]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetDocumentCookieDisabledRequest(
  /**
   * Whether document.coookie API should be disabled.
   */
  public val disabled: Boolean
)

/**
 * Request object containing input parameters for the [EmulationDomain.setEmitTouchEventsForMouse]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetEmitTouchEventsForMouseRequest(
  /**
   * Whether touch emulation based on mouse input should be enabled.
   */
  public val enabled: Boolean,
  /**
   * Touch/gesture events configuration. Default: current platform.
   */
  public val configuration: String? = null
)

/**
 * Request object containing input parameters for the [EmulationDomain.setEmulatedMedia] command.
 */
@Serializable
public data class SetEmulatedMediaRequest(
  /**
   * Media type to emulate. Empty string disables the override.
   */
  public val media: String? = null,
  /**
   * Media features to emulate.
   */
  public val features: List<MediaFeature>? = null
)

/**
 * Request object containing input parameters for the [EmulationDomain.setEmulatedVisionDeficiency]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetEmulatedVisionDeficiencyRequest(
  /**
   * Vision deficiency to emulate.
   */
  public val type: String
)

/**
 * Request object containing input parameters for the [EmulationDomain.setGeolocationOverride]
 * command.
 */
@Serializable
public data class SetGeolocationOverrideRequest(
  /**
   * Mock latitude
   */
  public val latitude: Double? = null,
  /**
   * Mock longitude
   */
  public val longitude: Double? = null,
  /**
   * Mock accuracy
   */
  public val accuracy: Double? = null
)

/**
 * Request object containing input parameters for the [EmulationDomain.setIdleOverride] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetIdleOverrideRequest(
  /**
   * Mock isUserActive
   */
  public val isUserActive: Boolean,
  /**
   * Mock isScreenUnlocked
   */
  public val isScreenUnlocked: Boolean
)

/**
 * Request object containing input parameters for the [EmulationDomain.setNavigatorOverrides]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetNavigatorOverridesRequest(
  /**
   * The platform navigator.platform should return.
   */
  public val platform: String
)

/**
 * Request object containing input parameters for the [EmulationDomain.setPageScaleFactor] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetPageScaleFactorRequest(
  /**
   * Page scale factor.
   */
  public val pageScaleFactor: Double
)

/**
 * Request object containing input parameters for the [EmulationDomain.setScriptExecutionDisabled]
 * command.
 */
@Serializable
public data class SetScriptExecutionDisabledRequest(
  /**
   * Whether script execution should be disabled in the page.
   */
  public val value: Boolean
)

/**
 * Request object containing input parameters for the [EmulationDomain.setTouchEmulationEnabled]
 * command.
 */
@Serializable
public data class SetTouchEmulationEnabledRequest(
  /**
   * Whether the touch event emulation should be enabled.
   */
  public val enabled: Boolean,
  /**
   * Maximum touch points supported. Defaults to one.
   */
  public val maxTouchPoints: Int? = null
)

/**
 * Request object containing input parameters for the [EmulationDomain.setVirtualTimePolicy]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetVirtualTimePolicyRequest(
  public val policy: VirtualTimePolicy,
  /**
   * If set, after this many virtual milliseconds have elapsed virtual time will be paused and a
   * virtualTimeBudgetExpired event is sent.
   */
  public val budget: Double? = null,
  /**
   * If set this specifies the maximum number of tasks that can be run before virtual is forced
   * forwards to prevent deadlock.
   */
  public val maxVirtualTimeTaskStarvationCount: Int? = null,
  /**
   * If set the virtual time policy change should be deferred until any frame starts navigating.
   * Note any previous deferred policy change is superseded.
   */
  public val waitForNavigation: Boolean? = null,
  /**
   * If set, base::Time::Now will be overriden to initially return this value.
   */
  public val initialVirtualTime: TimeSinceEpoch? = null
)

/**
 * Response type for the [EmulationDomain.setVirtualTimePolicy] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetVirtualTimePolicyResponse(
  /**
   * Absolute timestamp at which virtual time was first enabled (up time in milliseconds).
   */
  public val virtualTimeTicksBase: Double
)

/**
 * Request object containing input parameters for the [EmulationDomain.setLocaleOverride] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetLocaleOverrideRequest(
  /**
   * ICU style C locale (e.g. "en_US"). If not specified or empty, disables the override and
   * restores default host system locale.
   */
  public val locale: String? = null
)

/**
 * Request object containing input parameters for the [EmulationDomain.setTimezoneOverride] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetTimezoneOverrideRequest(
  /**
   * The timezone identifier. If empty, disables the override and
   * restores default host system timezone.
   */
  public val timezoneId: String
)

/**
 * Request object containing input parameters for the [EmulationDomain.setVisibleSize] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetVisibleSizeRequest(
  /**
   * Frame width (DIP).
   */
  public val width: Int,
  /**
   * Frame height (DIP).
   */
  public val height: Int
)

/**
 * Request object containing input parameters for the [EmulationDomain.setDisabledImageTypes]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetDisabledImageTypesRequest(
  /**
   * Image types to disable.
   */
  public val imageTypes: List<DisabledImageType>
)

/**
 * Request object containing input parameters for the [EmulationDomain.setUserAgentOverride]
 * command.
 */
@Serializable
public data class SetUserAgentOverrideRequest(
  /**
   * User agent to use.
   */
  public val userAgent: String,
  /**
   * Browser langugage to emulate.
   */
  public val acceptLanguage: String? = null,
  /**
   * The platform navigator.platform should return.
   */
  public val platform: String? = null,
  /**
   * To be sent in Sec-CH-UA-* headers and returned in navigator.userAgentData
   */
  @ExperimentalChromeApi
  public val userAgentMetadata: UserAgentMetadata? = null
)

/**
 * This domain emulates different environments for the page.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation)
 */
public class EmulationDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out EmulationEvent>> =
      mapOf(
      "Emulation.virtualTimeBudgetExpired" to
          serializer<EmulationEvent.VirtualTimeBudgetExpiredEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<EmulationEvent> = session.events(deserializersByEventName)

  /**
   * Notification sent after the virtual time budget for the current VirtualTimePolicy has run out.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#event-virtualTimeBudgetExpired)
   */
  @ExperimentalChromeApi
  public fun virtualTimeBudgetExpired(): Flow<EmulationEvent.VirtualTimeBudgetExpiredEvent> =
      session.events("Emulation.virtualTimeBudgetExpired")

  /**
   * Tells whether emulation is supported.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-canEmulate)
   */
  public suspend fun canEmulate(): CanEmulateResponse = session.request("Emulation.canEmulate",
      Unit)

  /**
   * Clears the overriden device metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-clearDeviceMetricsOverride)
   */
  public suspend fun clearDeviceMetricsOverride(): Unit =
      session.request("Emulation.clearDeviceMetricsOverride", Unit)

  /**
   * Clears the overriden Geolocation Position and Error.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-clearGeolocationOverride)
   */
  public suspend fun clearGeolocationOverride(): Unit =
      session.request("Emulation.clearGeolocationOverride", Unit)

  /**
   * Requests that page scale factor is reset to initial values.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-resetPageScaleFactor)
   */
  @ExperimentalChromeApi
  public suspend fun resetPageScaleFactor(): Unit =
      session.request("Emulation.resetPageScaleFactor", Unit)

  /**
   * Enables or disables simulating a focused and active page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setFocusEmulationEnabled)
   */
  @ExperimentalChromeApi
  public suspend fun setFocusEmulationEnabled(input: SetFocusEmulationEnabledRequest): Unit =
      session.request("Emulation.setFocusEmulationEnabled", input)

  /**
   * Enables CPU throttling to emulate slow CPUs.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setCPUThrottlingRate)
   */
  @ExperimentalChromeApi
  public suspend fun setCPUThrottlingRate(input: SetCPUThrottlingRateRequest): Unit =
      session.request("Emulation.setCPUThrottlingRate", input)

  /**
   * Sets or clears an override of the default background color of the frame. This override is used
   * if the content does not specify one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDefaultBackgroundColorOverride)
   */
  public suspend
      fun setDefaultBackgroundColorOverride(input: SetDefaultBackgroundColorOverrideRequest): Unit =
      session.request("Emulation.setDefaultBackgroundColorOverride", input)

  /**
   * Overrides the values of device screen dimensions (window.screen.width, window.screen.height,
   * window.innerWidth, window.innerHeight, and "device-width"/"device-height"-related CSS media
   * query results).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDeviceMetricsOverride)
   */
  public suspend fun setDeviceMetricsOverride(input: SetDeviceMetricsOverrideRequest): Unit =
      session.request("Emulation.setDeviceMetricsOverride", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setScrollbarsHidden)
   */
  @ExperimentalChromeApi
  public suspend fun setScrollbarsHidden(input: SetScrollbarsHiddenRequest): Unit =
      session.request("Emulation.setScrollbarsHidden", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDocumentCookieDisabled)
   */
  @ExperimentalChromeApi
  public suspend fun setDocumentCookieDisabled(input: SetDocumentCookieDisabledRequest): Unit =
      session.request("Emulation.setDocumentCookieDisabled", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setEmitTouchEventsForMouse)
   */
  @ExperimentalChromeApi
  public suspend fun setEmitTouchEventsForMouse(input: SetEmitTouchEventsForMouseRequest): Unit =
      session.request("Emulation.setEmitTouchEventsForMouse", input)

  /**
   * Emulates the given media type or media feature for CSS media queries.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setEmulatedMedia)
   */
  public suspend fun setEmulatedMedia(input: SetEmulatedMediaRequest): Unit =
      session.request("Emulation.setEmulatedMedia", input)

  /**
   * Emulates the given vision deficiency.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setEmulatedVisionDeficiency)
   */
  @ExperimentalChromeApi
  public suspend fun setEmulatedVisionDeficiency(input: SetEmulatedVisionDeficiencyRequest): Unit =
      session.request("Emulation.setEmulatedVisionDeficiency", input)

  /**
   * Overrides the Geolocation Position or Error. Omitting any of the parameters emulates position
   * unavailable.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setGeolocationOverride)
   */
  public suspend fun setGeolocationOverride(input: SetGeolocationOverrideRequest): Unit =
      session.request("Emulation.setGeolocationOverride", input)

  /**
   * Overrides the Idle state.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setIdleOverride)
   */
  @ExperimentalChromeApi
  public suspend fun setIdleOverride(input: SetIdleOverrideRequest): Unit =
      session.request("Emulation.setIdleOverride", input)

  /**
   * Clears Idle state overrides.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-clearIdleOverride)
   */
  @ExperimentalChromeApi
  public suspend fun clearIdleOverride(): Unit = session.request("Emulation.clearIdleOverride",
      Unit)

  /**
   * Overrides value returned by the javascript navigator object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setNavigatorOverrides)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setNavigatorOverrides(input: SetNavigatorOverridesRequest): Unit =
      session.request("Emulation.setNavigatorOverrides", input)

  /**
   * Sets a specified page scale factor.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setPageScaleFactor)
   */
  @ExperimentalChromeApi
  public suspend fun setPageScaleFactor(input: SetPageScaleFactorRequest): Unit =
      session.request("Emulation.setPageScaleFactor", input)

  /**
   * Switches script execution in the page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setScriptExecutionDisabled)
   */
  public suspend fun setScriptExecutionDisabled(input: SetScriptExecutionDisabledRequest): Unit =
      session.request("Emulation.setScriptExecutionDisabled", input)

  /**
   * Enables touch on platforms which do not support them.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setTouchEmulationEnabled)
   */
  public suspend fun setTouchEmulationEnabled(input: SetTouchEmulationEnabledRequest): Unit =
      session.request("Emulation.setTouchEmulationEnabled", input)

  /**
   * Turns on virtual time for all frames (replacing real-time with a synthetic time source) and
   * sets
   * the current virtual time policy.  Note this supersedes any previous time budget.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setVirtualTimePolicy)
   */
  @ExperimentalChromeApi
  public suspend fun setVirtualTimePolicy(input: SetVirtualTimePolicyRequest):
      SetVirtualTimePolicyResponse = session.request("Emulation.setVirtualTimePolicy", input)

  /**
   * Overrides default host system locale with the specified one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setLocaleOverride)
   */
  @ExperimentalChromeApi
  public suspend fun setLocaleOverride(input: SetLocaleOverrideRequest): Unit =
      session.request("Emulation.setLocaleOverride", input)

  /**
   * Overrides default host system timezone with the specified one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setTimezoneOverride)
   */
  @ExperimentalChromeApi
  public suspend fun setTimezoneOverride(input: SetTimezoneOverrideRequest): Unit =
      session.request("Emulation.setTimezoneOverride", input)

  /**
   * Resizes the frame/viewport of the page. Note that this does not affect the frame's container
   * (e.g. browser window). Can be used to produce screenshots of the specified size. Not supported
   * on Android.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setVisibleSize)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setVisibleSize(input: SetVisibleSizeRequest): Unit =
      session.request("Emulation.setVisibleSize", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setDisabledImageTypes)
   */
  @ExperimentalChromeApi
  public suspend fun setDisabledImageTypes(input: SetDisabledImageTypesRequest): Unit =
      session.request("Emulation.setDisabledImageTypes", input)

  /**
   * Allows overriding user agent with the given string.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Emulation/#method-setUserAgentOverride)
   */
  public suspend fun setUserAgentOverride(input: SetUserAgentOverrideRequest): Unit =
      session.request("Emulation.setUserAgentOverride", input)
}
