@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.fetch

import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.fetch.events.FetchEvent
import org.hildan.chrome.devtools.domains.io.StreamHandle
import org.hildan.chrome.devtools.domains.network.ErrorReason
import org.hildan.chrome.devtools.protocol.ChromeDPSession

/**
 * Request object containing input parameters for the [FetchDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * If specified, only requests matching any of these patterns will produce
   * fetchRequested event and will be paused until clients response. If not set,
   * all requests will be affected.
   */
  public val patterns: List<RequestPattern>? = null,
  /**
   * If true, authRequired events will be issued and requests will be paused
   * expecting a call to continueWithAuth.
   */
  public val handleAuthRequests: Boolean? = null
)

/**
 * Request object containing input parameters for the [FetchDomain.failRequest] command.
 */
@Serializable
public data class FailRequestRequest(
  /**
   * An id the client received in requestPaused event.
   */
  public val requestId: RequestId,
  /**
   * Causes the request to fail with the given reason.
   */
  public val errorReason: ErrorReason
)

/**
 * Request object containing input parameters for the [FetchDomain.fulfillRequest] command.
 */
@Serializable
public data class FulfillRequestRequest(
  /**
   * An id the client received in requestPaused event.
   */
  public val requestId: RequestId,
  /**
   * An HTTP response code.
   */
  public val responseCode: Int,
  /**
   * Response headers.
   */
  public val responseHeaders: List<HeaderEntry>? = null,
  /**
   * Alternative way of specifying response headers as a \0-separated
   * series of name: value pairs. Prefer the above method unless you
   * need to represent some non-UTF8 values that can't be transmitted
   * over the protocol as text. (Encoded as a base64 string when passed over JSON)
   */
  public val binaryResponseHeaders: String? = null,
  /**
   * A response body. (Encoded as a base64 string when passed over JSON)
   */
  public val body: String? = null,
  /**
   * A textual representation of responseCode.
   * If absent, a standard phrase matching responseCode is used.
   */
  public val responsePhrase: String? = null
)

/**
 * Request object containing input parameters for the [FetchDomain.continueRequest] command.
 */
@Serializable
public data class ContinueRequestRequest(
  /**
   * An id the client received in requestPaused event.
   */
  public val requestId: RequestId,
  /**
   * If set, the request url will be modified in a way that's not observable by page.
   */
  public val url: String? = null,
  /**
   * If set, the request method is overridden.
   */
  public val method: String? = null,
  /**
   * If set, overrides the post data in the request. (Encoded as a base64 string when passed over
   * JSON)
   */
  public val postData: String? = null,
  /**
   * If set, overrides the request headers.
   */
  public val headers: List<HeaderEntry>? = null
)

/**
 * Request object containing input parameters for the [FetchDomain.continueWithAuth] command.
 */
@Serializable
public data class ContinueWithAuthRequest(
  /**
   * An id the client received in authRequired event.
   */
  public val requestId: RequestId,
  /**
   * Response to  with an authChallenge.
   */
  public val authChallengeResponse: AuthChallengeResponse
)

/**
 * Request object containing input parameters for the [FetchDomain.getResponseBody] command.
 */
@Serializable
public data class GetResponseBodyRequest(
  /**
   * Identifier for the intercepted request to get body for.
   */
  public val requestId: RequestId
)

/**
 * Response type for the [FetchDomain.getResponseBody] command.
 */
@Serializable
public data class GetResponseBodyResponse(
  /**
   * Response body.
   */
  public val body: String,
  /**
   * True, if content was sent as base64.
   */
  public val base64Encoded: Boolean
)

/**
 * Request object containing input parameters for the [FetchDomain.takeResponseBodyAsStream]
 * command.
 */
@Serializable
public data class TakeResponseBodyAsStreamRequest(
  public val requestId: RequestId
)

/**
 * Response type for the [FetchDomain.takeResponseBodyAsStream] command.
 */
@Serializable
public data class TakeResponseBodyAsStreamResponse(
  public val stream: StreamHandle
)

/**
 * A domain for letting clients substitute browser's network layer with client code.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch)
 */
public class FetchDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out FetchEvent>> =
      mapOf(
      "Fetch.requestPaused" to serializer<FetchEvent.RequestPausedEvent>(),
      "Fetch.authRequired" to serializer<FetchEvent.AuthRequiredEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<FetchEvent> = session.events(deserializersByEventName)

  /**
   * Issued when the domain is enabled and the request URL matches the
   * specified filter. The request is paused until the client responds
   * with one of continueRequest, failRequest or fulfillRequest.
   * The stage of the request can be determined by presence of responseErrorReason
   * and responseStatusCode -- the request is at the response stage if either
   * of these fields is present and in the request stage otherwise.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#event-requestPaused)
   */
  public fun requestPaused(): Flow<FetchEvent.RequestPausedEvent> =
      session.events("Fetch.requestPaused")

  /**
   * Issued when the domain is enabled with handleAuthRequests set to true.
   * The request is paused until client responds with continueWithAuth.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#event-authRequired)
   */
  public fun authRequired(): Flow<FetchEvent.AuthRequiredEvent> =
      session.events("Fetch.authRequired")

  /**
   * Disables the fetch domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Fetch.disable", Unit)

  /**
   * Enables issuing of requestPaused events. A request will be paused until client
   * calls one of failRequest, fulfillRequest or continueRequest/continueWithAuth.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-enable)
   */
  public suspend fun enable(input: EnableRequest): Unit = session.request("Fetch.enable", input)

  /**
   * Causes the request to fail with specified reason.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-failRequest)
   */
  public suspend fun failRequest(input: FailRequestRequest): Unit =
      session.request("Fetch.failRequest", input)

  /**
   * Provides response to the request.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-fulfillRequest)
   */
  public suspend fun fulfillRequest(input: FulfillRequestRequest): Unit =
      session.request("Fetch.fulfillRequest", input)

  /**
   * Continues the request, optionally modifying some of its parameters.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-continueRequest)
   */
  public suspend fun continueRequest(input: ContinueRequestRequest): Unit =
      session.request("Fetch.continueRequest", input)

  /**
   * Continues a request supplying authChallengeResponse following authRequired event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-continueWithAuth)
   */
  public suspend fun continueWithAuth(input: ContinueWithAuthRequest): Unit =
      session.request("Fetch.continueWithAuth", input)

  /**
   * Causes the body of the response to be received from the server and
   * returned as a single string. May only be issued for a request that
   * is paused in the Response stage and is mutually exclusive with
   * takeResponseBodyForInterceptionAsStream. Calling other methods that
   * affect the request or disabling fetch domain before body is received
   * results in an undefined behavior.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-getResponseBody)
   */
  public suspend fun getResponseBody(input: GetResponseBodyRequest): GetResponseBodyResponse =
      session.request("Fetch.getResponseBody", input)

  /**
   * Returns a handle to the stream representing the response body.
   * The request must be paused in the HeadersReceived stage.
   * Note that after this command the request can't be continued
   * as is -- client either needs to cancel it or to provide the
   * response body.
   * The stream only supports sequential read, IO.read will fail if the position
   * is specified.
   * This method is mutually exclusive with getResponseBody.
   * Calling other methods that affect the request or disabling fetch
   * domain before body is received results in an undefined behavior.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#method-takeResponseBodyAsStream)
   */
  public suspend fun takeResponseBodyAsStream(input: TakeResponseBodyAsStreamRequest):
      TakeResponseBodyAsStreamResponse = session.request("Fetch.takeResponseBodyAsStream", input)
}
