@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.network

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import org.hildan.chrome.devtools.domains.io.StreamHandle
import org.hildan.chrome.devtools.domains.runtime.StackTrace
import org.hildan.chrome.devtools.domains.security.CertificateId
import org.hildan.chrome.devtools.domains.security.MixedContentType
import org.hildan.chrome.devtools.domains.security.SecurityState
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Resource type as it was perceived by the rendering engine.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-ResourceType)
 */
@Serializable
public enum class ResourceType {
  Document,
  Stylesheet,
  Image,
  Media,
  Font,
  Script,
  TextTrack,
  XHR,
  Fetch,
  EventSource,
  WebSocket,
  Manifest,
  SignedExchange,
  Ping,
  CSPViolationReport,
  Preflight,
  Other,
}

/**
 * Unique loader identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-LoaderId)
 */
public typealias LoaderId = String

/**
 * Unique request identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-RequestId)
 */
public typealias RequestId = String

/**
 * Unique intercepted request identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-InterceptionId)
 */
public typealias InterceptionId = String

/**
 * Network level fetch failure reason.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-ErrorReason)
 */
@Serializable
public enum class ErrorReason {
  Failed,
  Aborted,
  TimedOut,
  AccessDenied,
  ConnectionClosed,
  ConnectionReset,
  ConnectionRefused,
  ConnectionAborted,
  ConnectionFailed,
  NameNotResolved,
  InternetDisconnected,
  AddressUnreachable,
  BlockedByClient,
  BlockedByResponse,
}

/**
 * UTC time in seconds, counted from January 1, 1970.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-TimeSinceEpoch)
 */
public typealias TimeSinceEpoch = Double

/**
 * Monotonically increasing time in seconds since an arbitrary point in the past.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-MonotonicTime)
 */
public typealias MonotonicTime = Double

/**
 * Request / response headers as keys / values of JSON object.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-Headers)
 */
public typealias Headers = JsonElement

/**
 * The underlying connection technology that the browser is supposedly using.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-ConnectionType)
 */
@Serializable
public enum class ConnectionType {
  none,
  cellular2g,
  cellular3g,
  cellular4g,
  bluetooth,
  ethernet,
  wifi,
  wimax,
  other,
}

/**
 * Represents the cookie's 'SameSite' status:
 * https://tools.ietf.org/html/draft-west-first-party-cookies
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CookieSameSite)
 */
@Serializable
public enum class CookieSameSite {
  Strict,
  Lax,
  None,
}

/**
 * Represents the cookie's 'Priority' status:
 * https://tools.ietf.org/html/draft-west-cookie-priority-00
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CookiePriority)
 */
@ExperimentalChromeApi
@Serializable
public enum class CookiePriority {
  Low,
  Medium,
  High,
}

/**
 * Timing information for the request.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-ResourceTiming)
 */
@Serializable
public data class ResourceTiming(
  /**
   * Timing's requestTime is a baseline in seconds, while the other numbers are ticks in
   * milliseconds relatively to this requestTime.
   */
  public val requestTime: Double,
  /**
   * Started resolving proxy.
   */
  public val proxyStart: Double,
  /**
   * Finished resolving proxy.
   */
  public val proxyEnd: Double,
  /**
   * Started DNS address resolve.
   */
  public val dnsStart: Double,
  /**
   * Finished DNS address resolve.
   */
  public val dnsEnd: Double,
  /**
   * Started connecting to the remote host.
   */
  public val connectStart: Double,
  /**
   * Connected to the remote host.
   */
  public val connectEnd: Double,
  /**
   * Started SSL handshake.
   */
  public val sslStart: Double,
  /**
   * Finished SSL handshake.
   */
  public val sslEnd: Double,
  /**
   * Started running ServiceWorker.
   */
  @ExperimentalChromeApi
  public val workerStart: Double? = null,
  /**
   * Finished Starting ServiceWorker.
   */
  @ExperimentalChromeApi
  public val workerReady: Double? = null,
  /**
   * Started fetch event.
   */
  @ExperimentalChromeApi
  public val workerFetchStart: Double? = null,
  /**
   * Settled fetch event respondWith promise.
   */
  @ExperimentalChromeApi
  public val workerRespondWithSettled: Double? = null,
  /**
   * Started sending request.
   */
  public val sendStart: Double,
  /**
   * Finished sending request.
   */
  public val sendEnd: Double,
  /**
   * Time the server started pushing request.
   */
  @ExperimentalChromeApi
  public val pushStart: Double? = null,
  /**
   * Time the server finished pushing request.
   */
  @ExperimentalChromeApi
  public val pushEnd: Double? = null,
  /**
   * Finished receiving response headers.
   */
  public val receiveHeadersEnd: Double
)

/**
 * Loading priority of a resource request.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-ResourcePriority)
 */
@Serializable
public enum class ResourcePriority {
  VeryLow,
  Low,
  Medium,
  High,
  VeryHigh,
}

/**
 * Post data entry for HTTP request
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-PostDataEntry)
 */
@Serializable
public data class PostDataEntry(
  public val bytes: String? = null
)

/**
 * HTTP request data.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-Request)
 */
@Serializable
public data class Request(
  /**
   * Request URL (without fragment).
   */
  public val url: String,
  /**
   * Fragment of the requested URL starting with hash, if present.
   */
  public val urlFragment: String? = null,
  /**
   * HTTP request method.
   */
  public val method: String,
  /**
   * HTTP request headers.
   */
  public val headers: Headers,
  /**
   * HTTP POST request data.
   */
  public val postData: String? = null,
  /**
   * True when the request has POST data. Note that postData might still be omitted when this flag
   * is true when the data is too long.
   */
  public val hasPostData: Boolean? = null,
  /**
   * Request body elements. This will be converted from base64 to binary
   */
  @ExperimentalChromeApi
  public val postDataEntries: List<PostDataEntry>? = null,
  /**
   * The mixed content type of the request.
   */
  public val mixedContentType: MixedContentType? = null,
  /**
   * Priority of the resource request at the time request is sent.
   */
  public val initialPriority: ResourcePriority,
  /**
   * The referrer policy of the request, as defined in https://www.w3.org/TR/referrer-policy/
   */
  public val referrerPolicy: String,
  /**
   * Whether is loaded via link preload.
   */
  public val isLinkPreload: Boolean? = null,
  /**
   * Set for requests when the TrustToken API is used. Contains the parameters
   * passed by the developer (e.g. via "fetch") as understood by the backend.
   */
  @ExperimentalChromeApi
  public val trustTokenParams: TrustTokenParams? = null
)

/**
 * Details of a signed certificate timestamp (SCT).
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-SignedCertificateTimestamp)
 */
@Serializable
public data class SignedCertificateTimestamp(
  /**
   * Validation status.
   */
  public val status: String,
  /**
   * Origin.
   */
  public val origin: String,
  /**
   * Log name / description.
   */
  public val logDescription: String,
  /**
   * Log ID.
   */
  public val logId: String,
  /**
   * Issuance date.
   */
  public val timestamp: TimeSinceEpoch,
  /**
   * Hash algorithm.
   */
  public val hashAlgorithm: String,
  /**
   * Signature algorithm.
   */
  public val signatureAlgorithm: String,
  /**
   * Signature data.
   */
  public val signatureData: String
)

/**
 * Security details about a request.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-SecurityDetails)
 */
@Serializable
public data class SecurityDetails(
  /**
   * Protocol name (e.g. "TLS 1.2" or "QUIC").
   */
  public val protocol: String,
  /**
   * Key Exchange used by the connection, or the empty string if not applicable.
   */
  public val keyExchange: String,
  /**
   * (EC)DH group used by the connection, if applicable.
   */
  public val keyExchangeGroup: String? = null,
  /**
   * Cipher name.
   */
  public val cipher: String,
  /**
   * TLS MAC. Note that AEAD ciphers do not have separate MACs.
   */
  public val mac: String? = null,
  /**
   * Certificate ID value.
   */
  public val certificateId: CertificateId,
  /**
   * Certificate subject name.
   */
  public val subjectName: String,
  /**
   * Subject Alternative Name (SAN) DNS names and IP addresses.
   */
  public val sanList: List<String>,
  /**
   * Name of the issuing CA.
   */
  public val issuer: String,
  /**
   * Certificate valid from date.
   */
  public val validFrom: TimeSinceEpoch,
  /**
   * Certificate valid to (expiration) date
   */
  public val validTo: TimeSinceEpoch,
  /**
   * List of signed certificate timestamps (SCTs).
   */
  public val signedCertificateTimestampList: List<SignedCertificateTimestamp>,
  /**
   * Whether the request complied with Certificate Transparency policy
   */
  public val certificateTransparencyCompliance: CertificateTransparencyCompliance
)

/**
 * Whether the request complied with Certificate Transparency policy.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CertificateTransparencyCompliance)
 */
@Serializable
public enum class CertificateTransparencyCompliance {
  unknown,
  `not-compliant`,
  compliant,
}

/**
 * The reason why request was blocked.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-BlockedReason)
 */
@Serializable
public enum class BlockedReason {
  other,
  csp,
  `mixed-content`,
  origin,
  inspector,
  `subresource-filter`,
  `content-type`,
  `collapsed-by-client`,
  `coep-frame-resource-needs-coep-header`,
  `coop-sandboxed-iframe-cannot-navigate-to-coop-page`,
  `corp-not-same-origin`,
  `corp-not-same-origin-after-defaulted-to-same-origin-by-coep`,
  `corp-not-same-site`,
}

/**
 * The reason why request was blocked.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CorsError)
 */
@Serializable
public enum class CorsError {
  DisallowedByMode,
  InvalidResponse,
  WildcardOriginNotAllowed,
  MissingAllowOriginHeader,
  MultipleAllowOriginValues,
  InvalidAllowOriginValue,
  AllowOriginMismatch,
  InvalidAllowCredentials,
  CorsDisabledScheme,
  PreflightInvalidStatus,
  PreflightDisallowedRedirect,
  PreflightWildcardOriginNotAllowed,
  PreflightMissingAllowOriginHeader,
  PreflightMultipleAllowOriginValues,
  PreflightInvalidAllowOriginValue,
  PreflightAllowOriginMismatch,
  PreflightInvalidAllowCredentials,
  PreflightMissingAllowExternal,
  PreflightInvalidAllowExternal,
  InvalidAllowMethodsPreflightResponse,
  InvalidAllowHeadersPreflightResponse,
  MethodDisallowedByPreflightResponse,
  HeaderDisallowedByPreflightResponse,
  RedirectContainsCredentials,
  InsecurePrivateNetwork,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CorsErrorStatus)
 */
@Serializable
public data class CorsErrorStatus(
  public val corsError: CorsError,
  public val failedParameter: String
)

/**
 * Source of serviceworker response.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-ServiceWorkerResponseSource)
 */
@Serializable
public enum class ServiceWorkerResponseSource {
  `cache-storage`,
  `http-cache`,
  `fallback-code`,
  network,
}

/**
 * Determines what type of Trust Token operation is executed and
 * depending on the type, some additional parameters. The values
 * are specified in third_party/blink/renderer/core/fetch/trust_token.idl.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-TrustTokenParams)
 */
@ExperimentalChromeApi
@Serializable
public data class TrustTokenParams(
  public val type: TrustTokenOperationType,
  /**
   * Only set for "token-redemption" type and determine whether
   * to request a fresh SRR or use a still valid cached SRR.
   */
  public val refreshPolicy: String,
  /**
   * Origins of issuers from whom to request tokens or redemption
   * records.
   */
  public val issuers: List<String>? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-TrustTokenOperationType)
 */
@ExperimentalChromeApi
@Serializable
public enum class TrustTokenOperationType {
  Issuance,
  Redemption,
  Signing,
}

/**
 * HTTP response data.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-Response)
 */
@Serializable
public data class Response(
  /**
   * Response URL. This URL can be different from CachedResource.url in case of redirect.
   */
  public val url: String,
  /**
   * HTTP response status code.
   */
  public val status: Int,
  /**
   * HTTP response status text.
   */
  public val statusText: String,
  /**
   * HTTP response headers.
   */
  public val headers: Headers,
  /**
   * HTTP response headers text.
   */
  public val headersText: String? = null,
  /**
   * Resource mimeType as determined by the browser.
   */
  public val mimeType: String,
  /**
   * Refined HTTP request headers that were actually transmitted over the network.
   */
  public val requestHeaders: Headers? = null,
  /**
   * HTTP request headers text.
   */
  public val requestHeadersText: String? = null,
  /**
   * Specifies whether physical connection was actually reused for this request.
   */
  public val connectionReused: Boolean,
  /**
   * Physical connection id that was actually used for this request.
   */
  public val connectionId: Double,
  /**
   * Remote IP address.
   */
  public val remoteIPAddress: String? = null,
  /**
   * Remote port.
   */
  public val remotePort: Int? = null,
  /**
   * Specifies that the request was served from the disk cache.
   */
  public val fromDiskCache: Boolean? = null,
  /**
   * Specifies that the request was served from the ServiceWorker.
   */
  public val fromServiceWorker: Boolean? = null,
  /**
   * Specifies that the request was served from the prefetch cache.
   */
  public val fromPrefetchCache: Boolean? = null,
  /**
   * Total number of bytes received for this request so far.
   */
  public val encodedDataLength: Double,
  /**
   * Timing information for the given request.
   */
  public val timing: ResourceTiming? = null,
  /**
   * Response source of response from ServiceWorker.
   */
  public val serviceWorkerResponseSource: ServiceWorkerResponseSource? = null,
  /**
   * The time at which the returned response was generated.
   */
  public val responseTime: TimeSinceEpoch? = null,
  /**
   * Cache Storage Cache Name.
   */
  public val cacheStorageCacheName: String? = null,
  /**
   * Protocol used to fetch this request.
   */
  public val protocol: String? = null,
  /**
   * Security state of the request resource.
   */
  public val securityState: SecurityState,
  /**
   * Security details for the request.
   */
  public val securityDetails: SecurityDetails? = null
)

/**
 * WebSocket request data.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-WebSocketRequest)
 */
@Serializable
public data class WebSocketRequest(
  /**
   * HTTP request headers.
   */
  public val headers: Headers
)

/**
 * WebSocket response data.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-WebSocketResponse)
 */
@Serializable
public data class WebSocketResponse(
  /**
   * HTTP response status code.
   */
  public val status: Int,
  /**
   * HTTP response status text.
   */
  public val statusText: String,
  /**
   * HTTP response headers.
   */
  public val headers: Headers,
  /**
   * HTTP response headers text.
   */
  public val headersText: String? = null,
  /**
   * HTTP request headers.
   */
  public val requestHeaders: Headers? = null,
  /**
   * HTTP request headers text.
   */
  public val requestHeadersText: String? = null
)

/**
 * WebSocket message data. This represents an entire WebSocket message, not just a fragmented frame
 * as the name suggests.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-WebSocketFrame)
 */
@Serializable
public data class WebSocketFrame(
  /**
   * WebSocket message opcode.
   */
  public val opcode: Double,
  /**
   * WebSocket message mask.
   */
  public val mask: Boolean,
  /**
   * WebSocket message payload data.
   * If the opcode is 1, this is a text message and payloadData is a UTF-8 string.
   * If the opcode isn't 1, then payloadData is a base64 encoded string representing binary data.
   */
  public val payloadData: String
)

/**
 * Information about the cached resource.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CachedResource)
 */
@Serializable
public data class CachedResource(
  /**
   * Resource URL. This is the url of the original network request.
   */
  public val url: String,
  /**
   * Type of this resource.
   */
  public val type: ResourceType,
  /**
   * Cached response data.
   */
  public val response: Response? = null,
  /**
   * Cached response body size.
   */
  public val bodySize: Double
)

/**
 * Information about the request initiator.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-Initiator)
 */
@Serializable
public data class Initiator(
  /**
   * Type of this initiator.
   */
  public val type: String,
  /**
   * Initiator JavaScript stack trace, set for Script only.
   */
  public val stack: StackTrace? = null,
  /**
   * Initiator URL, set for Parser type or for Script type (when script is importing module) or for
   * SignedExchange type.
   */
  public val url: String? = null,
  /**
   * Initiator line number, set for Parser type or for Script type (when script is importing
   * module) (0-based).
   */
  public val lineNumber: Double? = null,
  /**
   * Initiator column number, set for Parser type or for Script type (when script is importing
   * module) (0-based).
   */
  public val columnNumber: Double? = null,
  /**
   * Set if another request triggered this request (e.g. preflight).
   */
  public val requestId: RequestId? = null
)

/**
 * Cookie object
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-Cookie)
 */
@Serializable
public data class Cookie(
  /**
   * Cookie name.
   */
  public val name: String,
  /**
   * Cookie value.
   */
  public val value: String,
  /**
   * Cookie domain.
   */
  public val domain: String,
  /**
   * Cookie path.
   */
  public val path: String,
  /**
   * Cookie expiration date as the number of seconds since the UNIX epoch.
   */
  public val expires: Double,
  /**
   * Cookie size.
   */
  public val size: Int,
  /**
   * True if cookie is http-only.
   */
  public val httpOnly: Boolean,
  /**
   * True if cookie is secure.
   */
  public val secure: Boolean,
  /**
   * True in case of session cookie.
   */
  public val session: Boolean,
  /**
   * Cookie SameSite type.
   */
  public val sameSite: CookieSameSite? = null,
  /**
   * Cookie Priority
   */
  @ExperimentalChromeApi
  public val priority: CookiePriority? = null,
  /**
   * True if cookie is SameParty.
   */
  @ExperimentalChromeApi
  public val sameParty: Boolean? = null
)

/**
 * Types of reasons why a cookie may not be stored from a response.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-SetCookieBlockedReason)
 */
@ExperimentalChromeApi
@Serializable
public enum class SetCookieBlockedReason {
  SecureOnly,
  SameSiteStrict,
  SameSiteLax,
  SameSiteUnspecifiedTreatedAsLax,
  SameSiteNoneInsecure,
  UserPreferences,
  SyntaxError,
  SchemeNotSupported,
  OverwriteSecure,
  InvalidDomain,
  InvalidPrefix,
  UnknownError,
  SchemefulSameSiteStrict,
  SchemefulSameSiteLax,
  SchemefulSameSiteUnspecifiedTreatedAsLax,
}

/**
 * Types of reasons why a cookie may not be sent with a request.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CookieBlockedReason)
 */
@ExperimentalChromeApi
@Serializable
public enum class CookieBlockedReason {
  SecureOnly,
  NotOnPath,
  DomainMismatch,
  SameSiteStrict,
  SameSiteLax,
  SameSiteUnspecifiedTreatedAsLax,
  SameSiteNoneInsecure,
  UserPreferences,
  UnknownError,
  SchemefulSameSiteStrict,
  SchemefulSameSiteLax,
  SchemefulSameSiteUnspecifiedTreatedAsLax,
}

/**
 * A cookie which was not stored from a response with the corresponding reason.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-BlockedSetCookieWithReason)
 */
@ExperimentalChromeApi
@Serializable
public data class BlockedSetCookieWithReason(
  /**
   * The reason(s) this cookie was blocked.
   */
  public val blockedReasons: List<SetCookieBlockedReason>,
  /**
   * The string representing this individual cookie as it would appear in the header.
   * This is not the entire "cookie" or "set-cookie" header which could have multiple cookies.
   */
  public val cookieLine: String,
  /**
   * The cookie object which represents the cookie which was not stored. It is optional because
   * sometimes complete cookie information is not available, such as in the case of parsing
   * errors.
   */
  public val cookie: Cookie? = null
)

/**
 * A cookie with was not sent with a request with the corresponding reason.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-BlockedCookieWithReason)
 */
@ExperimentalChromeApi
@Serializable
public data class BlockedCookieWithReason(
  /**
   * The reason(s) the cookie was blocked.
   */
  public val blockedReasons: List<CookieBlockedReason>,
  /**
   * The cookie object representing the cookie which was not sent.
   */
  public val cookie: Cookie
)

/**
 * Cookie parameter object
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CookieParam)
 */
@Serializable
public data class CookieParam(
  /**
   * Cookie name.
   */
  public val name: String,
  /**
   * Cookie value.
   */
  public val value: String,
  /**
   * The request-URI to associate with the setting of the cookie. This value can affect the
   * default domain and path values of the created cookie.
   */
  public val url: String? = null,
  /**
   * Cookie domain.
   */
  public val domain: String? = null,
  /**
   * Cookie path.
   */
  public val path: String? = null,
  /**
   * True if cookie is secure.
   */
  public val secure: Boolean? = null,
  /**
   * True if cookie is http-only.
   */
  public val httpOnly: Boolean? = null,
  /**
   * Cookie SameSite type.
   */
  public val sameSite: CookieSameSite? = null,
  /**
   * Cookie expiration date, session cookie if not set
   */
  public val expires: TimeSinceEpoch? = null,
  /**
   * Cookie Priority.
   */
  @ExperimentalChromeApi
  public val priority: CookiePriority? = null
)

/**
 * Authorization challenge for HTTP status code 401 or 407.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-AuthChallenge)
 */
@ExperimentalChromeApi
@Serializable
public data class AuthChallenge(
  /**
   * Source of the authentication challenge.
   */
  public val source: String? = null,
  /**
   * Origin of the challenger.
   */
  public val origin: String,
  /**
   * The authentication scheme used, such as basic or digest
   */
  public val scheme: String,
  /**
   * The realm of the challenge. May be empty.
   */
  public val realm: String
)

/**
 * Response to an AuthChallenge.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-AuthChallengeResponse)
 */
@ExperimentalChromeApi
@Serializable
public data class AuthChallengeResponse(
  /**
   * The decision on what to do in response to the authorization challenge.  Default means
   * deferring to the default behavior of the net stack, which will likely either the Cancel
   * authentication or display a popup dialog box.
   */
  public val response: String,
  /**
   * The username to provide, possibly empty. Should only be set if response is
   * ProvideCredentials.
   */
  public val username: String? = null,
  /**
   * The password to provide, possibly empty. Should only be set if response is
   * ProvideCredentials.
   */
  public val password: String? = null
)

/**
 * Stages of the interception to begin intercepting. Request will intercept before the request is
 * sent. Response will intercept after the response is received.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-InterceptionStage)
 */
@ExperimentalChromeApi
@Serializable
public enum class InterceptionStage {
  Request,
  HeadersReceived,
}

/**
 * Request pattern for interception.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-RequestPattern)
 */
@ExperimentalChromeApi
@Serializable
public data class RequestPattern(
  /**
   * Wildcards ('*' -> zero or more, '?' -> exactly one) are allowed. Escape character is
   * backslash. Omitting is equivalent to "*".
   */
  public val urlPattern: String? = null,
  /**
   * If set, only requests for matching resource types will be intercepted.
   */
  public val resourceType: ResourceType? = null,
  /**
   * Stage at wich to begin intercepting requests. Default is Request.
   */
  public val interceptionStage: InterceptionStage? = null
)

/**
 * Information about a signed exchange signature.
 * https://wicg.github.io/webpackage/draft-yasskin-httpbis-origin-signed-exchanges-impl.html#rfc.section.3.1
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-SignedExchangeSignature)
 */
@ExperimentalChromeApi
@Serializable
public data class SignedExchangeSignature(
  /**
   * Signed exchange signature label.
   */
  public val label: String,
  /**
   * The hex string of signed exchange signature.
   */
  public val signature: String,
  /**
   * Signed exchange signature integrity.
   */
  public val integrity: String,
  /**
   * Signed exchange signature cert Url.
   */
  public val certUrl: String? = null,
  /**
   * The hex string of signed exchange signature cert sha256.
   */
  public val certSha256: String? = null,
  /**
   * Signed exchange signature validity Url.
   */
  public val validityUrl: String,
  /**
   * Signed exchange signature date.
   */
  public val date: Int,
  /**
   * Signed exchange signature expires.
   */
  public val expires: Int,
  /**
   * The encoded certificates.
   */
  public val certificates: List<String>? = null
)

/**
 * Information about a signed exchange header.
 * https://wicg.github.io/webpackage/draft-yasskin-httpbis-origin-signed-exchanges-impl.html#cbor-representation
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-SignedExchangeHeader)
 */
@ExperimentalChromeApi
@Serializable
public data class SignedExchangeHeader(
  /**
   * Signed exchange request URL.
   */
  public val requestUrl: String,
  /**
   * Signed exchange response code.
   */
  public val responseCode: Int,
  /**
   * Signed exchange response headers.
   */
  public val responseHeaders: Headers,
  /**
   * Signed exchange response signature.
   */
  public val signatures: List<SignedExchangeSignature>,
  /**
   * Signed exchange header integrity hash in the form of "sha256-<base64-hash-value>".
   */
  public val headerIntegrity: String
)

/**
 * Field type for a signed exchange related error.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-SignedExchangeErrorField)
 */
@ExperimentalChromeApi
@Serializable
public enum class SignedExchangeErrorField {
  signatureSig,
  signatureIntegrity,
  signatureCertUrl,
  signatureCertSha256,
  signatureValidityUrl,
  signatureTimestamps,
}

/**
 * Information about a signed exchange response.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-SignedExchangeError)
 */
@ExperimentalChromeApi
@Serializable
public data class SignedExchangeError(
  /**
   * Error message.
   */
  public val message: String,
  /**
   * The index of the signature which caused the error.
   */
  public val signatureIndex: Int? = null,
  /**
   * The field which caused the error.
   */
  public val errorField: SignedExchangeErrorField? = null
)

/**
 * Information about a signed exchange response.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-SignedExchangeInfo)
 */
@ExperimentalChromeApi
@Serializable
public data class SignedExchangeInfo(
  /**
   * The outer response of signed HTTP exchange which was received from network.
   */
  public val outerResponse: Response,
  /**
   * Information about the signed exchange header.
   */
  public val header: SignedExchangeHeader? = null,
  /**
   * Security details for the signed exchange header.
   */
  public val securityDetails: SecurityDetails? = null,
  /**
   * Errors occurred while handling the signed exchagne.
   */
  public val errors: List<SignedExchangeError>? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-PrivateNetworkRequestPolicy)
 */
@ExperimentalChromeApi
@Serializable
public enum class PrivateNetworkRequestPolicy {
  Allow,
  BlockFromInsecureToMorePrivate,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-IPAddressSpace)
 */
@ExperimentalChromeApi
@Serializable
public enum class IPAddressSpace {
  Local,
  Private,
  Public,
  Unknown,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-ClientSecurityState)
 */
@ExperimentalChromeApi
@Serializable
public data class ClientSecurityState(
  public val initiatorIsSecureContext: Boolean,
  public val initiatorIPAddressSpace: IPAddressSpace,
  public val privateNetworkRequestPolicy: PrivateNetworkRequestPolicy
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CrossOriginOpenerPolicyValue)
 */
@ExperimentalChromeApi
@Serializable
public enum class CrossOriginOpenerPolicyValue {
  SameOrigin,
  SameOriginAllowPopups,
  UnsafeNone,
  SameOriginPlusCoep,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CrossOriginOpenerPolicyStatus)
 */
@ExperimentalChromeApi
@Serializable
public data class CrossOriginOpenerPolicyStatus(
  public val value: CrossOriginOpenerPolicyValue,
  public val reportOnlyValue: CrossOriginOpenerPolicyValue,
  public val reportingEndpoint: String? = null,
  public val reportOnlyReportingEndpoint: String? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CrossOriginEmbedderPolicyValue)
 */
@ExperimentalChromeApi
@Serializable
public enum class CrossOriginEmbedderPolicyValue {
  None,
  RequireCorp,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-CrossOriginEmbedderPolicyStatus)
 */
@ExperimentalChromeApi
@Serializable
public data class CrossOriginEmbedderPolicyStatus(
  public val value: CrossOriginEmbedderPolicyValue,
  public val reportOnlyValue: CrossOriginEmbedderPolicyValue,
  public val reportingEndpoint: String? = null,
  public val reportOnlyReportingEndpoint: String? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-SecurityIsolationStatus)
 */
@ExperimentalChromeApi
@Serializable
public data class SecurityIsolationStatus(
  public val coop: CrossOriginOpenerPolicyStatus? = null,
  public val coep: CrossOriginEmbedderPolicyStatus? = null
)

/**
 * An object providing the result of a network resource load.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-LoadNetworkResourcePageResult)
 */
@ExperimentalChromeApi
@Serializable
public data class LoadNetworkResourcePageResult(
  public val success: Boolean,
  /**
   * Optional values used for error reporting.
   */
  public val netError: Double? = null,
  public val netErrorName: String? = null,
  public val httpStatusCode: Double? = null,
  /**
   * If successful, one of the following two fields holds the result.
   */
  public val stream: StreamHandle? = null,
  /**
   * Response headers.
   */
  public val headers: Headers? = null
)

/**
 * An options object that may be extended later to better support CORS,
 * CORB and streaming.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Network/#type-LoadNetworkResourceOptions)
 */
@ExperimentalChromeApi
@Serializable
public data class LoadNetworkResourceOptions(
  public val disableCache: Boolean,
  public val includeCredentials: Boolean
)
