@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.tracing

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.tracing.events.TracingEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Response type for the [TracingDomain.getCategories] command.
 */
@Serializable
public data class GetCategoriesResponse(
  /**
   * A list of supported tracing categories.
   */
  public val categories: List<String>
)

/**
 * Request object containing input parameters for the [TracingDomain.recordClockSyncMarker] command.
 */
@Serializable
public data class RecordClockSyncMarkerRequest(
  /**
   * The ID of this clock sync marker
   */
  public val syncId: String
)

/**
 * Request object containing input parameters for the [TracingDomain.requestMemoryDump] command.
 */
@Serializable
public data class RequestMemoryDumpRequest(
  /**
   * Enables more deterministic results by forcing garbage collection
   */
  public val deterministic: Boolean? = null,
  /**
   * Specifies level of details in memory dump. Defaults to "detailed".
   */
  public val levelOfDetail: MemoryDumpLevelOfDetail? = null
)

/**
 * Response type for the [TracingDomain.requestMemoryDump] command.
 */
@Serializable
public data class RequestMemoryDumpResponse(
  /**
   * GUID of the resulting global memory dump.
   */
  public val dumpGuid: String,
  /**
   * True iff the global memory dump succeeded.
   */
  public val success: Boolean
)

/**
 * Request object containing input parameters for the [TracingDomain.start] command.
 */
@Serializable
public data class StartRequest(
  /**
   * Category/tag filter
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val categories: String? = null,
  /**
   * Tracing options
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val options: String? = null,
  /**
   * If set, the agent will issue bufferUsage events at this interval, specified in milliseconds
   */
  public val bufferUsageReportingInterval: Double? = null,
  /**
   * Whether to report trace events as series of dataCollected events or to save trace to a
   * stream (defaults to `ReportEvents`).
   */
  public val transferMode: String? = null,
  /**
   * Trace data format to use. This only applies when using `ReturnAsStream`
   * transfer mode (defaults to `json`).
   */
  public val streamFormat: StreamFormat? = null,
  /**
   * Compression format to use. This only applies when using `ReturnAsStream`
   * transfer mode (defaults to `none`)
   */
  public val streamCompression: StreamCompression? = null,
  public val traceConfig: TraceConfig? = null,
  /**
   * Base64-encoded serialized perfetto.protos.TraceConfig protobuf message
   * When specified, the parameters `categories`, `options`, `traceConfig`
   * are ignored. (Encoded as a base64 string when passed over JSON)
   */
  public val perfettoConfig: String? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing)
 */
@ExperimentalChromeApi
public class TracingDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out TracingEvent>> =
      mapOf(
      "Tracing.bufferUsage" to serializer<TracingEvent.BufferUsageEvent>(),
      "Tracing.dataCollected" to serializer<TracingEvent.DataCollectedEvent>(),
      "Tracing.tracingComplete" to serializer<TracingEvent.TracingCompleteEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<TracingEvent> = session.events(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#event-bufferUsage)
   */
  public fun bufferUsage(): Flow<TracingEvent.BufferUsageEvent> =
      session.events("Tracing.bufferUsage")

  /**
   * Contains an bucket of collected trace events. When tracing is stopped collected events will be
   * send as a sequence of dataCollected events followed by tracingComplete event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#event-dataCollected)
   */
  public fun dataCollected(): Flow<TracingEvent.DataCollectedEvent> =
      session.events("Tracing.dataCollected")

  /**
   * Signals that tracing is stopped and there is no trace buffers pending flush, all data were
   * delivered via dataCollected events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#event-tracingComplete)
   */
  public fun tracingComplete(): Flow<TracingEvent.TracingCompleteEvent> =
      session.events("Tracing.tracingComplete")

  /**
   * Stop trace events collection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-end)
   */
  public suspend fun end(): Unit = session.request("Tracing.end", Unit)

  /**
   * Gets supported tracing categories.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-getCategories)
   */
  public suspend fun getCategories(): GetCategoriesResponse =
      session.request("Tracing.getCategories", Unit)

  /**
   * Record a clock sync marker in the trace.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-recordClockSyncMarker)
   */
  public suspend fun recordClockSyncMarker(input: RecordClockSyncMarkerRequest): Unit =
      session.request("Tracing.recordClockSyncMarker", input)

  /**
   * Request a global memory dump.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-requestMemoryDump)
   */
  public suspend fun requestMemoryDump(input: RequestMemoryDumpRequest): RequestMemoryDumpResponse =
      session.request("Tracing.requestMemoryDump", input)

  /**
   * Start trace events collection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-start)
   */
  public suspend fun start(input: StartRequest): Unit = session.request("Tracing.start", input)
}
