@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.webauthn

import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [WebAuthnDomain.addVirtualAuthenticator]
 * command.
 */
@Serializable
public data class AddVirtualAuthenticatorRequest(
  public val options: VirtualAuthenticatorOptions
)

/**
 * Response type for the [WebAuthnDomain.addVirtualAuthenticator] command.
 */
@Serializable
public data class AddVirtualAuthenticatorResponse(
  public val authenticatorId: AuthenticatorId
)

/**
 * Request object containing input parameters for the [WebAuthnDomain.removeVirtualAuthenticator]
 * command.
 */
@Serializable
public data class RemoveVirtualAuthenticatorRequest(
  public val authenticatorId: AuthenticatorId
)

/**
 * Request object containing input parameters for the [WebAuthnDomain.addCredential] command.
 */
@Serializable
public data class AddCredentialRequest(
  public val authenticatorId: AuthenticatorId,
  public val credential: Credential
)

/**
 * Request object containing input parameters for the [WebAuthnDomain.getCredential] command.
 */
@Serializable
public data class GetCredentialRequest(
  public val authenticatorId: AuthenticatorId,
  public val credentialId: String
)

/**
 * Response type for the [WebAuthnDomain.getCredential] command.
 */
@Serializable
public data class GetCredentialResponse(
  public val credential: Credential
)

/**
 * Request object containing input parameters for the [WebAuthnDomain.getCredentials] command.
 */
@Serializable
public data class GetCredentialsRequest(
  public val authenticatorId: AuthenticatorId
)

/**
 * Response type for the [WebAuthnDomain.getCredentials] command.
 */
@Serializable
public data class GetCredentialsResponse(
  public val credentials: List<Credential>
)

/**
 * Request object containing input parameters for the [WebAuthnDomain.removeCredential] command.
 */
@Serializable
public data class RemoveCredentialRequest(
  public val authenticatorId: AuthenticatorId,
  public val credentialId: String
)

/**
 * Request object containing input parameters for the [WebAuthnDomain.clearCredentials] command.
 */
@Serializable
public data class ClearCredentialsRequest(
  public val authenticatorId: AuthenticatorId
)

/**
 * Request object containing input parameters for the [WebAuthnDomain.setUserVerified] command.
 */
@Serializable
public data class SetUserVerifiedRequest(
  public val authenticatorId: AuthenticatorId,
  public val isUserVerified: Boolean
)

/**
 * Request object containing input parameters for the
 * [WebAuthnDomain.setAutomaticPresenceSimulation] command.
 */
@Serializable
public data class SetAutomaticPresenceSimulationRequest(
  public val authenticatorId: AuthenticatorId,
  public val enabled: Boolean
)

/**
 * This domain allows configuring virtual authenticators to test the WebAuthn
 * API.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn)
 */
@ExperimentalChromeApi
public class WebAuthnDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Enable the WebAuthn domain and start intercepting credential storage and
   * retrieval with a virtual authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("WebAuthn.enable", Unit)

  /**
   * Disable the WebAuthn domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("WebAuthn.disable", Unit)

  /**
   * Creates and adds a virtual authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-addVirtualAuthenticator)
   */
  public suspend fun addVirtualAuthenticator(input: AddVirtualAuthenticatorRequest):
      AddVirtualAuthenticatorResponse = session.request("WebAuthn.addVirtualAuthenticator", input)

  /**
   * Removes the given authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-removeVirtualAuthenticator)
   */
  public suspend fun removeVirtualAuthenticator(input: RemoveVirtualAuthenticatorRequest): Unit =
      session.request("WebAuthn.removeVirtualAuthenticator", input)

  /**
   * Adds the credential to the specified authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-addCredential)
   */
  public suspend fun addCredential(input: AddCredentialRequest): Unit =
      session.request("WebAuthn.addCredential", input)

  /**
   * Returns a single credential stored in the given virtual authenticator that
   * matches the credential ID.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-getCredential)
   */
  public suspend fun getCredential(input: GetCredentialRequest): GetCredentialResponse =
      session.request("WebAuthn.getCredential", input)

  /**
   * Returns all the credentials stored in the given virtual authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-getCredentials)
   */
  public suspend fun getCredentials(input: GetCredentialsRequest): GetCredentialsResponse =
      session.request("WebAuthn.getCredentials", input)

  /**
   * Removes a credential from the authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-removeCredential)
   */
  public suspend fun removeCredential(input: RemoveCredentialRequest): Unit =
      session.request("WebAuthn.removeCredential", input)

  /**
   * Clears all the credentials from the specified device.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-clearCredentials)
   */
  public suspend fun clearCredentials(input: ClearCredentialsRequest): Unit =
      session.request("WebAuthn.clearCredentials", input)

  /**
   * Sets whether User Verification succeeds or fails for an authenticator.
   * The default is true.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-setUserVerified)
   */
  public suspend fun setUserVerified(input: SetUserVerifiedRequest): Unit =
      session.request("WebAuthn.setUserVerified", input)

  /**
   * Sets whether tests of user presence will succeed immediately (if true) or fail to resolve (if
   * false) for an authenticator.
   * The default is true.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-setAutomaticPresenceSimulation)
   */
  public suspend fun setAutomaticPresenceSimulation(input: SetAutomaticPresenceSimulationRequest):
      Unit = session.request("WebAuthn.setAutomaticPresenceSimulation", input)
}
