@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.browser

import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.target.TargetID
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [BrowserDomain.setPermission] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetPermissionRequest(
  /**
   * Descriptor of permission to override.
   */
  public val permission: PermissionDescriptor,
  /**
   * Setting of the permission.
   */
  public val setting: PermissionSetting,
  /**
   * Origin the permission applies to, all origins if not specified.
   */
  public val origin: String? = null,
  /**
   * Context to override. When omitted, default browser context is used.
   */
  public val browserContextId: BrowserContextID? = null
)

/**
 * Request object containing input parameters for the [BrowserDomain.grantPermissions] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GrantPermissionsRequest(
  public val permissions: List<PermissionType>,
  /**
   * Origin the permission applies to, all origins if not specified.
   */
  public val origin: String? = null,
  /**
   * BrowserContext to override permissions. When omitted, default browser context is used.
   */
  public val browserContextId: BrowserContextID? = null
)

/**
 * Request object containing input parameters for the [BrowserDomain.resetPermissions] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ResetPermissionsRequest(
  /**
   * BrowserContext to reset permissions. When omitted, default browser context is used.
   */
  public val browserContextId: BrowserContextID? = null
)

/**
 * Request object containing input parameters for the [BrowserDomain.setDownloadBehavior] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetDownloadBehaviorRequest(
  /**
   * Whether to allow all or deny all download requests, or use default Chrome behavior if
   * available (otherwise deny). |allowAndName| allows download and names files according to
   * their dowmload guids.
   */
  public val behavior: String,
  /**
   * BrowserContext to set download behavior. When omitted, default browser context is used.
   */
  public val browserContextId: BrowserContextID? = null,
  /**
   * The default path to save downloaded files to. This is requred if behavior is set to 'allow'
   * or 'allowAndName'.
   */
  public val downloadPath: String? = null
)

/**
 * Response type for the [BrowserDomain.getVersion] command.
 */
@Serializable
public data class GetVersionResponse(
  /**
   * Protocol version.
   */
  public val protocolVersion: String,
  /**
   * Product name.
   */
  public val product: String,
  /**
   * Product revision.
   */
  public val revision: String,
  /**
   * User-Agent.
   */
  public val userAgent: String,
  /**
   * V8 version.
   */
  public val jsVersion: String
)

/**
 * Response type for the [BrowserDomain.getBrowserCommandLine] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetBrowserCommandLineResponse(
  /**
   * Commandline parameters
   */
  public val arguments: List<String>
)

/**
 * Request object containing input parameters for the [BrowserDomain.getHistograms] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetHistogramsRequest(
  /**
   * Requested substring in name. Only histograms which have query as a
   * substring in their name are extracted. An empty or absent query returns
   * all histograms.
   */
  public val query: String? = null,
  /**
   * If true, retrieve delta since last call.
   */
  public val delta: Boolean? = null
)

/**
 * Response type for the [BrowserDomain.getHistograms] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetHistogramsResponse(
  /**
   * Histograms.
   */
  public val histograms: List<Histogram>
)

/**
 * Request object containing input parameters for the [BrowserDomain.getHistogram] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetHistogramRequest(
  /**
   * Requested histogram name.
   */
  public val name: String,
  /**
   * If true, retrieve delta since last call.
   */
  public val delta: Boolean? = null
)

/**
 * Response type for the [BrowserDomain.getHistogram] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetHistogramResponse(
  /**
   * Histogram.
   */
  public val histogram: Histogram
)

/**
 * Request object containing input parameters for the [BrowserDomain.getWindowBounds] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetWindowBoundsRequest(
  /**
   * Browser window id.
   */
  public val windowId: WindowID
)

/**
 * Response type for the [BrowserDomain.getWindowBounds] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetWindowBoundsResponse(
  /**
   * Bounds information of the window. When window state is 'minimized', the restored window
   * position and size are returned.
   */
  public val bounds: Bounds
)

/**
 * Request object containing input parameters for the [BrowserDomain.getWindowForTarget] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetWindowForTargetRequest(
  /**
   * Devtools agent host id. If called as a part of the session, associated targetId is used.
   */
  public val targetId: TargetID? = null
)

/**
 * Response type for the [BrowserDomain.getWindowForTarget] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetWindowForTargetResponse(
  /**
   * Browser window id.
   */
  public val windowId: WindowID,
  /**
   * Bounds information of the window. When window state is 'minimized', the restored window
   * position and size are returned.
   */
  public val bounds: Bounds
)

/**
 * Request object containing input parameters for the [BrowserDomain.setWindowBounds] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetWindowBoundsRequest(
  /**
   * Browser window id.
   */
  public val windowId: WindowID,
  /**
   * New window bounds. The 'minimized', 'maximized' and 'fullscreen' states cannot be combined
   * with 'left', 'top', 'width' or 'height'. Leaves unspecified fields unchanged.
   */
  public val bounds: Bounds
)

/**
 * Request object containing input parameters for the [BrowserDomain.setDockTile] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetDockTileRequest(
  public val badgeLabel: String? = null,
  /**
   * Png encoded image. (Encoded as a base64 string when passed over JSON)
   */
  public val image: String? = null
)

/**
 * Request object containing input parameters for the [BrowserDomain.executeBrowserCommand] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ExecuteBrowserCommandRequest(
  public val commandId: BrowserCommandId
)

/**
 * The Browser domain defines methods and events for browser managing.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser)
 */
public class BrowserDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Set permission settings for given origin.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setPermission)
   */
  @ExperimentalChromeApi
  public suspend fun setPermission(input: SetPermissionRequest): Unit =
      session.request("Browser.setPermission", input)

  /**
   * Grant specific permissions to the given origin and reject all others.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-grantPermissions)
   */
  @ExperimentalChromeApi
  public suspend fun grantPermissions(input: GrantPermissionsRequest): Unit =
      session.request("Browser.grantPermissions", input)

  /**
   * Reset all permission management for all origins.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-resetPermissions)
   */
  @ExperimentalChromeApi
  public suspend fun resetPermissions(input: ResetPermissionsRequest): Unit =
      session.request("Browser.resetPermissions", input)

  /**
   * Set the behavior when downloading a file.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setDownloadBehavior)
   */
  @ExperimentalChromeApi
  public suspend fun setDownloadBehavior(input: SetDownloadBehaviorRequest): Unit =
      session.request("Browser.setDownloadBehavior", input)

  /**
   * Close browser gracefully.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-close)
   */
  public suspend fun close(): Unit = session.request("Browser.close", Unit)

  /**
   * Crashes browser on the main thread.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-crash)
   */
  @ExperimentalChromeApi
  public suspend fun crash(): Unit = session.request("Browser.crash", Unit)

  /**
   * Crashes GPU process.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-crashGpuProcess)
   */
  @ExperimentalChromeApi
  public suspend fun crashGpuProcess(): Unit = session.request("Browser.crashGpuProcess", Unit)

  /**
   * Returns version information.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getVersion)
   */
  public suspend fun getVersion(): GetVersionResponse = session.request("Browser.getVersion", Unit)

  /**
   * Returns the command line switches for the browser process if, and only if
   * --enable-automation is on the commandline.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getBrowserCommandLine)
   */
  @ExperimentalChromeApi
  public suspend fun getBrowserCommandLine(): GetBrowserCommandLineResponse =
      session.request("Browser.getBrowserCommandLine", Unit)

  /**
   * Get Chrome histograms.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getHistograms)
   */
  @ExperimentalChromeApi
  public suspend fun getHistograms(input: GetHistogramsRequest): GetHistogramsResponse =
      session.request("Browser.getHistograms", input)

  /**
   * Get a Chrome histogram by name.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getHistogram)
   */
  @ExperimentalChromeApi
  public suspend fun getHistogram(input: GetHistogramRequest): GetHistogramResponse =
      session.request("Browser.getHistogram", input)

  /**
   * Get position and size of the browser window.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getWindowBounds)
   */
  @ExperimentalChromeApi
  public suspend fun getWindowBounds(input: GetWindowBoundsRequest): GetWindowBoundsResponse =
      session.request("Browser.getWindowBounds", input)

  /**
   * Get the browser window that contains the devtools target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-getWindowForTarget)
   */
  @ExperimentalChromeApi
  public suspend fun getWindowForTarget(input: GetWindowForTargetRequest):
      GetWindowForTargetResponse = session.request("Browser.getWindowForTarget", input)

  /**
   * Set position and/or size of the browser window.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setWindowBounds)
   */
  @ExperimentalChromeApi
  public suspend fun setWindowBounds(input: SetWindowBoundsRequest): Unit =
      session.request("Browser.setWindowBounds", input)

  /**
   * Set dock tile details, platform-specific.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-setDockTile)
   */
  @ExperimentalChromeApi
  public suspend fun setDockTile(input: SetDockTileRequest): Unit =
      session.request("Browser.setDockTile", input)

  /**
   * Invoke custom browser commands used by telemetry.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#method-executeBrowserCommand)
   */
  @ExperimentalChromeApi
  public suspend fun executeBrowserCommand(input: ExecuteBrowserCommandRequest): Unit =
      session.request("Browser.executeBrowserCommand", input)
}
