@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.performance

import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.performance.events.PerformanceEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [PerformanceDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * Time domain to use for collecting and reporting duration metrics.
   */
  public val timeDomain: String? = null
)

/**
 * Request object containing input parameters for the [PerformanceDomain.setTimeDomain] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetTimeDomainRequest(
  /**
   * Time domain
   */
  public val timeDomain: String
)

/**
 * Response type for the [PerformanceDomain.getMetrics] command.
 */
@Serializable
public data class GetMetricsResponse(
  /**
   * Current values for run-time metrics.
   */
  public val metrics: List<Metric>
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance)
 */
public class PerformanceDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out PerformanceEvent>> =
      mapOf(
      "Performance.metrics" to serializer<PerformanceEvent.MetricsEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<PerformanceEvent> = session.events(deserializersByEventName)

  /**
   * Current values of the metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#event-metrics)
   */
  public fun metrics(): Flow<PerformanceEvent.MetricsEvent> = session.events("Performance.metrics")

  /**
   * Disable collecting and reporting metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Performance.disable", Unit)

  /**
   * Enable collecting and reporting metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#method-enable)
   */
  public suspend fun enable(input: EnableRequest): Unit = session.request("Performance.enable",
      input)

  /**
   * Sets time domain to use for collecting and reporting duration metrics.
   * Note that this must be called before enabling metrics collection. Calling
   * this method while metrics collection is enabled returns an error.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#method-setTimeDomain)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setTimeDomain(input: SetTimeDomainRequest): Unit =
      session.request("Performance.setTimeDomain", input)

  /**
   * Retrieve current values of run-time metrics.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Performance/#method-getMetrics)
   */
  public suspend fun getMetrics(): GetMetricsResponse = session.request("Performance.getMetrics",
      Unit)
}
