@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.runtime

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.runtime.events.RuntimeEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [RuntimeDomain.awaitPromise] command.
 */
@Serializable
public data class AwaitPromiseRequest(
  /**
   * Identifier of the promise.
   */
  public val promiseObjectId: RemoteObjectId,
  /**
   * Whether the result is expected to be a JSON object that should be sent by value.
   */
  public val returnByValue: Boolean? = null,
  /**
   * Whether preview should be generated for the result.
   */
  public val generatePreview: Boolean? = null
)

/**
 * Response type for the [RuntimeDomain.awaitPromise] command.
 */
@Serializable
public data class AwaitPromiseResponse(
  /**
   * Promise result. Will contain rejected value if promise was rejected.
   */
  public val result: RemoteObject,
  /**
   * Exception details if stack strace is available.
   */
  public val exceptionDetails: ExceptionDetails? = null
)

/**
 * Request object containing input parameters for the [RuntimeDomain.callFunctionOn] command.
 */
@Serializable
public data class CallFunctionOnRequest(
  /**
   * Declaration of the function to call.
   */
  public val functionDeclaration: String,
  /**
   * Identifier of the object to call function on. Either objectId or executionContextId should
   * be specified.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * Call arguments. All call arguments must belong to the same JavaScript world as the target
   * object.
   */
  public val arguments: List<CallArgument>? = null,
  /**
   * In silent mode exceptions thrown during evaluation are not reported and do not pause
   * execution. Overrides `setPauseOnException` state.
   */
  public val silent: Boolean? = null,
  /**
   * Whether the result is expected to be a JSON object which should be sent by value.
   */
  public val returnByValue: Boolean? = null,
  /**
   * Whether preview should be generated for the result.
   */
  @ExperimentalChromeApi
  public val generatePreview: Boolean? = null,
  /**
   * Whether execution should be treated as initiated by user in the UI.
   */
  public val userGesture: Boolean? = null,
  /**
   * Whether execution should `await` for resulting value and return once awaited promise is
   * resolved.
   */
  public val awaitPromise: Boolean? = null,
  /**
   * Specifies execution context which global object will be used to call function on. Either
   * executionContextId or objectId should be specified.
   */
  public val executionContextId: ExecutionContextId? = null,
  /**
   * Symbolic group name that can be used to release multiple objects. If objectGroup is not
   * specified and objectId is, objectGroup will be inherited from object.
   */
  public val objectGroup: String? = null
)

/**
 * Response type for the [RuntimeDomain.callFunctionOn] command.
 */
@Serializable
public data class CallFunctionOnResponse(
  /**
   * Call result.
   */
  public val result: RemoteObject,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null
)

/**
 * Request object containing input parameters for the [RuntimeDomain.compileScript] command.
 */
@Serializable
public data class CompileScriptRequest(
  /**
   * Expression to compile.
   */
  public val expression: String,
  /**
   * Source url to be set for the script.
   */
  public val sourceURL: String,
  /**
   * Specifies whether the compiled script should be persisted.
   */
  public val persistScript: Boolean,
  /**
   * Specifies in which execution context to perform script run. If the parameter is omitted the
   * evaluation will be performed in the context of the inspected page.
   */
  public val executionContextId: ExecutionContextId? = null
)

/**
 * Response type for the [RuntimeDomain.compileScript] command.
 */
@Serializable
public data class CompileScriptResponse(
  /**
   * Id of the script.
   */
  public val scriptId: ScriptId? = null,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null
)

/**
 * Request object containing input parameters for the [RuntimeDomain.evaluate] command.
 */
@Serializable
public data class EvaluateRequest(
  /**
   * Expression to evaluate.
   */
  public val expression: String,
  /**
   * Symbolic group name that can be used to release multiple objects.
   */
  public val objectGroup: String? = null,
  /**
   * Determines whether Command Line API should be available during the evaluation.
   */
  public val includeCommandLineAPI: Boolean? = null,
  /**
   * In silent mode exceptions thrown during evaluation are not reported and do not pause
   * execution. Overrides `setPauseOnException` state.
   */
  public val silent: Boolean? = null,
  /**
   * Specifies in which execution context to perform evaluation. If the parameter is omitted the
   * evaluation will be performed in the context of the inspected page.
   */
  public val contextId: ExecutionContextId? = null,
  /**
   * Whether the result is expected to be a JSON object that should be sent by value.
   */
  public val returnByValue: Boolean? = null,
  /**
   * Whether preview should be generated for the result.
   */
  @ExperimentalChromeApi
  public val generatePreview: Boolean? = null,
  /**
   * Whether execution should be treated as initiated by user in the UI.
   */
  public val userGesture: Boolean? = null,
  /**
   * Whether execution should `await` for resulting value and return once awaited promise is
   * resolved.
   */
  public val awaitPromise: Boolean? = null,
  /**
   * Whether to throw an exception if side effect cannot be ruled out during evaluation.
   * This implies `disableBreaks` below.
   */
  @ExperimentalChromeApi
  public val throwOnSideEffect: Boolean? = null,
  /**
   * Terminate execution after timing out (number of milliseconds).
   */
  @ExperimentalChromeApi
  public val timeout: TimeDelta? = null,
  /**
   * Disable breakpoints during execution.
   */
  @ExperimentalChromeApi
  public val disableBreaks: Boolean? = null,
  /**
   * Setting this flag to true enables `let` re-declaration and top-level `await`.
   * Note that `let` variables can only be re-declared if they originate from
   * `replMode` themselves.
   */
  @ExperimentalChromeApi
  public val replMode: Boolean? = null,
  /**
   * The Content Security Policy (CSP) for the target might block 'unsafe-eval'
   * which includes eval(), Function(), setTimeout() and setInterval()
   * when called with non-callable arguments. This flag bypasses CSP for this
   * evaluation and allows unsafe-eval. Defaults to true.
   */
  @ExperimentalChromeApi
  public val allowUnsafeEvalBlockedByCSP: Boolean? = null
)

/**
 * Response type for the [RuntimeDomain.evaluate] command.
 */
@Serializable
public data class EvaluateResponse(
  /**
   * Evaluation result.
   */
  public val result: RemoteObject,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null
)

/**
 * Response type for the [RuntimeDomain.getIsolateId] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetIsolateIdResponse(
  /**
   * The isolate id.
   */
  public val id: String
)

/**
 * Response type for the [RuntimeDomain.getHeapUsage] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetHeapUsageResponse(
  /**
   * Used heap size in bytes.
   */
  public val usedSize: Double,
  /**
   * Allocated heap size in bytes.
   */
  public val totalSize: Double
)

/**
 * Request object containing input parameters for the [RuntimeDomain.getProperties] command.
 */
@Serializable
public data class GetPropertiesRequest(
  /**
   * Identifier of the object to return properties for.
   */
  public val objectId: RemoteObjectId,
  /**
   * If true, returns properties belonging only to the element itself, not to its prototype
   * chain.
   */
  public val ownProperties: Boolean? = null,
  /**
   * If true, returns accessor properties (with getter/setter) only; internal properties are not
   * returned either.
   */
  @ExperimentalChromeApi
  public val accessorPropertiesOnly: Boolean? = null,
  /**
   * Whether preview should be generated for the results.
   */
  @ExperimentalChromeApi
  public val generatePreview: Boolean? = null
)

/**
 * Response type for the [RuntimeDomain.getProperties] command.
 */
@Serializable
public data class GetPropertiesResponse(
  /**
   * Object properties.
   */
  public val result: List<PropertyDescriptor>,
  /**
   * Internal object properties (only of the element itself).
   */
  public val internalProperties: List<InternalPropertyDescriptor>? = null,
  /**
   * Object private properties.
   */
  @ExperimentalChromeApi
  public val privateProperties: List<PrivatePropertyDescriptor>? = null,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null
)

/**
 * Request object containing input parameters for the [RuntimeDomain.globalLexicalScopeNames]
 * command.
 */
@Serializable
public data class GlobalLexicalScopeNamesRequest(
  /**
   * Specifies in which execution context to lookup global scope variables.
   */
  public val executionContextId: ExecutionContextId? = null
)

/**
 * Response type for the [RuntimeDomain.globalLexicalScopeNames] command.
 */
@Serializable
public data class GlobalLexicalScopeNamesResponse(
  public val names: List<String>
)

/**
 * Request object containing input parameters for the [RuntimeDomain.queryObjects] command.
 */
@Serializable
public data class QueryObjectsRequest(
  /**
   * Identifier of the prototype to return objects for.
   */
  public val prototypeObjectId: RemoteObjectId,
  /**
   * Symbolic group name that can be used to release the results.
   */
  public val objectGroup: String? = null
)

/**
 * Response type for the [RuntimeDomain.queryObjects] command.
 */
@Serializable
public data class QueryObjectsResponse(
  /**
   * Array with objects.
   */
  public val objects: RemoteObject
)

/**
 * Request object containing input parameters for the [RuntimeDomain.releaseObject] command.
 */
@Serializable
public data class ReleaseObjectRequest(
  /**
   * Identifier of the object to release.
   */
  public val objectId: RemoteObjectId
)

/**
 * Request object containing input parameters for the [RuntimeDomain.releaseObjectGroup] command.
 */
@Serializable
public data class ReleaseObjectGroupRequest(
  /**
   * Symbolic object group name.
   */
  public val objectGroup: String
)

/**
 * Request object containing input parameters for the [RuntimeDomain.runScript] command.
 */
@Serializable
public data class RunScriptRequest(
  /**
   * Id of the script to run.
   */
  public val scriptId: ScriptId,
  /**
   * Specifies in which execution context to perform script run. If the parameter is omitted the
   * evaluation will be performed in the context of the inspected page.
   */
  public val executionContextId: ExecutionContextId? = null,
  /**
   * Symbolic group name that can be used to release multiple objects.
   */
  public val objectGroup: String? = null,
  /**
   * In silent mode exceptions thrown during evaluation are not reported and do not pause
   * execution. Overrides `setPauseOnException` state.
   */
  public val silent: Boolean? = null,
  /**
   * Determines whether Command Line API should be available during the evaluation.
   */
  public val includeCommandLineAPI: Boolean? = null,
  /**
   * Whether the result is expected to be a JSON object which should be sent by value.
   */
  public val returnByValue: Boolean? = null,
  /**
   * Whether preview should be generated for the result.
   */
  public val generatePreview: Boolean? = null,
  /**
   * Whether execution should `await` for resulting value and return once awaited promise is
   * resolved.
   */
  public val awaitPromise: Boolean? = null
)

/**
 * Response type for the [RuntimeDomain.runScript] command.
 */
@Serializable
public data class RunScriptResponse(
  /**
   * Run result.
   */
  public val result: RemoteObject,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null
)

/**
 * Request object containing input parameters for the [RuntimeDomain.setAsyncCallStackDepth]
 * command.
 */
@Serializable
public data class SetAsyncCallStackDepthRequest(
  /**
   * Maximum depth of async call stacks. Setting to `0` will effectively disable collecting async
   * call stacks (default).
   */
  public val maxDepth: Int
)

/**
 * Request object containing input parameters for the
 * [RuntimeDomain.setCustomObjectFormatterEnabled] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetCustomObjectFormatterEnabledRequest(
  public val enabled: Boolean
)

/**
 * Request object containing input parameters for the [RuntimeDomain.setMaxCallStackSizeToCapture]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetMaxCallStackSizeToCaptureRequest(
  public val size: Int
)

/**
 * Request object containing input parameters for the [RuntimeDomain.addBinding] command.
 */
@Serializable
@ExperimentalChromeApi
public data class AddBindingRequest(
  public val name: String,
  public val executionContextId: ExecutionContextId? = null
)

/**
 * Request object containing input parameters for the [RuntimeDomain.removeBinding] command.
 */
@Serializable
@ExperimentalChromeApi
public data class RemoveBindingRequest(
  public val name: String
)

/**
 * Runtime domain exposes JavaScript runtime by means of remote evaluation and mirror objects.
 * Evaluation results are returned as mirror object that expose object type, string representation
 * and unique identifier that can be used for further object reference. Original objects are
 * maintained in memory unless they are either explicitly released or are released along with the
 * other objects in their object group.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime)
 */
public class RuntimeDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out RuntimeEvent>> =
      mapOf(
      "Runtime.bindingCalled" to serializer<RuntimeEvent.BindingCalledEvent>(),
      "Runtime.consoleAPICalled" to serializer<RuntimeEvent.ConsoleAPICalledEvent>(),
      "Runtime.exceptionRevoked" to serializer<RuntimeEvent.ExceptionRevokedEvent>(),
      "Runtime.exceptionThrown" to serializer<RuntimeEvent.ExceptionThrownEvent>(),
      "Runtime.executionContextCreated" to serializer<RuntimeEvent.ExecutionContextCreatedEvent>(),
      "Runtime.executionContextDestroyed" to
          serializer<RuntimeEvent.ExecutionContextDestroyedEvent>(),
      "Runtime.executionContextsCleared" to
          serializer<RuntimeEvent.ExecutionContextsClearedEvent>(),
      "Runtime.inspectRequested" to serializer<RuntimeEvent.InspectRequestedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<RuntimeEvent> = session.events(deserializersByEventName)

  /**
   * Notification is issued every time when binding is called.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-bindingCalled)
   */
  @ExperimentalChromeApi
  public fun bindingCalled(): Flow<RuntimeEvent.BindingCalledEvent> =
      session.events("Runtime.bindingCalled")

  /**
   * Issued when console API was called.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-consoleAPICalled)
   */
  public fun consoleAPICalled(): Flow<RuntimeEvent.ConsoleAPICalledEvent> =
      session.events("Runtime.consoleAPICalled")

  /**
   * Issued when unhandled exception was revoked.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-exceptionRevoked)
   */
  public fun exceptionRevoked(): Flow<RuntimeEvent.ExceptionRevokedEvent> =
      session.events("Runtime.exceptionRevoked")

  /**
   * Issued when exception was thrown and unhandled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-exceptionThrown)
   */
  public fun exceptionThrown(): Flow<RuntimeEvent.ExceptionThrownEvent> =
      session.events("Runtime.exceptionThrown")

  /**
   * Issued when new execution context is created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-executionContextCreated)
   */
  public fun executionContextCreated(): Flow<RuntimeEvent.ExecutionContextCreatedEvent> =
      session.events("Runtime.executionContextCreated")

  /**
   * Issued when execution context is destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-executionContextDestroyed)
   */
  public fun executionContextDestroyed(): Flow<RuntimeEvent.ExecutionContextDestroyedEvent> =
      session.events("Runtime.executionContextDestroyed")

  /**
   * Issued when all executionContexts were cleared in browser
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-executionContextsCleared)
   */
  public fun executionContextsCleared(): Flow<RuntimeEvent.ExecutionContextsClearedEvent> =
      session.events("Runtime.executionContextsCleared")

  /**
   * Issued when object should be inspected (for example, as a result of inspect() command line API
   * call).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#event-inspectRequested)
   */
  public fun inspectRequested(): Flow<RuntimeEvent.InspectRequestedEvent> =
      session.events("Runtime.inspectRequested")

  /**
   * Add handler to promise with given promise object id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-awaitPromise)
   */
  public suspend fun awaitPromise(input: AwaitPromiseRequest): AwaitPromiseResponse =
      session.request("Runtime.awaitPromise", input)

  /**
   * Calls function with given declaration on the given object. Object group of the result is
   * inherited from the target object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-callFunctionOn)
   */
  public suspend fun callFunctionOn(input: CallFunctionOnRequest): CallFunctionOnResponse =
      session.request("Runtime.callFunctionOn", input)

  /**
   * Compiles expression.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-compileScript)
   */
  public suspend fun compileScript(input: CompileScriptRequest): CompileScriptResponse =
      session.request("Runtime.compileScript", input)

  /**
   * Disables reporting of execution contexts creation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Runtime.disable", Unit)

  /**
   * Discards collected exceptions and console API calls.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-discardConsoleEntries)
   */
  public suspend fun discardConsoleEntries(): Unit =
      session.request("Runtime.discardConsoleEntries", Unit)

  /**
   * Enables reporting of execution contexts creation by means of `executionContextCreated` event.
   * When the reporting gets enabled the event will be sent immediately for each existing execution
   * context.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("Runtime.enable", Unit)

  /**
   * Evaluates expression on global object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-evaluate)
   */
  public suspend fun evaluate(input: EvaluateRequest): EvaluateResponse =
      session.request("Runtime.evaluate", input)

  /**
   * Returns the isolate id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-getIsolateId)
   */
  @ExperimentalChromeApi
  public suspend fun getIsolateId(): GetIsolateIdResponse = session.request("Runtime.getIsolateId",
      Unit)

  /**
   * Returns the JavaScript heap usage.
   * It is the total usage of the corresponding isolate not scoped to a particular Runtime.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-getHeapUsage)
   */
  @ExperimentalChromeApi
  public suspend fun getHeapUsage(): GetHeapUsageResponse = session.request("Runtime.getHeapUsage",
      Unit)

  /**
   * Returns properties of a given object. Object group of the result is inherited from the target
   * object.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-getProperties)
   */
  public suspend fun getProperties(input: GetPropertiesRequest): GetPropertiesResponse =
      session.request("Runtime.getProperties", input)

  /**
   * Returns all let, const and class variables from global scope.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-globalLexicalScopeNames)
   */
  public suspend fun globalLexicalScopeNames(input: GlobalLexicalScopeNamesRequest):
      GlobalLexicalScopeNamesResponse = session.request("Runtime.globalLexicalScopeNames", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-queryObjects)
   */
  public suspend fun queryObjects(input: QueryObjectsRequest): QueryObjectsResponse =
      session.request("Runtime.queryObjects", input)

  /**
   * Releases remote object with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-releaseObject)
   */
  public suspend fun releaseObject(input: ReleaseObjectRequest): Unit =
      session.request("Runtime.releaseObject", input)

  /**
   * Releases all remote objects that belong to a given group.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-releaseObjectGroup)
   */
  public suspend fun releaseObjectGroup(input: ReleaseObjectGroupRequest): Unit =
      session.request("Runtime.releaseObjectGroup", input)

  /**
   * Tells inspected instance to run if it was waiting for debugger to attach.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-runIfWaitingForDebugger)
   */
  public suspend fun runIfWaitingForDebugger(): Unit =
      session.request("Runtime.runIfWaitingForDebugger", Unit)

  /**
   * Runs script with given id in a given context.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-runScript)
   */
  public suspend fun runScript(input: RunScriptRequest): RunScriptResponse =
      session.request("Runtime.runScript", input)

  /**
   * Enables or disables async call stacks tracking.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-setAsyncCallStackDepth)
   */
  public suspend fun setAsyncCallStackDepth(input: SetAsyncCallStackDepthRequest): Unit =
      session.request("Runtime.setAsyncCallStackDepth", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-setCustomObjectFormatterEnabled)
   */
  @ExperimentalChromeApi
  public suspend fun setCustomObjectFormatterEnabled(input: SetCustomObjectFormatterEnabledRequest):
      Unit = session.request("Runtime.setCustomObjectFormatterEnabled", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-setMaxCallStackSizeToCapture)
   */
  @ExperimentalChromeApi
  public suspend fun setMaxCallStackSizeToCapture(input: SetMaxCallStackSizeToCaptureRequest): Unit
      = session.request("Runtime.setMaxCallStackSizeToCapture", input)

  /**
   * Terminate current or next JavaScript execution.
   * Will cancel the termination when the outer-most script execution ends.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-terminateExecution)
   */
  @ExperimentalChromeApi
  public suspend fun terminateExecution(): Unit = session.request("Runtime.terminateExecution",
      Unit)

  /**
   * If executionContextId is empty, adds binding with the given name on the
   * global objects of all inspected contexts, including those created later,
   * bindings survive reloads.
   * If executionContextId is specified, adds binding only on global object of
   * given execution context.
   * Binding function takes exactly one argument, this argument should be string,
   * in case of any other input, function throws an exception.
   * Each binding function call produces Runtime.bindingCalled notification.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-addBinding)
   */
  @ExperimentalChromeApi
  public suspend fun addBinding(input: AddBindingRequest): Unit =
      session.request("Runtime.addBinding", input)

  /**
   * This method does not remove binding function from global object but
   * unsubscribes current runtime agent from Runtime.bindingCalled notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Runtime/#method-removeBinding)
   */
  @ExperimentalChromeApi
  public suspend fun removeBinding(input: RemoveBindingRequest): Unit =
      session.request("Runtime.removeBinding", input)
}
