@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.fetch

import kotlin.String
import kotlin.Suppress
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.network.ResourceType

/**
 * Unique request identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#type-RequestId)
 */
public typealias RequestId = String

/**
 * Stages of the request to handle. Request will intercept before the request is
 * sent. Response will intercept after the response is received (but before response
 * body is received.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#type-RequestStage)
 */
@Serializable
public enum class RequestStage {
  Request,
  Response,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#type-RequestPattern)
 */
@Serializable
public data class RequestPattern(
  /**
   * Wildcards ('*' -> zero or more, '?' -> exactly one) are allowed. Escape character is
   * backslash. Omitting is equivalent to "*".
   */
  public val urlPattern: String? = null,
  /**
   * If set, only requests for matching resource types will be intercepted.
   */
  public val resourceType: ResourceType? = null,
  /**
   * Stage at wich to begin intercepting requests. Default is Request.
   */
  public val requestStage: RequestStage? = null
)

/**
 * Response HTTP header entry
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#type-HeaderEntry)
 */
@Serializable
public data class HeaderEntry(
  public val name: String,
  public val value: String
)

/**
 * Authorization challenge for HTTP status code 401 or 407.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#type-AuthChallenge)
 */
@Serializable
public data class AuthChallenge(
  /**
   * Source of the authentication challenge.
   */
  public val source: String? = null,
  /**
   * Origin of the challenger.
   */
  public val origin: String,
  /**
   * The authentication scheme used, such as basic or digest
   */
  public val scheme: String,
  /**
   * The realm of the challenge. May be empty.
   */
  public val realm: String
)

/**
 * Response to an AuthChallenge.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Fetch/#type-AuthChallengeResponse)
 */
@Serializable
public data class AuthChallengeResponse(
  /**
   * The decision on what to do in response to the authorization challenge.  Default means
   * deferring to the default behavior of the net stack, which will likely either the Cancel
   * authentication or display a popup dialog box.
   */
  public val response: String,
  /**
   * The username to provide, possibly empty. Should only be set if response is
   * ProvideCredentials.
   */
  public val username: String? = null,
  /**
   * The password to provide, possibly empty. Should only be set if response is
   * ProvideCredentials.
   */
  public val password: String? = null
)
