@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.indexeddb

import kotlin.Boolean
import kotlin.Double
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.runtime.RemoteObject

/**
 * Database with an array of object stores.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#type-DatabaseWithObjectStores)
 */
@Serializable
public data class DatabaseWithObjectStores(
  /**
   * Database name.
   */
  public val name: String,
  /**
   * Database version (type is not 'integer', as the standard
   * requires the version number to be 'unsigned long long')
   */
  public val version: Double,
  /**
   * Object stores in this database.
   */
  public val objectStores: List<ObjectStore>
)

/**
 * Object store.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#type-ObjectStore)
 */
@Serializable
public data class ObjectStore(
  /**
   * Object store name.
   */
  public val name: String,
  /**
   * Object store key path.
   */
  public val keyPath: KeyPath,
  /**
   * If true, object store has auto increment flag set.
   */
  public val autoIncrement: Boolean,
  /**
   * Indexes in this object store.
   */
  public val indexes: List<ObjectStoreIndex>
)

/**
 * Object store index.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#type-ObjectStoreIndex)
 */
@Serializable
public data class ObjectStoreIndex(
  /**
   * Index name.
   */
  public val name: String,
  /**
   * Index key path.
   */
  public val keyPath: KeyPath,
  /**
   * If true, index is unique.
   */
  public val unique: Boolean,
  /**
   * If true, index allows multiple entries for a key.
   */
  public val multiEntry: Boolean
)

/**
 * Key.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#type-Key)
 */
@Serializable
public data class Key(
  /**
   * Key type.
   */
  public val type: String,
  /**
   * Number value.
   */
  public val number: Double? = null,
  /**
   * String value.
   */
  public val string: String? = null,
  /**
   * Date value.
   */
  public val date: Double? = null,
  /**
   * Array value.
   */
  public val array: List<Key>? = null
)

/**
 * Key range.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#type-KeyRange)
 */
@Serializable
public data class KeyRange(
  /**
   * Lower bound.
   */
  public val lower: Key? = null,
  /**
   * Upper bound.
   */
  public val upper: Key? = null,
  /**
   * If true lower bound is open.
   */
  public val lowerOpen: Boolean,
  /**
   * If true upper bound is open.
   */
  public val upperOpen: Boolean
)

/**
 * Data entry.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#type-DataEntry)
 */
@Serializable
public data class DataEntry(
  /**
   * Key object.
   */
  public val key: RemoteObject,
  /**
   * Primary key object.
   */
  public val primaryKey: RemoteObject,
  /**
   * Value object.
   */
  public val value: RemoteObject
)

/**
 * Key path.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/IndexedDB/#type-KeyPath)
 */
@Serializable
public data class KeyPath(
  /**
   * Key path type.
   */
  public val type: String,
  /**
   * String value.
   */
  public val string: String? = null,
  /**
   * Array value.
   */
  public val array: List<String>? = null
)
