@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.database

import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.database.events.DatabaseEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [DatabaseDomain.executeSQL] command.
 */
@Serializable
public data class ExecuteSQLRequest(
  public val databaseId: DatabaseId,
  public val query: String
)

/**
 * Response type for the [DatabaseDomain.executeSQL] command.
 */
@Serializable
public data class ExecuteSQLResponse(
  public val columnNames: List<String>? = null,
  public val values: List<JsonElement>? = null,
  public val sqlError: Error? = null
)

/**
 * Request object containing input parameters for the [DatabaseDomain.getDatabaseTableNames]
 * command.
 */
@Serializable
public data class GetDatabaseTableNamesRequest(
  public val databaseId: DatabaseId
)

/**
 * Response type for the [DatabaseDomain.getDatabaseTableNames] command.
 */
@Serializable
public data class GetDatabaseTableNamesResponse(
  public val tableNames: List<String>
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database)
 */
@ExperimentalChromeApi
public class DatabaseDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out DatabaseEvent>> =
      mapOf(
      "Database.addDatabase" to serializer<DatabaseEvent.AddDatabaseEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<DatabaseEvent> = session.events(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#event-addDatabase)
   */
  public fun addDatabase(): Flow<DatabaseEvent.AddDatabaseEvent> =
      session.events("Database.addDatabase")

  /**
   * Disables database tracking, prevents database events from being sent to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Database.disable", Unit)

  /**
   * Enables database tracking, database events will now be delivered to the client.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("Database.enable", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#method-executeSQL)
   */
  public suspend fun executeSQL(input: ExecuteSQLRequest): ExecuteSQLResponse =
      session.request("Database.executeSQL", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Database/#method-getDatabaseTableNames)
   */
  public suspend fun getDatabaseTableNames(input: GetDatabaseTableNamesRequest):
      GetDatabaseTableNamesResponse = session.request("Database.getDatabaseTableNames", input)
}
