@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.media.events

import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.media.PlayerError
import org.hildan.chrome.devtools.domains.media.PlayerEvent
import org.hildan.chrome.devtools.domains.media.PlayerId
import org.hildan.chrome.devtools.domains.media.PlayerMessage
import org.hildan.chrome.devtools.domains.media.PlayerProperty

@Serializable
public sealed class MediaEvent {
  /**
   * This can be called multiple times, and can be used to set / override /
   * remove player properties. A null propValue indicates removal.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerPropertiesChanged)
   */
  @Serializable
  public data class PlayerPropertiesChangedEvent(
    public val playerId: PlayerId,
    public val properties: List<PlayerProperty>
  ) : MediaEvent()

  /**
   * Send events as a list, allowing them to be batched on the browser for less
   * congestion. If batched, events must ALWAYS be in chronological order.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerEventsAdded)
   */
  @Serializable
  public data class PlayerEventsAddedEvent(
    public val playerId: PlayerId,
    public val events: List<PlayerEvent>
  ) : MediaEvent()

  /**
   * Send a list of any messages that need to be delivered.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerMessagesLogged)
   */
  @Serializable
  public data class PlayerMessagesLoggedEvent(
    public val playerId: PlayerId,
    public val messages: List<PlayerMessage>
  ) : MediaEvent()

  /**
   * Send a list of any errors that need to be delivered.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerErrorsRaised)
   */
  @Serializable
  public data class PlayerErrorsRaisedEvent(
    public val playerId: PlayerId,
    public val errors: List<PlayerError>
  ) : MediaEvent()

  /**
   * Called whenever a player is created, or when a new agent joins and recieves
   * a list of active players. If an agent is restored, it will recieve the full
   * list of player ids and all events again.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playersCreated)
   */
  @Serializable
  public data class PlayersCreatedEvent(
    public val players: List<PlayerId>
  ) : MediaEvent()
}
