@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.serviceworker

import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.serviceworker.events.ServiceWorkerEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.deliverPushMessage]
 * command.
 */
@Serializable
public data class DeliverPushMessageRequest(
  public val origin: String,
  public val registrationId: RegistrationID,
  public val `data`: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.dispatchSyncEvent]
 * command.
 */
@Serializable
public data class DispatchSyncEventRequest(
  public val origin: String,
  public val registrationId: RegistrationID,
  public val tag: String,
  public val lastChance: Boolean
)

/**
 * Request object containing input parameters for the
 * [ServiceWorkerDomain.dispatchPeriodicSyncEvent] command.
 */
@Serializable
public data class DispatchPeriodicSyncEventRequest(
  public val origin: String,
  public val registrationId: RegistrationID,
  public val tag: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.inspectWorker] command.
 */
@Serializable
public data class InspectWorkerRequest(
  public val versionId: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.setForceUpdateOnPageLoad]
 * command.
 */
@Serializable
public data class SetForceUpdateOnPageLoadRequest(
  public val forceUpdateOnPageLoad: Boolean
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.skipWaiting] command.
 */
@Serializable
public data class SkipWaitingRequest(
  public val scopeURL: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.startWorker] command.
 */
@Serializable
public data class StartWorkerRequest(
  public val scopeURL: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.stopWorker] command.
 */
@Serializable
public data class StopWorkerRequest(
  public val versionId: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.unregister] command.
 */
@Serializable
public data class UnregisterRequest(
  public val scopeURL: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.updateRegistration]
 * command.
 */
@Serializable
public data class UpdateRegistrationRequest(
  public val scopeURL: String
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker)
 */
@ExperimentalChromeApi
public class ServiceWorkerDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out ServiceWorkerEvent>>
      = mapOf(
      "ServiceWorker.workerErrorReported" to
          serializer<ServiceWorkerEvent.WorkerErrorReportedEvent>(),
      "ServiceWorker.workerRegistrationUpdated" to
          serializer<ServiceWorkerEvent.WorkerRegistrationUpdatedEvent>(),
      "ServiceWorker.workerVersionUpdated" to
          serializer<ServiceWorkerEvent.WorkerVersionUpdatedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<ServiceWorkerEvent> = session.events(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#event-workerErrorReported)
   */
  public fun workerErrorReported(): Flow<ServiceWorkerEvent.WorkerErrorReportedEvent> =
      session.events("ServiceWorker.workerErrorReported")

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#event-workerRegistrationUpdated)
   */
  public fun workerRegistrationUpdated(): Flow<ServiceWorkerEvent.WorkerRegistrationUpdatedEvent> =
      session.events("ServiceWorker.workerRegistrationUpdated")

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#event-workerVersionUpdated)
   */
  public fun workerVersionUpdated(): Flow<ServiceWorkerEvent.WorkerVersionUpdatedEvent> =
      session.events("ServiceWorker.workerVersionUpdated")

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-deliverPushMessage)
   */
  public suspend fun deliverPushMessage(input: DeliverPushMessageRequest): Unit =
      session.request("ServiceWorker.deliverPushMessage", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("ServiceWorker.disable", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-dispatchSyncEvent)
   */
  public suspend fun dispatchSyncEvent(input: DispatchSyncEventRequest): Unit =
      session.request("ServiceWorker.dispatchSyncEvent", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-dispatchPeriodicSyncEvent)
   */
  public suspend fun dispatchPeriodicSyncEvent(input: DispatchPeriodicSyncEventRequest): Unit =
      session.request("ServiceWorker.dispatchPeriodicSyncEvent", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("ServiceWorker.enable", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-inspectWorker)
   */
  public suspend fun inspectWorker(input: InspectWorkerRequest): Unit =
      session.request("ServiceWorker.inspectWorker", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-setForceUpdateOnPageLoad)
   */
  public suspend fun setForceUpdateOnPageLoad(input: SetForceUpdateOnPageLoadRequest): Unit =
      session.request("ServiceWorker.setForceUpdateOnPageLoad", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-skipWaiting)
   */
  public suspend fun skipWaiting(input: SkipWaitingRequest): Unit =
      session.request("ServiceWorker.skipWaiting", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-startWorker)
   */
  public suspend fun startWorker(input: StartWorkerRequest): Unit =
      session.request("ServiceWorker.startWorker", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-stopAllWorkers)
   */
  public suspend fun stopAllWorkers(): Unit = session.request("ServiceWorker.stopAllWorkers", Unit)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-stopWorker)
   */
  public suspend fun stopWorker(input: StopWorkerRequest): Unit =
      session.request("ServiceWorker.stopWorker", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-unregister)
   */
  public suspend fun unregister(input: UnregisterRequest): Unit =
      session.request("ServiceWorker.unregister", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/ServiceWorker/#method-updateRegistration)
   */
  public suspend fun updateRegistration(input: UpdateRegistrationRequest): Unit =
      session.request("ServiceWorker.updateRegistration", input)
}
