@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.animation

import kotlin.Boolean
import kotlin.Double
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.animation.events.AnimationEvent
import org.hildan.chrome.devtools.domains.runtime.RemoteObject
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [AnimationDomain.getCurrentTime] command.
 */
@Serializable
public data class GetCurrentTimeRequest(
  /**
   * Id of animation.
   */
  public val id: String
)

/**
 * Response type for the [AnimationDomain.getCurrentTime] command.
 */
@Serializable
public data class GetCurrentTimeResponse(
  /**
   * Current time of the page.
   */
  public val currentTime: Double
)

/**
 * Response type for the [AnimationDomain.getPlaybackRate] command.
 */
@Serializable
public data class GetPlaybackRateResponse(
  /**
   * Playback rate for animations on page.
   */
  public val playbackRate: Double
)

/**
 * Request object containing input parameters for the [AnimationDomain.releaseAnimations] command.
 */
@Serializable
public data class ReleaseAnimationsRequest(
  /**
   * List of animation ids to seek.
   */
  public val animations: List<String>
)

/**
 * Request object containing input parameters for the [AnimationDomain.resolveAnimation] command.
 */
@Serializable
public data class ResolveAnimationRequest(
  /**
   * Animation id.
   */
  public val animationId: String
)

/**
 * Response type for the [AnimationDomain.resolveAnimation] command.
 */
@Serializable
public data class ResolveAnimationResponse(
  /**
   * Corresponding remote object.
   */
  public val remoteObject: RemoteObject
)

/**
 * Request object containing input parameters for the [AnimationDomain.seekAnimations] command.
 */
@Serializable
public data class SeekAnimationsRequest(
  /**
   * List of animation ids to seek.
   */
  public val animations: List<String>,
  /**
   * Set the current time of each animation.
   */
  public val currentTime: Double
)

/**
 * Request object containing input parameters for the [AnimationDomain.setPaused] command.
 */
@Serializable
public data class SetPausedRequest(
  /**
   * Animations to set the pause state of.
   */
  public val animations: List<String>,
  /**
   * Paused state to set to.
   */
  public val paused: Boolean
)

/**
 * Request object containing input parameters for the [AnimationDomain.setPlaybackRate] command.
 */
@Serializable
public data class SetPlaybackRateRequest(
  /**
   * Playback rate for animations on page
   */
  public val playbackRate: Double
)

/**
 * Request object containing input parameters for the [AnimationDomain.setTiming] command.
 */
@Serializable
public data class SetTimingRequest(
  /**
   * Animation id.
   */
  public val animationId: String,
  /**
   * Duration of the animation.
   */
  public val duration: Double,
  /**
   * Delay of the animation.
   */
  public val delay: Double
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation)
 */
@ExperimentalChromeApi
public class AnimationDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out AnimationEvent>> =
      mapOf(
      "Animation.animationCanceled" to serializer<AnimationEvent.AnimationCanceledEvent>(),
      "Animation.animationCreated" to serializer<AnimationEvent.AnimationCreatedEvent>(),
      "Animation.animationStarted" to serializer<AnimationEvent.AnimationStartedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<AnimationEvent> = session.events(deserializersByEventName)

  /**
   * Event for when an animation has been cancelled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#event-animationCanceled)
   */
  public fun animationCanceled(): Flow<AnimationEvent.AnimationCanceledEvent> =
      session.events("Animation.animationCanceled")

  /**
   * Event for each animation that has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#event-animationCreated)
   */
  public fun animationCreated(): Flow<AnimationEvent.AnimationCreatedEvent> =
      session.events("Animation.animationCreated")

  /**
   * Event for animation that has been started.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#event-animationStarted)
   */
  public fun animationStarted(): Flow<AnimationEvent.AnimationStartedEvent> =
      session.events("Animation.animationStarted")

  /**
   * Disables animation domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Animation.disable", Unit)

  /**
   * Enables animation domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("Animation.enable", Unit)

  /**
   * Returns the current time of the an animation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-getCurrentTime)
   */
  public suspend fun getCurrentTime(input: GetCurrentTimeRequest): GetCurrentTimeResponse =
      session.request("Animation.getCurrentTime", input)

  /**
   * Gets the playback rate of the document timeline.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-getPlaybackRate)
   */
  public suspend fun getPlaybackRate(): GetPlaybackRateResponse =
      session.request("Animation.getPlaybackRate", Unit)

  /**
   * Releases a set of animations to no longer be manipulated.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-releaseAnimations)
   */
  public suspend fun releaseAnimations(input: ReleaseAnimationsRequest): Unit =
      session.request("Animation.releaseAnimations", input)

  /**
   * Gets the remote object of the Animation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-resolveAnimation)
   */
  public suspend fun resolveAnimation(input: ResolveAnimationRequest): ResolveAnimationResponse =
      session.request("Animation.resolveAnimation", input)

  /**
   * Seek a set of animations to a particular time within each animation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-seekAnimations)
   */
  public suspend fun seekAnimations(input: SeekAnimationsRequest): Unit =
      session.request("Animation.seekAnimations", input)

  /**
   * Sets the paused state of a set of animations.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-setPaused)
   */
  public suspend fun setPaused(input: SetPausedRequest): Unit =
      session.request("Animation.setPaused", input)

  /**
   * Sets the playback rate of the document timeline.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-setPlaybackRate)
   */
  public suspend fun setPlaybackRate(input: SetPlaybackRateRequest): Unit =
      session.request("Animation.setPlaybackRate", input)

  /**
   * Sets the timing of an animation node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Animation/#method-setTiming)
   */
  public suspend fun setTiming(input: SetTimingRequest): Unit =
      session.request("Animation.setTiming", input)
}
