@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.cast

import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.cast.events.CastEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [CastDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  public val presentationUrl: String? = null
)

/**
 * Request object containing input parameters for the [CastDomain.setSinkToUse] command.
 */
@Serializable
public data class SetSinkToUseRequest(
  public val sinkName: String
)

/**
 * Request object containing input parameters for the [CastDomain.startTabMirroring] command.
 */
@Serializable
public data class StartTabMirroringRequest(
  public val sinkName: String
)

/**
 * Request object containing input parameters for the [CastDomain.stopCasting] command.
 */
@Serializable
public data class StopCastingRequest(
  public val sinkName: String
)

/**
 * A domain for interacting with Cast, Presentation API, and Remote Playback API
 * functionalities.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast)
 */
@ExperimentalChromeApi
public class CastDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out CastEvent>> = mapOf(
      "Cast.sinksUpdated" to serializer<CastEvent.SinksUpdatedEvent>(),
      "Cast.issueUpdated" to serializer<CastEvent.IssueUpdatedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<CastEvent> = session.events(deserializersByEventName)

  /**
   * This is fired whenever the list of available sinks changes. A sink is a
   * device or a software surface that you can cast to.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#event-sinksUpdated)
   */
  public fun sinksUpdated(): Flow<CastEvent.SinksUpdatedEvent> = session.events("Cast.sinksUpdated")

  /**
   * This is fired whenever the outstanding issue/error message changes.
   * |issueMessage| is empty if there is no issue.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#event-issueUpdated)
   */
  public fun issueUpdated(): Flow<CastEvent.IssueUpdatedEvent> = session.events("Cast.issueUpdated")

  /**
   * Starts observing for sinks that can be used for tab mirroring, and if set,
   * sinks compatible with |presentationUrl| as well. When sinks are found, a
   * |sinksUpdated| event is fired.
   * Also starts observing for issue messages. When an issue is added or removed,
   * an |issueUpdated| event is fired.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-enable)
   */
  public suspend fun enable(input: EnableRequest): Unit = session.request("Cast.enable", input)

  /**
   * Stops observing for sinks and issues.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Cast.disable", Unit)

  /**
   * Sets a sink to be used when the web page requests the browser to choose a
   * sink via Presentation API, Remote Playback API, or Cast SDK.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-setSinkToUse)
   */
  public suspend fun setSinkToUse(input: SetSinkToUseRequest): Unit =
      session.request("Cast.setSinkToUse", input)

  /**
   * Starts mirroring the tab to the sink.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-startTabMirroring)
   */
  public suspend fun startTabMirroring(input: StartTabMirroringRequest): Unit =
      session.request("Cast.startTabMirroring", input)

  /**
   * Stops the active Cast session on the sink.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Cast/#method-stopCasting)
   */
  public suspend fun stopCasting(input: StopCastingRequest): Unit =
      session.request("Cast.stopCasting", input)
}
