@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.dom

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import org.hildan.chrome.devtools.domains.page.FrameId

/**
 * Unique DOM node identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-NodeId)
 */
public typealias NodeId = Int

/**
 * Unique DOM node identifier used to reference a node that may not have been pushed to the
 * front-end.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-BackendNodeId)
 */
public typealias BackendNodeId = Int

/**
 * Backend node with a friendly name.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-BackendNode)
 */
@Serializable
public data class BackendNode(
  /**
   * `Node`'s nodeType.
   */
  public val nodeType: Int,
  /**
   * `Node`'s nodeName.
   */
  public val nodeName: String,
  public val backendNodeId: BackendNodeId
)

/**
 * Pseudo element type.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-PseudoType)
 */
@Serializable
public enum class PseudoType {
  `first-line`,
  `first-letter`,
  before,
  after,
  marker,
  backdrop,
  selection,
  `target-text`,
  `spelling-error`,
  `grammar-error`,
  `first-line-inherited`,
  scrollbar,
  `scrollbar-thumb`,
  `scrollbar-button`,
  `scrollbar-track`,
  `scrollbar-track-piece`,
  `scrollbar-corner`,
  resizer,
  `input-list-button`,
}

/**
 * Shadow root type.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-ShadowRootType)
 */
@Serializable
public enum class ShadowRootType {
  `user-agent`,
  `open`,
  closed,
}

/**
 * DOM interaction is implemented in terms of mirror objects that represent the actual DOM nodes.
 * DOMNode is a base node mirror type.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-Node)
 */
@Serializable
public data class Node(
  /**
   * Node identifier that is passed into the rest of the DOM messages as the `nodeId`. Backend
   * will only push node with given `id` once. It is aware of all requested nodes and will only
   * fire DOM events for nodes known to the client.
   */
  public val nodeId: NodeId,
  /**
   * The id of the parent node if any.
   */
  public val parentId: NodeId? = null,
  /**
   * The BackendNodeId for this node.
   */
  public val backendNodeId: BackendNodeId,
  /**
   * `Node`'s nodeType.
   */
  public val nodeType: Int,
  /**
   * `Node`'s nodeName.
   */
  public val nodeName: String,
  /**
   * `Node`'s localName.
   */
  public val localName: String,
  /**
   * `Node`'s nodeValue.
   */
  public val nodeValue: String,
  /**
   * Child count for `Container` nodes.
   */
  public val childNodeCount: Int? = null,
  /**
   * Child nodes of this node when requested with children.
   */
  public val children: List<Node>? = null,
  /**
   * Attributes of the `Element` node in the form of flat array `[name1, value1, name2, value2]`.
   */
  public val attributes: List<String>? = null,
  /**
   * Document URL that `Document` or `FrameOwner` node points to.
   */
  public val documentURL: String? = null,
  /**
   * Base URL that `Document` or `FrameOwner` node uses for URL completion.
   */
  public val baseURL: String? = null,
  /**
   * `DocumentType`'s publicId.
   */
  public val publicId: String? = null,
  /**
   * `DocumentType`'s systemId.
   */
  public val systemId: String? = null,
  /**
   * `DocumentType`'s internalSubset.
   */
  public val internalSubset: String? = null,
  /**
   * `Document`'s XML version in case of XML documents.
   */
  public val xmlVersion: String? = null,
  /**
   * `Attr`'s name.
   */
  public val name: String? = null,
  /**
   * `Attr`'s value.
   */
  public val value: String? = null,
  /**
   * Pseudo element type for this node.
   */
  public val pseudoType: PseudoType? = null,
  /**
   * Shadow root type.
   */
  public val shadowRootType: ShadowRootType? = null,
  /**
   * Frame ID for frame owner elements.
   */
  public val frameId: FrameId? = null,
  /**
   * Content document for frame owner elements.
   */
  public val contentDocument: Node? = null,
  /**
   * Shadow root list for given element host.
   */
  public val shadowRoots: List<Node>? = null,
  /**
   * Content document fragment for template elements.
   */
  public val templateContent: Node? = null,
  /**
   * Pseudo elements associated with this node.
   */
  public val pseudoElements: List<Node>? = null,
  /**
   * Import document for the HTMLImport links.
   */
  public val importedDocument: Node? = null,
  /**
   * Distributed nodes for given insertion point.
   */
  public val distributedNodes: List<BackendNode>? = null,
  /**
   * Whether the node is SVG.
   */
  public val isSVG: Boolean? = null
)

/**
 * A structure holding an RGBA color.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-RGBA)
 */
@Serializable
public data class RGBA(
  /**
   * The red component, in the [0-255] range.
   */
  public val r: Int,
  /**
   * The green component, in the [0-255] range.
   */
  public val g: Int,
  /**
   * The blue component, in the [0-255] range.
   */
  public val b: Int,
  /**
   * The alpha component, in the [0-1] range (default: 1).
   */
  public val a: Double? = null
)

/**
 * An array of quad vertices, x immediately followed by y for each point, points clock-wise.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-Quad)
 */
public typealias Quad = List<Double>

/**
 * Box model.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-BoxModel)
 */
@Serializable
public data class BoxModel(
  /**
   * Content box
   */
  public val content: Quad,
  /**
   * Padding box
   */
  public val padding: Quad,
  /**
   * Border box
   */
  public val border: Quad,
  /**
   * Margin box
   */
  public val margin: Quad,
  /**
   * Node width
   */
  public val width: Int,
  /**
   * Node height
   */
  public val height: Int,
  /**
   * Shape outside coordinates
   */
  public val shapeOutside: ShapeOutsideInfo? = null
)

/**
 * CSS Shape Outside details.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-ShapeOutsideInfo)
 */
@Serializable
public data class ShapeOutsideInfo(
  /**
   * Shape bounds
   */
  public val bounds: Quad,
  /**
   * Shape coordinate details
   */
  public val shape: List<JsonElement>,
  /**
   * Margin shape bounds
   */
  public val marginShape: List<JsonElement>
)

/**
 * Rectangle.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-Rect)
 */
@Serializable
public data class Rect(
  /**
   * X coordinate
   */
  public val x: Double,
  /**
   * Y coordinate
   */
  public val y: Double,
  /**
   * Rectangle width
   */
  public val width: Double,
  /**
   * Rectangle height
   */
  public val height: Double
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#type-CSSComputedStyleProperty)
 */
@Serializable
public data class CSSComputedStyleProperty(
  /**
   * Computed style property name.
   */
  public val name: String,
  /**
   * Computed style property value.
   */
  public val value: String
)
