package org.hildan.chrome.devtools.domains.css

import kotlin.Boolean
import kotlin.Double
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.css.events.CSSEvent
import org.hildan.chrome.devtools.domains.dom.NodeId
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [CSSDomain.addRule] command.
 */
@Serializable
public data class AddRuleRequest(
  /**
   * The css style sheet identifier where a new rule should be inserted.
   */
  public val styleSheetId: StyleSheetId,
  /**
   * The text of a new rule.
   */
  public val ruleText: String,
  /**
   * Text position of a new rule in the target style sheet.
   */
  public val location: SourceRange
)

/**
 * Response type for the [CSSDomain.addRule] command.
 */
@Serializable
public data class AddRuleResponse(
  /**
   * The newly created rule.
   */
  public val rule: CSSRule
)

/**
 * Request object containing input parameters for the [CSSDomain.collectClassNames] command.
 */
@Serializable
public data class CollectClassNamesRequest(
  public val styleSheetId: StyleSheetId
)

/**
 * Response type for the [CSSDomain.collectClassNames] command.
 */
@Serializable
public data class CollectClassNamesResponse(
  /**
   * Class name list.
   */
  public val classNames: List<String>
)

/**
 * Request object containing input parameters for the [CSSDomain.createStyleSheet] command.
 */
@Serializable
public data class CreateStyleSheetRequest(
  /**
   * Identifier of the frame where "via-inspector" stylesheet should be created.
   */
  public val frameId: FrameId
)

/**
 * Response type for the [CSSDomain.createStyleSheet] command.
 */
@Serializable
public data class CreateStyleSheetResponse(
  /**
   * Identifier of the created "via-inspector" stylesheet.
   */
  public val styleSheetId: StyleSheetId
)

/**
 * Request object containing input parameters for the [CSSDomain.forcePseudoState] command.
 */
@Serializable
public data class ForcePseudoStateRequest(
  /**
   * The element id for which to force the pseudo state.
   */
  public val nodeId: NodeId,
  /**
   * Element pseudo classes to force when computing the element's style.
   */
  public val forcedPseudoClasses: List<String>
)

/**
 * Request object containing input parameters for the [CSSDomain.getBackgroundColors] command.
 */
@Serializable
public data class GetBackgroundColorsRequest(
  /**
   * Id of the node to get background colors for.
   */
  public val nodeId: NodeId
)

/**
 * Response type for the [CSSDomain.getBackgroundColors] command.
 */
@Serializable
public data class GetBackgroundColorsResponse(
  /**
   * The range of background colors behind this element, if it contains any visible text. If no
   * visible text is present, this will be undefined. In the case of a flat background color,
   * this will consist of simply that color. In the case of a gradient, this will consist of each
   * of the color stops. For anything more complicated, this will be an empty array. Images will
   * be ignored (as if the image had failed to load).
   */
  public val backgroundColors: List<String>? = null,
  /**
   * The computed font size for this node, as a CSS computed value string (e.g. '12px').
   */
  public val computedFontSize: String? = null,
  /**
   * The computed font weight for this node, as a CSS computed value string (e.g. 'normal' or
   * '100').
   */
  public val computedFontWeight: String? = null
)

/**
 * Request object containing input parameters for the [CSSDomain.getComputedStyleForNode] command.
 */
@Serializable
public data class GetComputedStyleForNodeRequest(
  public val nodeId: NodeId
)

/**
 * Response type for the [CSSDomain.getComputedStyleForNode] command.
 */
@Serializable
public data class GetComputedStyleForNodeResponse(
  /**
   * Computed style for the specified DOM node.
   */
  public val computedStyle: List<CSSComputedStyleProperty>
)

/**
 * Request object containing input parameters for the [CSSDomain.getInlineStylesForNode] command.
 */
@Serializable
public data class GetInlineStylesForNodeRequest(
  public val nodeId: NodeId
)

/**
 * Response type for the [CSSDomain.getInlineStylesForNode] command.
 */
@Serializable
public data class GetInlineStylesForNodeResponse(
  /**
   * Inline style for the specified DOM node.
   */
  public val inlineStyle: CSSStyle? = null,
  /**
   * Attribute-defined element style (e.g. resulting from "width=20 height=100%").
   */
  public val attributesStyle: CSSStyle? = null
)

/**
 * Request object containing input parameters for the [CSSDomain.getMatchedStylesForNode] command.
 */
@Serializable
public data class GetMatchedStylesForNodeRequest(
  public val nodeId: NodeId
)

/**
 * Response type for the [CSSDomain.getMatchedStylesForNode] command.
 */
@Serializable
public data class GetMatchedStylesForNodeResponse(
  /**
   * Inline style for the specified DOM node.
   */
  public val inlineStyle: CSSStyle? = null,
  /**
   * Attribute-defined element style (e.g. resulting from "width=20 height=100%").
   */
  public val attributesStyle: CSSStyle? = null,
  /**
   * CSS rules matching this node, from all applicable stylesheets.
   */
  public val matchedCSSRules: List<RuleMatch>? = null,
  /**
   * Pseudo style matches for this node.
   */
  public val pseudoElements: List<PseudoElementMatches>? = null,
  /**
   * A chain of inherited styles (from the immediate node parent up to the DOM tree root).
   */
  public val inherited: List<InheritedStyleEntry>? = null,
  /**
   * A list of CSS keyframed animations matching this node.
   */
  public val cssKeyframesRules: List<CSSKeyframesRule>? = null
)

/**
 * Response type for the [CSSDomain.getMediaQueries] command.
 */
@Serializable
public data class GetMediaQueriesResponse(
  public val medias: List<CSSMedia>
)

/**
 * Request object containing input parameters for the [CSSDomain.getPlatformFontsForNode] command.
 */
@Serializable
public data class GetPlatformFontsForNodeRequest(
  public val nodeId: NodeId
)

/**
 * Response type for the [CSSDomain.getPlatformFontsForNode] command.
 */
@Serializable
public data class GetPlatformFontsForNodeResponse(
  /**
   * Usage statistics for every employed platform font.
   */
  public val fonts: List<PlatformFontUsage>
)

/**
 * Request object containing input parameters for the [CSSDomain.getStyleSheetText] command.
 */
@Serializable
public data class GetStyleSheetTextRequest(
  public val styleSheetId: StyleSheetId
)

/**
 * Response type for the [CSSDomain.getStyleSheetText] command.
 */
@Serializable
public data class GetStyleSheetTextResponse(
  /**
   * The stylesheet text.
   */
  public val text: String
)

/**
 * Request object containing input parameters for the [CSSDomain.trackComputedStyleUpdates] command.
 */
@Serializable
@ExperimentalChromeApi
public data class TrackComputedStyleUpdatesRequest(
  public val propertiesToTrack: List<CSSComputedStyleProperty>
)

/**
 * Response type for the [CSSDomain.takeComputedStyleUpdates] command.
 */
@Serializable
@ExperimentalChromeApi
public data class TakeComputedStyleUpdatesResponse(
  /**
   * The list of node Ids that have their tracked computed styles updated
   */
  public val nodeIds: List<NodeId>
)

/**
 * Request object containing input parameters for the [CSSDomain.setEffectivePropertyValueForNode]
 * command.
 */
@Serializable
public data class SetEffectivePropertyValueForNodeRequest(
  /**
   * The element id for which to set property.
   */
  public val nodeId: NodeId,
  public val propertyName: String,
  public val value: String
)

/**
 * Request object containing input parameters for the [CSSDomain.setKeyframeKey] command.
 */
@Serializable
public data class SetKeyframeKeyRequest(
  public val styleSheetId: StyleSheetId,
  public val range: SourceRange,
  public val keyText: String
)

/**
 * Response type for the [CSSDomain.setKeyframeKey] command.
 */
@Serializable
public data class SetKeyframeKeyResponse(
  /**
   * The resulting key text after modification.
   */
  public val keyText: Value
)

/**
 * Request object containing input parameters for the [CSSDomain.setMediaText] command.
 */
@Serializable
public data class SetMediaTextRequest(
  public val styleSheetId: StyleSheetId,
  public val range: SourceRange,
  public val text: String
)

/**
 * Response type for the [CSSDomain.setMediaText] command.
 */
@Serializable
public data class SetMediaTextResponse(
  /**
   * The resulting CSS media rule after modification.
   */
  public val media: CSSMedia
)

/**
 * Request object containing input parameters for the [CSSDomain.setRuleSelector] command.
 */
@Serializable
public data class SetRuleSelectorRequest(
  public val styleSheetId: StyleSheetId,
  public val range: SourceRange,
  public val selector: String
)

/**
 * Response type for the [CSSDomain.setRuleSelector] command.
 */
@Serializable
public data class SetRuleSelectorResponse(
  /**
   * The resulting selector list after modification.
   */
  public val selectorList: SelectorList
)

/**
 * Request object containing input parameters for the [CSSDomain.setStyleSheetText] command.
 */
@Serializable
public data class SetStyleSheetTextRequest(
  public val styleSheetId: StyleSheetId,
  public val text: String
)

/**
 * Response type for the [CSSDomain.setStyleSheetText] command.
 */
@Serializable
public data class SetStyleSheetTextResponse(
  /**
   * URL of source map associated with script (if any).
   */
  public val sourceMapURL: String? = null
)

/**
 * Request object containing input parameters for the [CSSDomain.setStyleTexts] command.
 */
@Serializable
public data class SetStyleTextsRequest(
  public val edits: List<StyleDeclarationEdit>
)

/**
 * Response type for the [CSSDomain.setStyleTexts] command.
 */
@Serializable
public data class SetStyleTextsResponse(
  /**
   * The resulting styles after modification.
   */
  public val styles: List<CSSStyle>
)

/**
 * Response type for the [CSSDomain.stopRuleUsageTracking] command.
 */
@Serializable
public data class StopRuleUsageTrackingResponse(
  public val ruleUsage: List<RuleUsage>
)

/**
 * Response type for the [CSSDomain.takeCoverageDelta] command.
 */
@Serializable
public data class TakeCoverageDeltaResponse(
  public val coverage: List<RuleUsage>,
  /**
   * Monotonically increasing time, in seconds.
   */
  public val timestamp: Double
)

/**
 * Request object containing input parameters for the [CSSDomain.setLocalFontsEnabled] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetLocalFontsEnabledRequest(
  /**
   * Whether rendering of local fonts is enabled.
   */
  public val enabled: Boolean
)

/**
 * This domain exposes CSS read/write operations. All CSS objects (stylesheets, rules, and styles)
 * have an associated `id` used in subsequent operations on the related object. Each object type has
 * a specific `id` structure, and those are not interchangeable between objects of different kinds.
 * CSS objects can be loaded using the `get*ForNode()` calls (which accept a DOM node id). A client
 * can also keep track of stylesheets via the `styleSheetAdded`/`styleSheetRemoved` events and
 * subsequently load the required stylesheet contents using the `getStyleSheet[Text]()` methods.
 */
@ExperimentalChromeApi
public class CSSDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out CSSEvent>> = mapOf(
      "CSS.fontsUpdated" to serializer<CSSEvent.FontsUpdatedEvent>(),
      "CSS.mediaQueryResultChanged" to serializer<CSSEvent.MediaQueryResultChangedEvent>(),
      "CSS.styleSheetAdded" to serializer<CSSEvent.StyleSheetAddedEvent>(),
      "CSS.styleSheetChanged" to serializer<CSSEvent.StyleSheetChangedEvent>(),
      "CSS.styleSheetRemoved" to serializer<CSSEvent.StyleSheetRemovedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<CSSEvent> = session.events(deserializersByEventName)

  /**
   * Fires whenever a web font is updated.  A non-empty font parameter indicates a successfully
   * loaded
   * web font
   */
  public fun fontsUpdated(): Flow<CSSEvent.FontsUpdatedEvent> = session.events("CSS.fontsUpdated")

  /**
   * Fires whenever a MediaQuery result changes (for example, after a browser window has been
   * resized.) The current implementation considers only viewport-dependent media features.
   */
  public fun mediaQueryResultChanged(): Flow<CSSEvent.MediaQueryResultChangedEvent> =
      session.events("CSS.mediaQueryResultChanged")

  /**
   * Fired whenever an active document stylesheet is added.
   */
  public fun styleSheetAdded(): Flow<CSSEvent.StyleSheetAddedEvent> =
      session.events("CSS.styleSheetAdded")

  /**
   * Fired whenever a stylesheet is changed as a result of the client operation.
   */
  public fun styleSheetChanged(): Flow<CSSEvent.StyleSheetChangedEvent> =
      session.events("CSS.styleSheetChanged")

  /**
   * Fired whenever an active document stylesheet is removed.
   */
  public fun styleSheetRemoved(): Flow<CSSEvent.StyleSheetRemovedEvent> =
      session.events("CSS.styleSheetRemoved")

  /**
   * Inserts a new rule with the given `ruleText` in a stylesheet with given `styleSheetId`, at the
   * position specified by `location`.
   */
  public suspend fun addRule(input: AddRuleRequest): AddRuleResponse =
      session.request("CSS.addRule", input)

  /**
   * Returns all class names from specified stylesheet.
   */
  public suspend fun collectClassNames(input: CollectClassNamesRequest): CollectClassNamesResponse =
      session.request("CSS.collectClassNames", input)

  /**
   * Creates a new special "via-inspector" stylesheet in the frame with given `frameId`.
   */
  public suspend fun createStyleSheet(input: CreateStyleSheetRequest): CreateStyleSheetResponse =
      session.request("CSS.createStyleSheet", input)

  /**
   * Disables the CSS agent for the given page.
   */
  public suspend fun disable(): Unit = session.request("CSS.disable", Unit)

  /**
   * Enables the CSS agent for the given page. Clients should not assume that the CSS agent has been
   * enabled until the result of this command is received.
   */
  public suspend fun enable(): Unit = session.request("CSS.enable", Unit)

  /**
   * Ensures that the given node will have specified pseudo-classes whenever its style is computed
   * by
   * the browser.
   */
  public suspend fun forcePseudoState(input: ForcePseudoStateRequest): Unit =
      session.request("CSS.forcePseudoState", input)

  public suspend fun getBackgroundColors(input: GetBackgroundColorsRequest):
      GetBackgroundColorsResponse = session.request("CSS.getBackgroundColors", input)

  /**
   * Returns the computed style for a DOM node identified by `nodeId`.
   */
  public suspend fun getComputedStyleForNode(input: GetComputedStyleForNodeRequest):
      GetComputedStyleForNodeResponse = session.request("CSS.getComputedStyleForNode", input)

  /**
   * Returns the styles defined inline (explicitly in the "style" attribute and implicitly, using
   * DOM
   * attributes) for a DOM node identified by `nodeId`.
   */
  public suspend fun getInlineStylesForNode(input: GetInlineStylesForNodeRequest):
      GetInlineStylesForNodeResponse = session.request("CSS.getInlineStylesForNode", input)

  /**
   * Returns requested styles for a DOM node identified by `nodeId`.
   */
  public suspend fun getMatchedStylesForNode(input: GetMatchedStylesForNodeRequest):
      GetMatchedStylesForNodeResponse = session.request("CSS.getMatchedStylesForNode", input)

  /**
   * Returns all media queries parsed by the rendering engine.
   */
  public suspend fun getMediaQueries(): GetMediaQueriesResponse =
      session.request("CSS.getMediaQueries", Unit)

  /**
   * Requests information about platform fonts which we used to render child TextNodes in the given
   * node.
   */
  public suspend fun getPlatformFontsForNode(input: GetPlatformFontsForNodeRequest):
      GetPlatformFontsForNodeResponse = session.request("CSS.getPlatformFontsForNode", input)

  /**
   * Returns the current textual content for a stylesheet.
   */
  public suspend fun getStyleSheetText(input: GetStyleSheetTextRequest): GetStyleSheetTextResponse =
      session.request("CSS.getStyleSheetText", input)

  /**
   * Starts tracking the given computed styles for updates. The specified array of properties
   * replaces the one previously specified. Pass empty array to disable tracking.
   * Use takeComputedStyleUpdates to retrieve the list of nodes that had properties modified.
   * The changes to computed style properties are only tracked for nodes pushed to the front-end
   * by the DOM agent. If no changes to the tracked properties occur after the node has been pushed
   * to the front-end, no updates will be issued for the node.
   */
  @ExperimentalChromeApi
  public suspend fun trackComputedStyleUpdates(input: TrackComputedStyleUpdatesRequest): Unit =
      session.request("CSS.trackComputedStyleUpdates", input)

  /**
   * Polls the next batch of computed style updates.
   */
  @ExperimentalChromeApi
  public suspend fun takeComputedStyleUpdates(): TakeComputedStyleUpdatesResponse =
      session.request("CSS.takeComputedStyleUpdates", Unit)

  /**
   * Find a rule with the given active property for the given node and set the new value for this
   * property
   */
  public suspend
      fun setEffectivePropertyValueForNode(input: SetEffectivePropertyValueForNodeRequest): Unit =
      session.request("CSS.setEffectivePropertyValueForNode", input)

  /**
   * Modifies the keyframe rule key text.
   */
  public suspend fun setKeyframeKey(input: SetKeyframeKeyRequest): SetKeyframeKeyResponse =
      session.request("CSS.setKeyframeKey", input)

  /**
   * Modifies the rule selector.
   */
  public suspend fun setMediaText(input: SetMediaTextRequest): SetMediaTextResponse =
      session.request("CSS.setMediaText", input)

  /**
   * Modifies the rule selector.
   */
  public suspend fun setRuleSelector(input: SetRuleSelectorRequest): SetRuleSelectorResponse =
      session.request("CSS.setRuleSelector", input)

  /**
   * Sets the new stylesheet text.
   */
  public suspend fun setStyleSheetText(input: SetStyleSheetTextRequest): SetStyleSheetTextResponse =
      session.request("CSS.setStyleSheetText", input)

  /**
   * Applies specified style edits one after another in the given order.
   */
  public suspend fun setStyleTexts(input: SetStyleTextsRequest): SetStyleTextsResponse =
      session.request("CSS.setStyleTexts", input)

  /**
   * Enables the selector recording.
   */
  public suspend fun startRuleUsageTracking(): Unit = session.request("CSS.startRuleUsageTracking",
      Unit)

  /**
   * Stop tracking rule usage and return the list of rules that were used since last call to
   * `takeCoverageDelta` (or since start of coverage instrumentation)
   */
  public suspend fun stopRuleUsageTracking(): StopRuleUsageTrackingResponse =
      session.request("CSS.stopRuleUsageTracking", Unit)

  /**
   * Obtain list of rules that became used since last call to this method (or since start of
   * coverage
   * instrumentation)
   */
  public suspend fun takeCoverageDelta(): TakeCoverageDeltaResponse =
      session.request("CSS.takeCoverageDelta", Unit)

  /**
   * Enables/disables rendering of local CSS fonts (enabled by default).
   */
  @ExperimentalChromeApi
  public suspend fun setLocalFontsEnabled(input: SetLocalFontsEnabledRequest): Unit =
      session.request("CSS.setLocalFontsEnabled", input)
}
