package org.hildan.chrome.devtools.domains.domdebugger

import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.NodeId
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.getEventListeners] command.
 */
@Serializable
public data class GetEventListenersRequest(
  /**
   * Identifier of the object to return listeners for.
   */
  public val objectId: RemoteObjectId,
  /**
   * The maximum depth at which Node children should be retrieved, defaults to 1. Use -1 for the
   * entire subtree or provide an integer larger than 0.
   */
  public val depth: Int? = null,
  /**
   * Whether or not iframes and shadow roots should be traversed when returning the subtree
   * (default is false). Reports listeners for all contexts if pierce is enabled.
   */
  public val pierce: Boolean? = null
)

/**
 * Response type for the [DOMDebuggerDomain.getEventListeners] command.
 */
@Serializable
public data class GetEventListenersResponse(
  /**
   * Array of relevant listeners.
   */
  public val listeners: List<EventListener>
)

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.removeDOMBreakpoint]
 * command.
 */
@Serializable
public data class RemoveDOMBreakpointRequest(
  /**
   * Identifier of the node to remove breakpoint from.
   */
  public val nodeId: NodeId,
  /**
   * Type of the breakpoint to remove.
   */
  public val type: DOMBreakpointType
)

/**
 * Request object containing input parameters for the
 * [DOMDebuggerDomain.removeEventListenerBreakpoint] command.
 */
@Serializable
public data class RemoveEventListenerBreakpointRequest(
  /**
   * Event name.
   */
  public val eventName: String,
  /**
   * EventTarget interface name.
   */
  @ExperimentalChromeApi
  public val targetName: String? = null
)

/**
 * Request object containing input parameters for the
 * [DOMDebuggerDomain.removeInstrumentationBreakpoint] command.
 */
@Serializable
@ExperimentalChromeApi
public data class RemoveInstrumentationBreakpointRequest(
  /**
   * Instrumentation name to stop on.
   */
  public val eventName: String
)

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.removeXHRBreakpoint]
 * command.
 */
@Serializable
public data class RemoveXHRBreakpointRequest(
  /**
   * Resource URL substring.
   */
  public val url: String
)

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.setDOMBreakpoint] command.
 */
@Serializable
public data class SetDOMBreakpointRequest(
  /**
   * Identifier of the node to set breakpoint on.
   */
  public val nodeId: NodeId,
  /**
   * Type of the operation to stop upon.
   */
  public val type: DOMBreakpointType
)

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.setEventListenerBreakpoint]
 * command.
 */
@Serializable
public data class SetEventListenerBreakpointRequest(
  /**
   * DOM Event name to stop on (any DOM event will do).
   */
  public val eventName: String,
  /**
   * EventTarget interface name to stop on. If equal to `"*"` or not provided, will stop on any
   * EventTarget.
   */
  @ExperimentalChromeApi
  public val targetName: String? = null
)

/**
 * Request object containing input parameters for the
 * [DOMDebuggerDomain.setInstrumentationBreakpoint] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetInstrumentationBreakpointRequest(
  /**
   * Instrumentation name to stop on.
   */
  public val eventName: String
)

/**
 * Request object containing input parameters for the [DOMDebuggerDomain.setXHRBreakpoint] command.
 */
@Serializable
public data class SetXHRBreakpointRequest(
  /**
   * Resource URL substring. All XHRs having this substring in the URL will get stopped upon.
   */
  public val url: String
)

/**
 * DOM debugging allows setting breakpoints on particular DOM operations and events. JavaScript
 * execution will stop on these operations as if there was a regular breakpoint set.
 */
public class DOMDebuggerDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Returns event listeners of the given object.
   */
  public suspend fun getEventListeners(input: GetEventListenersRequest): GetEventListenersResponse =
      session.request("DOMDebugger.getEventListeners", input)

  /**
   * Removes DOM breakpoint that was set using `setDOMBreakpoint`.
   */
  public suspend fun removeDOMBreakpoint(input: RemoveDOMBreakpointRequest): Unit =
      session.request("DOMDebugger.removeDOMBreakpoint", input)

  /**
   * Removes breakpoint on particular DOM event.
   */
  public suspend fun removeEventListenerBreakpoint(input: RemoveEventListenerBreakpointRequest):
      Unit = session.request("DOMDebugger.removeEventListenerBreakpoint", input)

  /**
   * Removes breakpoint on particular native event.
   */
  @ExperimentalChromeApi
  public suspend fun removeInstrumentationBreakpoint(input: RemoveInstrumentationBreakpointRequest):
      Unit = session.request("DOMDebugger.removeInstrumentationBreakpoint", input)

  /**
   * Removes breakpoint from XMLHttpRequest.
   */
  public suspend fun removeXHRBreakpoint(input: RemoveXHRBreakpointRequest): Unit =
      session.request("DOMDebugger.removeXHRBreakpoint", input)

  /**
   * Sets breakpoint on particular operation with DOM.
   */
  public suspend fun setDOMBreakpoint(input: SetDOMBreakpointRequest): Unit =
      session.request("DOMDebugger.setDOMBreakpoint", input)

  /**
   * Sets breakpoint on particular DOM event.
   */
  public suspend fun setEventListenerBreakpoint(input: SetEventListenerBreakpointRequest): Unit =
      session.request("DOMDebugger.setEventListenerBreakpoint", input)

  /**
   * Sets breakpoint on particular native event.
   */
  @ExperimentalChromeApi
  public suspend fun setInstrumentationBreakpoint(input: SetInstrumentationBreakpointRequest): Unit
      = session.request("DOMDebugger.setInstrumentationBreakpoint", input)

  /**
   * Sets breakpoint on XMLHttpRequest.
   */
  public suspend fun setXHRBreakpoint(input: SetXHRBreakpointRequest): Unit =
      session.request("DOMDebugger.setXHRBreakpoint", input)
}
