package org.hildan.chrome.devtools.domains.memory

import kotlin.Boolean
import kotlin.Int
import kotlin.Unit
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Response type for the [MemoryDomain.getDOMCounters] command.
 */
@Serializable
public data class GetDOMCountersResponse(
  public val documents: Int,
  public val nodes: Int,
  public val jsEventListeners: Int
)

/**
 * Request object containing input parameters for the
 * [MemoryDomain.setPressureNotificationsSuppressed] command.
 */
@Serializable
public data class SetPressureNotificationsSuppressedRequest(
  /**
   * If true, memory pressure notifications will be suppressed.
   */
  public val suppressed: Boolean
)

/**
 * Request object containing input parameters for the [MemoryDomain.simulatePressureNotification]
 * command.
 */
@Serializable
public data class SimulatePressureNotificationRequest(
  /**
   * Memory pressure level of the notification.
   */
  public val level: PressureLevel
)

/**
 * Request object containing input parameters for the [MemoryDomain.startSampling] command.
 */
@Serializable
public data class StartSamplingRequest(
  /**
   * Average number of bytes between samples.
   */
  public val samplingInterval: Int? = null,
  /**
   * Do not randomize intervals between samples.
   */
  public val suppressRandomness: Boolean? = null
)

/**
 * Response type for the [MemoryDomain.getAllTimeSamplingProfile] command.
 */
@Serializable
public data class GetAllTimeSamplingProfileResponse(
  public val profile: SamplingProfile
)

/**
 * Response type for the [MemoryDomain.getBrowserSamplingProfile] command.
 */
@Serializable
public data class GetBrowserSamplingProfileResponse(
  public val profile: SamplingProfile
)

/**
 * Response type for the [MemoryDomain.getSamplingProfile] command.
 */
@Serializable
public data class GetSamplingProfileResponse(
  public val profile: SamplingProfile
)

@ExperimentalChromeApi
public class MemoryDomain internal constructor(
  private val session: ChromeDPSession
) {
  public suspend fun getDOMCounters(): GetDOMCountersResponse =
      session.request("Memory.getDOMCounters", Unit)

  public suspend fun prepareForLeakDetection(): Unit =
      session.request("Memory.prepareForLeakDetection", Unit)

  /**
   * Simulate OomIntervention by purging V8 memory.
   */
  public suspend fun forciblyPurgeJavaScriptMemory(): Unit =
      session.request("Memory.forciblyPurgeJavaScriptMemory", Unit)

  /**
   * Enable/disable suppressing memory pressure notifications in all processes.
   */
  public suspend
      fun setPressureNotificationsSuppressed(input: SetPressureNotificationsSuppressedRequest): Unit
      = session.request("Memory.setPressureNotificationsSuppressed", input)

  /**
   * Simulate a memory pressure notification in all processes.
   */
  public suspend fun simulatePressureNotification(input: SimulatePressureNotificationRequest): Unit
      = session.request("Memory.simulatePressureNotification", input)

  /**
   * Start collecting native memory profile.
   */
  public suspend fun startSampling(input: StartSamplingRequest): Unit =
      session.request("Memory.startSampling", input)

  /**
   * Stop collecting native memory profile.
   */
  public suspend fun stopSampling(): Unit = session.request("Memory.stopSampling", Unit)

  /**
   * Retrieve native memory allocations profile
   * collected since renderer process startup.
   */
  public suspend fun getAllTimeSamplingProfile(): GetAllTimeSamplingProfileResponse =
      session.request("Memory.getAllTimeSamplingProfile", Unit)

  /**
   * Retrieve native memory allocations profile
   * collected since browser process startup.
   */
  public suspend fun getBrowserSamplingProfile(): GetBrowserSamplingProfileResponse =
      session.request("Memory.getBrowserSamplingProfile", Unit)

  /**
   * Retrieve native memory allocations profile collected since last
   * `startSampling` call.
   */
  public suspend fun getSamplingProfile(): GetSamplingProfileResponse =
      session.request("Memory.getSamplingProfile", Unit)
}
