package org.hildan.chrome.devtools.domains.network

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.debugger.SearchMatch
import org.hildan.chrome.devtools.domains.emulation.UserAgentMetadata
import org.hildan.chrome.devtools.domains.io.StreamHandle
import org.hildan.chrome.devtools.domains.network.events.NetworkEvent
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Response type for the [NetworkDomain.canClearBrowserCache] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class CanClearBrowserCacheResponse(
  /**
   * True if browser cache can be cleared.
   */
  public val result: Boolean
)

/**
 * Response type for the [NetworkDomain.canClearBrowserCookies] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class CanClearBrowserCookiesResponse(
  /**
   * True if browser cookies can be cleared.
   */
  public val result: Boolean
)

/**
 * Response type for the [NetworkDomain.canEmulateNetworkConditions] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class CanEmulateNetworkConditionsResponse(
  /**
   * True if emulation of network conditions is supported.
   */
  public val result: Boolean
)

/**
 * Request object containing input parameters for the [NetworkDomain.continueInterceptedRequest]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class ContinueInterceptedRequestRequest(
  public val interceptionId: InterceptionId,
  /**
   * If set this causes the request to fail with the given reason. Passing `Aborted` for requests
   * marked with `isNavigationRequest` also cancels the navigation. Must not be set in response
   * to an authChallenge.
   */
  public val errorReason: ErrorReason? = null,
  /**
   * If set the requests completes using with the provided base64 encoded raw response, including
   * HTTP status line and headers etc... Must not be set in response to an authChallenge.
   */
  public val rawResponse: String? = null,
  /**
   * If set the request url will be modified in a way that's not observable by page. Must not be
   * set in response to an authChallenge.
   */
  public val url: String? = null,
  /**
   * If set this allows the request method to be overridden. Must not be set in response to an
   * authChallenge.
   */
  public val method: String? = null,
  /**
   * If set this allows postData to be set. Must not be set in response to an authChallenge.
   */
  public val postData: String? = null,
  /**
   * If set this allows the request headers to be changed. Must not be set in response to an
   * authChallenge.
   */
  public val headers: Headers? = null,
  /**
   * Response to a requestIntercepted with an authChallenge. Must not be set otherwise.
   */
  public val authChallengeResponse: AuthChallengeResponse? = null
)

/**
 * Request object containing input parameters for the [NetworkDomain.deleteCookies] command.
 */
@Serializable
public data class DeleteCookiesRequest(
  /**
   * Name of the cookies to remove.
   */
  public val name: String,
  /**
   * If specified, deletes all the cookies with the given name where domain and path match
   * provided URL.
   */
  public val url: String? = null,
  /**
   * If specified, deletes only cookies with the exact domain.
   */
  public val domain: String? = null,
  /**
   * If specified, deletes only cookies with the exact path.
   */
  public val path: String? = null
)

/**
 * Request object containing input parameters for the [NetworkDomain.emulateNetworkConditions]
 * command.
 */
@Serializable
public data class EmulateNetworkConditionsRequest(
  /**
   * True to emulate internet disconnection.
   */
  public val offline: Boolean,
  /**
   * Minimum latency from request sent to response headers received (ms).
   */
  public val latency: Double,
  /**
   * Maximal aggregated download throughput (bytes/sec). -1 disables download throttling.
   */
  public val downloadThroughput: Double,
  /**
   * Maximal aggregated upload throughput (bytes/sec).  -1 disables upload throttling.
   */
  public val uploadThroughput: Double,
  /**
   * Connection type if known.
   */
  public val connectionType: ConnectionType? = null
)

/**
 * Request object containing input parameters for the [NetworkDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * Buffer size in bytes to use when preserving network payloads (XHRs, etc).
   */
  @ExperimentalChromeApi
  public val maxTotalBufferSize: Int? = null,
  /**
   * Per-resource buffer size in bytes to use when preserving network payloads (XHRs, etc).
   */
  @ExperimentalChromeApi
  public val maxResourceBufferSize: Int? = null,
  /**
   * Longest post body size (in bytes) that would be included in requestWillBeSent notification
   */
  public val maxPostDataSize: Int? = null
)

/**
 * Response type for the [NetworkDomain.getAllCookies] command.
 */
@Serializable
public data class GetAllCookiesResponse(
  /**
   * Array of cookie objects.
   */
  public val cookies: List<Cookie>
)

/**
 * Request object containing input parameters for the [NetworkDomain.getCertificate] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetCertificateRequest(
  /**
   * Origin to get certificate for.
   */
  public val origin: String
)

/**
 * Response type for the [NetworkDomain.getCertificate] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetCertificateResponse(
  public val tableNames: List<String>
)

/**
 * Request object containing input parameters for the [NetworkDomain.getCookies] command.
 */
@Serializable
public data class GetCookiesRequest(
  /**
   * The list of URLs for which applicable cookies will be fetched.
   * If not specified, it's assumed to be set to the list containing
   * the URLs of the page and all of its subframes.
   */
  public val urls: List<String>? = null
)

/**
 * Response type for the [NetworkDomain.getCookies] command.
 */
@Serializable
public data class GetCookiesResponse(
  /**
   * Array of cookie objects.
   */
  public val cookies: List<Cookie>
)

/**
 * Request object containing input parameters for the [NetworkDomain.getResponseBody] command.
 */
@Serializable
public data class GetResponseBodyRequest(
  /**
   * Identifier of the network request to get content for.
   */
  public val requestId: RequestId
)

/**
 * Response type for the [NetworkDomain.getResponseBody] command.
 */
@Serializable
public data class GetResponseBodyResponse(
  /**
   * Response body.
   */
  public val body: String,
  /**
   * True, if content was sent as base64.
   */
  public val base64Encoded: Boolean
)

/**
 * Request object containing input parameters for the [NetworkDomain.getRequestPostData] command.
 */
@Serializable
public data class GetRequestPostDataRequest(
  /**
   * Identifier of the network request to get content for.
   */
  public val requestId: RequestId
)

/**
 * Response type for the [NetworkDomain.getRequestPostData] command.
 */
@Serializable
public data class GetRequestPostDataResponse(
  /**
   * Request body string, omitting files from multipart requests
   */
  public val postData: String
)

/**
 * Request object containing input parameters for the [NetworkDomain.getResponseBodyForInterception]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetResponseBodyForInterceptionRequest(
  /**
   * Identifier for the intercepted request to get body for.
   */
  public val interceptionId: InterceptionId
)

/**
 * Response type for the [NetworkDomain.getResponseBodyForInterception] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetResponseBodyForInterceptionResponse(
  /**
   * Response body.
   */
  public val body: String,
  /**
   * True, if content was sent as base64.
   */
  public val base64Encoded: Boolean
)

/**
 * Request object containing input parameters for the
 * [NetworkDomain.takeResponseBodyForInterceptionAsStream] command.
 */
@Serializable
@ExperimentalChromeApi
public data class TakeResponseBodyForInterceptionAsStreamRequest(
  public val interceptionId: InterceptionId
)

/**
 * Response type for the [NetworkDomain.takeResponseBodyForInterceptionAsStream] command.
 */
@Serializable
@ExperimentalChromeApi
public data class TakeResponseBodyForInterceptionAsStreamResponse(
  public val stream: StreamHandle
)

/**
 * Request object containing input parameters for the [NetworkDomain.replayXHR] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ReplayXHRRequest(
  /**
   * Identifier of XHR to replay.
   */
  public val requestId: RequestId
)

/**
 * Request object containing input parameters for the [NetworkDomain.searchInResponseBody] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SearchInResponseBodyRequest(
  /**
   * Identifier of the network response to search.
   */
  public val requestId: RequestId,
  /**
   * String to search for.
   */
  public val query: String,
  /**
   * If true, search is case sensitive.
   */
  public val caseSensitive: Boolean? = null,
  /**
   * If true, treats string parameter as regex.
   */
  public val isRegex: Boolean? = null
)

/**
 * Response type for the [NetworkDomain.searchInResponseBody] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SearchInResponseBodyResponse(
  /**
   * List of search matches.
   */
  public val result: List<SearchMatch>
)

/**
 * Request object containing input parameters for the [NetworkDomain.setBlockedURLs] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBlockedURLsRequest(
  /**
   * URL patterns to block. Wildcards ('*') are allowed.
   */
  public val urls: List<String>
)

/**
 * Request object containing input parameters for the [NetworkDomain.setBypassServiceWorker]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBypassServiceWorkerRequest(
  /**
   * Bypass service worker and load from network.
   */
  public val bypass: Boolean
)

/**
 * Request object containing input parameters for the [NetworkDomain.setCacheDisabled] command.
 */
@Serializable
public data class SetCacheDisabledRequest(
  /**
   * Cache disabled state.
   */
  public val cacheDisabled: Boolean
)

/**
 * Request object containing input parameters for the [NetworkDomain.setCookie] command.
 */
@Serializable
public data class SetCookieRequest(
  /**
   * Cookie name.
   */
  public val name: String,
  /**
   * Cookie value.
   */
  public val value: String,
  /**
   * The request-URI to associate with the setting of the cookie. This value can affect the
   * default domain and path values of the created cookie.
   */
  public val url: String? = null,
  /**
   * Cookie domain.
   */
  public val domain: String? = null,
  /**
   * Cookie path.
   */
  public val path: String? = null,
  /**
   * True if cookie is secure.
   */
  public val secure: Boolean? = null,
  /**
   * True if cookie is http-only.
   */
  public val httpOnly: Boolean? = null,
  /**
   * Cookie SameSite type.
   */
  public val sameSite: CookieSameSite? = null,
  /**
   * Cookie expiration date, session cookie if not set
   */
  public val expires: TimeSinceEpoch? = null,
  /**
   * Cookie Priority type.
   */
  @ExperimentalChromeApi
  public val priority: CookiePriority? = null
)

/**
 * Response type for the [NetworkDomain.setCookie] command.
 */
@Serializable
public data class SetCookieResponse(
  /**
   * Always set to true. If an error occurs, the response indicates protocol error.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val success: Boolean
)

/**
 * Request object containing input parameters for the [NetworkDomain.setCookies] command.
 */
@Serializable
public data class SetCookiesRequest(
  /**
   * Cookies to be set.
   */
  public val cookies: List<CookieParam>
)

/**
 * Request object containing input parameters for the [NetworkDomain.setDataSizeLimitsForTest]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetDataSizeLimitsForTestRequest(
  /**
   * Maximum total buffer size.
   */
  public val maxTotalSize: Int,
  /**
   * Maximum per-resource size.
   */
  public val maxResourceSize: Int
)

/**
 * Request object containing input parameters for the [NetworkDomain.setExtraHTTPHeaders] command.
 */
@Serializable
public data class SetExtraHTTPHeadersRequest(
  /**
   * Map with extra HTTP headers.
   */
  public val headers: Headers
)

/**
 * Request object containing input parameters for the [NetworkDomain.setAttachDebugHeader] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetAttachDebugHeaderRequest(
  /**
   * Whether to send a debug header.
   */
  public val enabled: Boolean
)

/**
 * Request object containing input parameters for the [NetworkDomain.setRequestInterception]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class SetRequestInterceptionRequest(
  /**
   * Requests matching any of these patterns will be forwarded and wait for the corresponding
   * continueInterceptedRequest call.
   */
  public val patterns: List<RequestPattern>
)

/**
 * Request object containing input parameters for the [NetworkDomain.setUserAgentOverride] command.
 */
@Serializable
public data class SetUserAgentOverrideRequest(
  /**
   * User agent to use.
   */
  public val userAgent: String,
  /**
   * Browser langugage to emulate.
   */
  public val acceptLanguage: String? = null,
  /**
   * The platform navigator.platform should return.
   */
  public val platform: String? = null,
  /**
   * To be sent in Sec-CH-UA-* headers and returned in navigator.userAgentData
   */
  @ExperimentalChromeApi
  public val userAgentMetadata: UserAgentMetadata? = null
)

/**
 * Request object containing input parameters for the [NetworkDomain.getSecurityIsolationStatus]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSecurityIsolationStatusRequest(
  /**
   * If no frameId is provided, the status of the target is provided.
   */
  public val frameId: FrameId? = null
)

/**
 * Response type for the [NetworkDomain.getSecurityIsolationStatus] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSecurityIsolationStatusResponse(
  public val status: SecurityIsolationStatus
)

/**
 * Request object containing input parameters for the [NetworkDomain.loadNetworkResource] command.
 */
@Serializable
@ExperimentalChromeApi
public data class LoadNetworkResourceRequest(
  /**
   * Frame id to get the resource for.
   */
  public val frameId: FrameId,
  /**
   * URL of the resource to get content for.
   */
  public val url: String,
  /**
   * Options for the request.
   */
  public val options: LoadNetworkResourceOptions
)

/**
 * Response type for the [NetworkDomain.loadNetworkResource] command.
 */
@Serializable
@ExperimentalChromeApi
public data class LoadNetworkResourceResponse(
  public val resource: LoadNetworkResourcePageResult
)

/**
 * Network domain allows tracking network activities of the page. It exposes information about http,
 * file, data and other requests and responses, their headers, bodies, timing, etc.
 */
public class NetworkDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out NetworkEvent>> =
      mapOf(
      "Network.dataReceived" to serializer<NetworkEvent.DataReceivedEvent>(),
      "Network.eventSourceMessageReceived" to
          serializer<NetworkEvent.EventSourceMessageReceivedEvent>(),
      "Network.loadingFailed" to serializer<NetworkEvent.LoadingFailedEvent>(),
      "Network.loadingFinished" to serializer<NetworkEvent.LoadingFinishedEvent>(),
      "Network.requestIntercepted" to serializer<NetworkEvent.RequestInterceptedEvent>(),
      "Network.requestServedFromCache" to serializer<NetworkEvent.RequestServedFromCacheEvent>(),
      "Network.requestWillBeSent" to serializer<NetworkEvent.RequestWillBeSentEvent>(),
      "Network.resourceChangedPriority" to serializer<NetworkEvent.ResourceChangedPriorityEvent>(),
      "Network.signedExchangeReceived" to serializer<NetworkEvent.SignedExchangeReceivedEvent>(),
      "Network.responseReceived" to serializer<NetworkEvent.ResponseReceivedEvent>(),
      "Network.webSocketClosed" to serializer<NetworkEvent.WebSocketClosedEvent>(),
      "Network.webSocketCreated" to serializer<NetworkEvent.WebSocketCreatedEvent>(),
      "Network.webSocketFrameError" to serializer<NetworkEvent.WebSocketFrameErrorEvent>(),
      "Network.webSocketFrameReceived" to serializer<NetworkEvent.WebSocketFrameReceivedEvent>(),
      "Network.webSocketFrameSent" to serializer<NetworkEvent.WebSocketFrameSentEvent>(),
      "Network.webSocketHandshakeResponseReceived" to
          serializer<NetworkEvent.WebSocketHandshakeResponseReceivedEvent>(),
      "Network.webSocketWillSendHandshakeRequest" to
          serializer<NetworkEvent.WebSocketWillSendHandshakeRequestEvent>(),
      "Network.requestWillBeSentExtraInfo" to
          serializer<NetworkEvent.RequestWillBeSentExtraInfoEvent>(),
      "Network.responseReceivedExtraInfo" to
          serializer<NetworkEvent.ResponseReceivedExtraInfoEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<NetworkEvent> = session.events(deserializersByEventName)

  /**
   * Fired when data chunk was received over the network.
   */
  public fun dataReceived(): Flow<NetworkEvent.DataReceivedEvent> =
      session.events("Network.dataReceived")

  /**
   * Fired when EventSource message is received.
   */
  public fun eventSourceMessageReceived(): Flow<NetworkEvent.EventSourceMessageReceivedEvent> =
      session.events("Network.eventSourceMessageReceived")

  /**
   * Fired when HTTP request has failed to load.
   */
  public fun loadingFailed(): Flow<NetworkEvent.LoadingFailedEvent> =
      session.events("Network.loadingFailed")

  /**
   * Fired when HTTP request has finished loading.
   */
  public fun loadingFinished(): Flow<NetworkEvent.LoadingFinishedEvent> =
      session.events("Network.loadingFinished")

  /**
   * Details of an intercepted HTTP request, which must be either allowed, blocked, modified or
   * mocked.
   * Deprecated, use Fetch.requestPaused instead.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public fun requestIntercepted(): Flow<NetworkEvent.RequestInterceptedEvent> =
      session.events("Network.requestIntercepted")

  /**
   * Fired if request ended up loading from cache.
   */
  public fun requestServedFromCache(): Flow<NetworkEvent.RequestServedFromCacheEvent> =
      session.events("Network.requestServedFromCache")

  /**
   * Fired when page is about to send HTTP request.
   */
  public fun requestWillBeSent(): Flow<NetworkEvent.RequestWillBeSentEvent> =
      session.events("Network.requestWillBeSent")

  /**
   * Fired when resource loading priority is changed
   */
  @ExperimentalChromeApi
  public fun resourceChangedPriority(): Flow<NetworkEvent.ResourceChangedPriorityEvent> =
      session.events("Network.resourceChangedPriority")

  /**
   * Fired when a signed exchange was received over the network
   */
  @ExperimentalChromeApi
  public fun signedExchangeReceived(): Flow<NetworkEvent.SignedExchangeReceivedEvent> =
      session.events("Network.signedExchangeReceived")

  /**
   * Fired when HTTP response is available.
   */
  public fun responseReceived(): Flow<NetworkEvent.ResponseReceivedEvent> =
      session.events("Network.responseReceived")

  /**
   * Fired when WebSocket is closed.
   */
  public fun webSocketClosed(): Flow<NetworkEvent.WebSocketClosedEvent> =
      session.events("Network.webSocketClosed")

  /**
   * Fired upon WebSocket creation.
   */
  public fun webSocketCreated(): Flow<NetworkEvent.WebSocketCreatedEvent> =
      session.events("Network.webSocketCreated")

  /**
   * Fired when WebSocket message error occurs.
   */
  public fun webSocketFrameError(): Flow<NetworkEvent.WebSocketFrameErrorEvent> =
      session.events("Network.webSocketFrameError")

  /**
   * Fired when WebSocket message is received.
   */
  public fun webSocketFrameReceived(): Flow<NetworkEvent.WebSocketFrameReceivedEvent> =
      session.events("Network.webSocketFrameReceived")

  /**
   * Fired when WebSocket message is sent.
   */
  public fun webSocketFrameSent(): Flow<NetworkEvent.WebSocketFrameSentEvent> =
      session.events("Network.webSocketFrameSent")

  /**
   * Fired when WebSocket handshake response becomes available.
   */
  public fun webSocketHandshakeResponseReceived():
      Flow<NetworkEvent.WebSocketHandshakeResponseReceivedEvent> =
      session.events("Network.webSocketHandshakeResponseReceived")

  /**
   * Fired when WebSocket is about to initiate handshake.
   */
  public fun webSocketWillSendHandshakeRequest():
      Flow<NetworkEvent.WebSocketWillSendHandshakeRequestEvent> =
      session.events("Network.webSocketWillSendHandshakeRequest")

  /**
   * Fired when additional information about a requestWillBeSent event is available from the
   * network stack. Not every requestWillBeSent event will have an additional
   * requestWillBeSentExtraInfo fired for it, and there is no guarantee whether requestWillBeSent
   * or requestWillBeSentExtraInfo will be fired first for the same request.
   */
  @ExperimentalChromeApi
  public fun requestWillBeSentExtraInfo(): Flow<NetworkEvent.RequestWillBeSentExtraInfoEvent> =
      session.events("Network.requestWillBeSentExtraInfo")

  /**
   * Fired when additional information about a responseReceived event is available from the network
   * stack. Not every responseReceived event will have an additional responseReceivedExtraInfo for
   * it, and responseReceivedExtraInfo may be fired before or after responseReceived.
   */
  @ExperimentalChromeApi
  public fun responseReceivedExtraInfo(): Flow<NetworkEvent.ResponseReceivedExtraInfoEvent> =
      session.events("Network.responseReceivedExtraInfo")

  /**
   * Tells whether clearing browser cache is supported.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun canClearBrowserCache(): CanClearBrowserCacheResponse =
      session.request("Network.canClearBrowserCache", Unit)

  /**
   * Tells whether clearing browser cookies is supported.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun canClearBrowserCookies(): CanClearBrowserCookiesResponse =
      session.request("Network.canClearBrowserCookies", Unit)

  /**
   * Tells whether emulation of network conditions is supported.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun canEmulateNetworkConditions(): CanEmulateNetworkConditionsResponse =
      session.request("Network.canEmulateNetworkConditions", Unit)

  /**
   * Clears browser cache.
   */
  public suspend fun clearBrowserCache(): Unit = session.request("Network.clearBrowserCache", Unit)

  /**
   * Clears browser cookies.
   */
  public suspend fun clearBrowserCookies(): Unit = session.request("Network.clearBrowserCookies",
      Unit)

  /**
   * Response to Network.requestIntercepted which either modifies the request to continue with any
   * modifications, or blocks it, or completes it with the provided response bytes. If a network
   * fetch occurs as a result which encounters a redirect an additional Network.requestIntercepted
   * event will be sent with the same InterceptionId.
   * Deprecated, use Fetch.continueRequest, Fetch.fulfillRequest and Fetch.failRequest instead.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun continueInterceptedRequest(input: ContinueInterceptedRequestRequest): Unit =
      session.request("Network.continueInterceptedRequest", input)

  /**
   * Deletes browser cookies with matching name and url or domain/path pair.
   */
  public suspend fun deleteCookies(input: DeleteCookiesRequest): Unit =
      session.request("Network.deleteCookies", input)

  /**
   * Disables network tracking, prevents network events from being sent to the client.
   */
  public suspend fun disable(): Unit = session.request("Network.disable", Unit)

  /**
   * Activates emulation of network conditions.
   */
  public suspend fun emulateNetworkConditions(input: EmulateNetworkConditionsRequest): Unit =
      session.request("Network.emulateNetworkConditions", input)

  /**
   * Enables network tracking, network events will now be delivered to the client.
   */
  public suspend fun enable(input: EnableRequest): Unit = session.request("Network.enable", input)

  /**
   * Returns all browser cookies. Depending on the backend support, will return detailed cookie
   * information in the `cookies` field.
   */
  public suspend fun getAllCookies(): GetAllCookiesResponse =
      session.request("Network.getAllCookies", Unit)

  /**
   * Returns the DER-encoded certificate.
   */
  @ExperimentalChromeApi
  public suspend fun getCertificate(input: GetCertificateRequest): GetCertificateResponse =
      session.request("Network.getCertificate", input)

  /**
   * Returns all browser cookies for the current URL. Depending on the backend support, will return
   * detailed cookie information in the `cookies` field.
   */
  public suspend fun getCookies(input: GetCookiesRequest): GetCookiesResponse =
      session.request("Network.getCookies", input)

  /**
   * Returns content served for the given request.
   */
  public suspend fun getResponseBody(input: GetResponseBodyRequest): GetResponseBodyResponse =
      session.request("Network.getResponseBody", input)

  /**
   * Returns post data sent with the request. Returns an error when no data was sent with the
   * request.
   */
  public suspend fun getRequestPostData(input: GetRequestPostDataRequest):
      GetRequestPostDataResponse = session.request("Network.getRequestPostData", input)

  /**
   * Returns content served for the given currently intercepted request.
   */
  @ExperimentalChromeApi
  public suspend fun getResponseBodyForInterception(input: GetResponseBodyForInterceptionRequest):
      GetResponseBodyForInterceptionResponse =
      session.request("Network.getResponseBodyForInterception", input)

  /**
   * Returns a handle to the stream representing the response body. Note that after this command,
   * the intercepted request can't be continued as is -- you either need to cancel it or to provide
   * the response body. The stream only supports sequential read, IO.read will fail if the position
   * is specified.
   */
  @ExperimentalChromeApi
  public suspend
      fun takeResponseBodyForInterceptionAsStream(input: TakeResponseBodyForInterceptionAsStreamRequest):
      TakeResponseBodyForInterceptionAsStreamResponse =
      session.request("Network.takeResponseBodyForInterceptionAsStream", input)

  /**
   * This method sends a new XMLHttpRequest which is identical to the original one. The following
   * parameters should be identical: method, url, async, request body, extra headers,
   * withCredentials
   * attribute, user, password.
   */
  @ExperimentalChromeApi
  public suspend fun replayXHR(input: ReplayXHRRequest): Unit = session.request("Network.replayXHR",
      input)

  /**
   * Searches for given string in response content.
   */
  @ExperimentalChromeApi
  public suspend fun searchInResponseBody(input: SearchInResponseBodyRequest):
      SearchInResponseBodyResponse = session.request("Network.searchInResponseBody", input)

  /**
   * Blocks URLs from loading.
   */
  @ExperimentalChromeApi
  public suspend fun setBlockedURLs(input: SetBlockedURLsRequest): Unit =
      session.request("Network.setBlockedURLs", input)

  /**
   * Toggles ignoring of service worker for each request.
   */
  @ExperimentalChromeApi
  public suspend fun setBypassServiceWorker(input: SetBypassServiceWorkerRequest): Unit =
      session.request("Network.setBypassServiceWorker", input)

  /**
   * Toggles ignoring cache for each request. If `true`, cache will not be used.
   */
  public suspend fun setCacheDisabled(input: SetCacheDisabledRequest): Unit =
      session.request("Network.setCacheDisabled", input)

  /**
   * Sets a cookie with the given cookie data; may overwrite equivalent cookies if they exist.
   */
  public suspend fun setCookie(input: SetCookieRequest): SetCookieResponse =
      session.request("Network.setCookie", input)

  /**
   * Sets given cookies.
   */
  public suspend fun setCookies(input: SetCookiesRequest): Unit =
      session.request("Network.setCookies", input)

  /**
   * For testing.
   */
  @ExperimentalChromeApi
  public suspend fun setDataSizeLimitsForTest(input: SetDataSizeLimitsForTestRequest): Unit =
      session.request("Network.setDataSizeLimitsForTest", input)

  /**
   * Specifies whether to always send extra HTTP headers with the requests from this page.
   */
  public suspend fun setExtraHTTPHeaders(input: SetExtraHTTPHeadersRequest): Unit =
      session.request("Network.setExtraHTTPHeaders", input)

  /**
   * Specifies whether to sned a debug header to all outgoing requests.
   */
  @ExperimentalChromeApi
  public suspend fun setAttachDebugHeader(input: SetAttachDebugHeaderRequest): Unit =
      session.request("Network.setAttachDebugHeader", input)

  /**
   * Sets the requests to intercept that match the provided patterns and optionally resource types.
   * Deprecated, please use Fetch.enable instead.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun setRequestInterception(input: SetRequestInterceptionRequest): Unit =
      session.request("Network.setRequestInterception", input)

  /**
   * Allows overriding user agent with the given string.
   */
  public suspend fun setUserAgentOverride(input: SetUserAgentOverrideRequest): Unit =
      session.request("Network.setUserAgentOverride", input)

  /**
   * Returns information about the COEP/COOP isolation status.
   */
  @ExperimentalChromeApi
  public suspend fun getSecurityIsolationStatus(input: GetSecurityIsolationStatusRequest):
      GetSecurityIsolationStatusResponse = session.request("Network.getSecurityIsolationStatus",
      input)

  /**
   * Fetches the resource and returns the content.
   */
  @ExperimentalChromeApi
  public suspend fun loadNetworkResource(input: LoadNetworkResourceRequest):
      LoadNetworkResourceResponse = session.request("Network.loadNetworkResource", input)
}
