package org.hildan.chrome.devtools.domains.runtime.events

import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import org.hildan.chrome.devtools.domains.runtime.ExceptionDetails
import org.hildan.chrome.devtools.domains.runtime.ExecutionContextDescription
import org.hildan.chrome.devtools.domains.runtime.ExecutionContextId
import org.hildan.chrome.devtools.domains.runtime.RemoteObject
import org.hildan.chrome.devtools.domains.runtime.StackTrace
import org.hildan.chrome.devtools.domains.runtime.Timestamp
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

@Serializable
public sealed class RuntimeEvent {
  @Serializable
  public data class BindingCalledEvent(
    public val name: String,
    public val payload: String,
    /**
     * Identifier of the context where the call was made.
     */
    public val executionContextId: ExecutionContextId
  ) : RuntimeEvent()

  @Serializable
  public data class ConsoleAPICalledEvent(
    /**
     * Type of the call.
     */
    public val type: String,
    /**
     * Call arguments.
     */
    public val args: List<RemoteObject>,
    /**
     * Identifier of the context where the call was made.
     */
    public val executionContextId: ExecutionContextId,
    /**
     * Call timestamp.
     */
    public val timestamp: Timestamp,
    /**
     * Stack trace captured when the call was made. The async stack chain is automatically reported
     * for
     * the following call types: `assert`, `error`, `trace`, `warning`. For other types the async
     * call
     * chain can be retrieved using `Debugger.getStackTrace` and `stackTrace.parentId` field.
     */
    public val stackTrace: StackTrace? = null,
    /**
     * Console context descriptor for calls on non-default console context (not console.*):
     * 'anonymous#unique-logger-id' for call on unnamed context, 'name#unique-logger-id' for call
     * on named context.
     */
    @ExperimentalChromeApi
    public val context: String? = null
  ) : RuntimeEvent()

  @Serializable
  public data class ExceptionRevokedEvent(
    /**
     * Reason describing why exception was revoked.
     */
    public val reason: String,
    /**
     * The id of revoked exception, as reported in `exceptionThrown`.
     */
    public val exceptionId: Int
  ) : RuntimeEvent()

  @Serializable
  public data class ExceptionThrownEvent(
    /**
     * Timestamp of the exception.
     */
    public val timestamp: Timestamp,
    public val exceptionDetails: ExceptionDetails
  ) : RuntimeEvent()

  @Serializable
  public data class ExecutionContextCreatedEvent(
    /**
     * A newly created execution context.
     */
    public val context: ExecutionContextDescription
  ) : RuntimeEvent()

  @Serializable
  public data class ExecutionContextDestroyedEvent(
    /**
     * Id of the destroyed context
     */
    public val executionContextId: ExecutionContextId
  ) : RuntimeEvent()

  @Serializable
  public object ExecutionContextsClearedEvent : RuntimeEvent()

  @Serializable
  public data class InspectRequestedEvent(
    public val `object`: RemoteObject,
    public val hints: JsonElement
  ) : RuntimeEvent()
}
