package org.hildan.chrome.devtools.domains.serviceworker

import kotlin.Boolean
import kotlin.String
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.serviceworker.events.ServiceWorkerEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.deliverPushMessage]
 * command.
 */
@Serializable
public data class DeliverPushMessageRequest(
  public val origin: String,
  public val registrationId: RegistrationID,
  public val `data`: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.dispatchSyncEvent]
 * command.
 */
@Serializable
public data class DispatchSyncEventRequest(
  public val origin: String,
  public val registrationId: RegistrationID,
  public val tag: String,
  public val lastChance: Boolean
)

/**
 * Request object containing input parameters for the
 * [ServiceWorkerDomain.dispatchPeriodicSyncEvent] command.
 */
@Serializable
public data class DispatchPeriodicSyncEventRequest(
  public val origin: String,
  public val registrationId: RegistrationID,
  public val tag: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.inspectWorker] command.
 */
@Serializable
public data class InspectWorkerRequest(
  public val versionId: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.setForceUpdateOnPageLoad]
 * command.
 */
@Serializable
public data class SetForceUpdateOnPageLoadRequest(
  public val forceUpdateOnPageLoad: Boolean
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.skipWaiting] command.
 */
@Serializable
public data class SkipWaitingRequest(
  public val scopeURL: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.startWorker] command.
 */
@Serializable
public data class StartWorkerRequest(
  public val scopeURL: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.stopWorker] command.
 */
@Serializable
public data class StopWorkerRequest(
  public val versionId: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.unregister] command.
 */
@Serializable
public data class UnregisterRequest(
  public val scopeURL: String
)

/**
 * Request object containing input parameters for the [ServiceWorkerDomain.updateRegistration]
 * command.
 */
@Serializable
public data class UpdateRegistrationRequest(
  public val scopeURL: String
)

@ExperimentalChromeApi
public class ServiceWorkerDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out ServiceWorkerEvent>>
      = mapOf(
      "ServiceWorker.workerErrorReported" to
          serializer<ServiceWorkerEvent.WorkerErrorReportedEvent>(),
      "ServiceWorker.workerRegistrationUpdated" to
          serializer<ServiceWorkerEvent.WorkerRegistrationUpdatedEvent>(),
      "ServiceWorker.workerVersionUpdated" to
          serializer<ServiceWorkerEvent.WorkerVersionUpdatedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<ServiceWorkerEvent> = session.events(deserializersByEventName)

  public fun workerErrorReported(): Flow<ServiceWorkerEvent.WorkerErrorReportedEvent> =
      session.events("ServiceWorker.workerErrorReported")

  public fun workerRegistrationUpdated(): Flow<ServiceWorkerEvent.WorkerRegistrationUpdatedEvent> =
      session.events("ServiceWorker.workerRegistrationUpdated")

  public fun workerVersionUpdated(): Flow<ServiceWorkerEvent.WorkerVersionUpdatedEvent> =
      session.events("ServiceWorker.workerVersionUpdated")

  public suspend fun deliverPushMessage(input: DeliverPushMessageRequest): Unit =
      session.request("ServiceWorker.deliverPushMessage", input)

  public suspend fun disable(): Unit = session.request("ServiceWorker.disable", Unit)

  public suspend fun dispatchSyncEvent(input: DispatchSyncEventRequest): Unit =
      session.request("ServiceWorker.dispatchSyncEvent", input)

  public suspend fun dispatchPeriodicSyncEvent(input: DispatchPeriodicSyncEventRequest): Unit =
      session.request("ServiceWorker.dispatchPeriodicSyncEvent", input)

  public suspend fun enable(): Unit = session.request("ServiceWorker.enable", Unit)

  public suspend fun inspectWorker(input: InspectWorkerRequest): Unit =
      session.request("ServiceWorker.inspectWorker", input)

  public suspend fun setForceUpdateOnPageLoad(input: SetForceUpdateOnPageLoadRequest): Unit =
      session.request("ServiceWorker.setForceUpdateOnPageLoad", input)

  public suspend fun skipWaiting(input: SkipWaitingRequest): Unit =
      session.request("ServiceWorker.skipWaiting", input)

  public suspend fun startWorker(input: StartWorkerRequest): Unit =
      session.request("ServiceWorker.startWorker", input)

  public suspend fun stopAllWorkers(): Unit = session.request("ServiceWorker.stopAllWorkers", Unit)

  public suspend fun stopWorker(input: StopWorkerRequest): Unit =
      session.request("ServiceWorker.stopWorker", input)

  public suspend fun unregister(input: UnregisterRequest): Unit =
      session.request("ServiceWorker.unregister", input)

  public suspend fun updateRegistration(input: UpdateRegistrationRequest): Unit =
      session.request("ServiceWorker.updateRegistration", input)
}
