package org.hildan.chrome.devtools.domains.storage

import kotlin.Boolean
import kotlin.Double
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.browser.BrowserContextID
import org.hildan.chrome.devtools.domains.network.Cookie
import org.hildan.chrome.devtools.domains.network.CookieParam
import org.hildan.chrome.devtools.domains.storage.events.StorageEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [StorageDomain.clearDataForOrigin] command.
 */
@Serializable
public data class ClearDataForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String,
  /**
   * Comma separated list of StorageType to clear.
   */
  public val storageTypes: String
)

/**
 * Request object containing input parameters for the [StorageDomain.getCookies] command.
 */
@Serializable
public data class GetCookiesRequest(
  /**
   * Browser context to use when called on the browser endpoint.
   */
  public val browserContextId: BrowserContextID? = null
)

/**
 * Response type for the [StorageDomain.getCookies] command.
 */
@Serializable
public data class GetCookiesResponse(
  /**
   * Array of cookie objects.
   */
  public val cookies: List<Cookie>
)

/**
 * Request object containing input parameters for the [StorageDomain.setCookies] command.
 */
@Serializable
public data class SetCookiesRequest(
  /**
   * Cookies to be set.
   */
  public val cookies: List<CookieParam>,
  /**
   * Browser context to use when called on the browser endpoint.
   */
  public val browserContextId: BrowserContextID? = null
)

/**
 * Request object containing input parameters for the [StorageDomain.clearCookies] command.
 */
@Serializable
public data class ClearCookiesRequest(
  /**
   * Browser context to use when called on the browser endpoint.
   */
  public val browserContextId: BrowserContextID? = null
)

/**
 * Request object containing input parameters for the [StorageDomain.getUsageAndQuota] command.
 */
@Serializable
public data class GetUsageAndQuotaRequest(
  /**
   * Security origin.
   */
  public val origin: String
)

/**
 * Response type for the [StorageDomain.getUsageAndQuota] command.
 */
@Serializable
public data class GetUsageAndQuotaResponse(
  /**
   * Storage usage (bytes).
   */
  public val usage: Double,
  /**
   * Storage quota (bytes).
   */
  public val quota: Double,
  /**
   * Whether or not the origin has an active storage quota override
   */
  public val overrideActive: Boolean,
  /**
   * Storage usage per type (bytes).
   */
  public val usageBreakdown: List<UsageForType>
)

/**
 * Request object containing input parameters for the [StorageDomain.overrideQuotaForOrigin]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class OverrideQuotaForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String,
  /**
   * The quota size (in bytes) to override the original quota with.
   * If this is called multiple times, the overriden quota will be equal to
   * the quotaSize provided in the final call. If this is called without
   * specifying a quotaSize, the quota will be reset to the default value for
   * the specified origin. If this is called multiple times with different
   * origins, the override will be maintained for each origin until it is
   * disabled (called without a quotaSize).
   */
  public val quotaSize: Double? = null
)

/**
 * Request object containing input parameters for the [StorageDomain.trackCacheStorageForOrigin]
 * command.
 */
@Serializable
public data class TrackCacheStorageForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String
)

/**
 * Request object containing input parameters for the [StorageDomain.trackIndexedDBForOrigin]
 * command.
 */
@Serializable
public data class TrackIndexedDBForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String
)

/**
 * Request object containing input parameters for the [StorageDomain.untrackCacheStorageForOrigin]
 * command.
 */
@Serializable
public data class UntrackCacheStorageForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String
)

/**
 * Request object containing input parameters for the [StorageDomain.untrackIndexedDBForOrigin]
 * command.
 */
@Serializable
public data class UntrackIndexedDBForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String
)

@ExperimentalChromeApi
public class StorageDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out StorageEvent>> =
      mapOf(
      "Storage.cacheStorageContentUpdated" to
          serializer<StorageEvent.CacheStorageContentUpdatedEvent>(),
      "Storage.cacheStorageListUpdated" to serializer<StorageEvent.CacheStorageListUpdatedEvent>(),
      "Storage.indexedDBContentUpdated" to serializer<StorageEvent.IndexedDBContentUpdatedEvent>(),
      "Storage.indexedDBListUpdated" to serializer<StorageEvent.IndexedDBListUpdatedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<StorageEvent> = session.events(deserializersByEventName)

  /**
   * A cache's contents have been modified.
   */
  public fun cacheStorageContentUpdated(): Flow<StorageEvent.CacheStorageContentUpdatedEvent> =
      session.events("Storage.cacheStorageContentUpdated")

  /**
   * A cache has been added/deleted.
   */
  public fun cacheStorageListUpdated(): Flow<StorageEvent.CacheStorageListUpdatedEvent> =
      session.events("Storage.cacheStorageListUpdated")

  /**
   * The origin's IndexedDB object store has been modified.
   */
  public fun indexedDBContentUpdated(): Flow<StorageEvent.IndexedDBContentUpdatedEvent> =
      session.events("Storage.indexedDBContentUpdated")

  /**
   * The origin's IndexedDB database list has been modified.
   */
  public fun indexedDBListUpdated(): Flow<StorageEvent.IndexedDBListUpdatedEvent> =
      session.events("Storage.indexedDBListUpdated")

  /**
   * Clears storage for origin.
   */
  public suspend fun clearDataForOrigin(input: ClearDataForOriginRequest): Unit =
      session.request("Storage.clearDataForOrigin", input)

  /**
   * Returns all browser cookies.
   */
  public suspend fun getCookies(input: GetCookiesRequest): GetCookiesResponse =
      session.request("Storage.getCookies", input)

  /**
   * Sets given cookies.
   */
  public suspend fun setCookies(input: SetCookiesRequest): Unit =
      session.request("Storage.setCookies", input)

  /**
   * Clears cookies.
   */
  public suspend fun clearCookies(input: ClearCookiesRequest): Unit =
      session.request("Storage.clearCookies", input)

  /**
   * Returns usage and quota in bytes.
   */
  public suspend fun getUsageAndQuota(input: GetUsageAndQuotaRequest): GetUsageAndQuotaResponse =
      session.request("Storage.getUsageAndQuota", input)

  /**
   * Override quota for the specified origin
   */
  @ExperimentalChromeApi
  public suspend fun overrideQuotaForOrigin(input: OverrideQuotaForOriginRequest): Unit =
      session.request("Storage.overrideQuotaForOrigin", input)

  /**
   * Registers origin to be notified when an update occurs to its cache storage list.
   */
  public suspend fun trackCacheStorageForOrigin(input: TrackCacheStorageForOriginRequest): Unit =
      session.request("Storage.trackCacheStorageForOrigin", input)

  /**
   * Registers origin to be notified when an update occurs to its IndexedDB.
   */
  public suspend fun trackIndexedDBForOrigin(input: TrackIndexedDBForOriginRequest): Unit =
      session.request("Storage.trackIndexedDBForOrigin", input)

  /**
   * Unregisters origin from receiving notifications for cache storage.
   */
  public suspend fun untrackCacheStorageForOrigin(input: UntrackCacheStorageForOriginRequest): Unit
      = session.request("Storage.untrackCacheStorageForOrigin", input)

  /**
   * Unregisters origin from receiving notifications for IndexedDB.
   */
  public suspend fun untrackIndexedDBForOrigin(input: UntrackIndexedDBForOriginRequest): Unit =
      session.request("Storage.untrackIndexedDBForOrigin", input)
}
