package org.hildan.chrome.devtools.domains.backgroundservice

import kotlin.Boolean
import kotlin.String
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.backgroundservice.events.BackgroundServiceEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [BackgroundServiceDomain.startObserving]
 * command.
 */
@Serializable
public data class StartObservingRequest(
  public val service: ServiceName
)

/**
 * Request object containing input parameters for the [BackgroundServiceDomain.stopObserving]
 * command.
 */
@Serializable
public data class StopObservingRequest(
  public val service: ServiceName
)

/**
 * Request object containing input parameters for the [BackgroundServiceDomain.setRecording]
 * command.
 */
@Serializable
public data class SetRecordingRequest(
  public val shouldRecord: Boolean,
  public val service: ServiceName
)

/**
 * Request object containing input parameters for the [BackgroundServiceDomain.clearEvents] command.
 */
@Serializable
public data class ClearEventsRequest(
  public val service: ServiceName
)

/**
 * Defines events for background web platform features.
 */
@ExperimentalChromeApi
public class BackgroundServiceDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out
      BackgroundServiceEvent>> = mapOf(
      "BackgroundService.recordingStateChanged" to
          serializer<BackgroundServiceEvent.RecordingStateChangedEvent>(),
      "BackgroundService.backgroundServiceEventReceived" to
          serializer<BackgroundServiceEvent.BackgroundServiceEventReceivedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<BackgroundServiceEvent> = session.events(deserializersByEventName)

  /**
   * Called when the recording state for the service has been updated.
   */
  public fun recordingStateChanged(): Flow<BackgroundServiceEvent.RecordingStateChangedEvent> =
      session.events("BackgroundService.recordingStateChanged")

  /**
   * Called with all existing backgroundServiceEvents when enabled, and all new
   * events afterwards if enabled and recording.
   */
  public fun backgroundServiceEventReceived():
      Flow<BackgroundServiceEvent.BackgroundServiceEventReceivedEvent> =
      session.events("BackgroundService.backgroundServiceEventReceived")

  /**
   * Enables event updates for the service.
   */
  public suspend fun startObserving(input: StartObservingRequest): Unit =
      session.request("BackgroundService.startObserving", input)

  /**
   * Disables event updates for the service.
   */
  public suspend fun stopObserving(input: StopObservingRequest): Unit =
      session.request("BackgroundService.stopObserving", input)

  /**
   * Set the recording state for the service.
   */
  public suspend fun setRecording(input: SetRecordingRequest): Unit =
      session.request("BackgroundService.setRecording", input)

  /**
   * Clears all stored data for the service.
   */
  public suspend fun clearEvents(input: ClearEventsRequest): Unit =
      session.request("BackgroundService.clearEvents", input)
}
