package org.hildan.chrome.devtools.domains.debugger

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.debugger.events.DebuggerEvent
import org.hildan.chrome.devtools.domains.runtime.CallArgument
import org.hildan.chrome.devtools.domains.runtime.ExceptionDetails
import org.hildan.chrome.devtools.domains.runtime.RemoteObject
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.domains.runtime.ScriptId
import org.hildan.chrome.devtools.domains.runtime.StackTrace
import org.hildan.chrome.devtools.domains.runtime.StackTraceId
import org.hildan.chrome.devtools.domains.runtime.TimeDelta
import org.hildan.chrome.devtools.domains.runtime.UniqueDebuggerId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [DebuggerDomain.continueToLocation] command.
 */
@Serializable
public data class ContinueToLocationRequest(
  /**
   * Location to continue to.
   */
  public val location: Location,
  public val targetCallFrames: String? = null
)

/**
 * Request object containing input parameters for the [DebuggerDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * The maximum size in bytes of collected scripts (not referenced by other heap objects)
   * the debugger can hold. Puts no limit if paramter is omitted.
   */
  @ExperimentalChromeApi
  public val maxScriptsCacheSize: Double? = null
)

/**
 * Response type for the [DebuggerDomain.enable] command.
 */
@Serializable
public data class EnableResponse(
  /**
   * Unique identifier of the debugger.
   */
  @ExperimentalChromeApi
  public val debuggerId: UniqueDebuggerId? = null
)

/**
 * Request object containing input parameters for the [DebuggerDomain.evaluateOnCallFrame] command.
 */
@Serializable
public data class EvaluateOnCallFrameRequest(
  /**
   * Call frame identifier to evaluate on.
   */
  public val callFrameId: CallFrameId,
  /**
   * Expression to evaluate.
   */
  public val expression: String,
  /**
   * String object group name to put result into (allows rapid releasing resulting object handles
   * using `releaseObjectGroup`).
   */
  public val objectGroup: String? = null,
  /**
   * Specifies whether command line API should be available to the evaluated expression, defaults
   * to false.
   */
  public val includeCommandLineAPI: Boolean? = null,
  /**
   * In silent mode exceptions thrown during evaluation are not reported and do not pause
   * execution. Overrides `setPauseOnException` state.
   */
  public val silent: Boolean? = null,
  /**
   * Whether the result is expected to be a JSON object that should be sent by value.
   */
  public val returnByValue: Boolean? = null,
  /**
   * Whether preview should be generated for the result.
   */
  @ExperimentalChromeApi
  public val generatePreview: Boolean? = null,
  /**
   * Whether to throw an exception if side effect cannot be ruled out during evaluation.
   */
  public val throwOnSideEffect: Boolean? = null,
  /**
   * Terminate execution after timing out (number of milliseconds).
   */
  @ExperimentalChromeApi
  public val timeout: TimeDelta? = null
)

/**
 * Response type for the [DebuggerDomain.evaluateOnCallFrame] command.
 */
@Serializable
public data class EvaluateOnCallFrameResponse(
  /**
   * Object wrapper for the evaluation result.
   */
  public val result: RemoteObject,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null
)

/**
 * Request object containing input parameters for the [DebuggerDomain.executeWasmEvaluator] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ExecuteWasmEvaluatorRequest(
  /**
   * WebAssembly call frame identifier to evaluate on.
   */
  public val callFrameId: CallFrameId,
  /**
   * Code of the evaluator module.
   */
  public val evaluator: String,
  /**
   * Terminate execution after timing out (number of milliseconds).
   */
  @ExperimentalChromeApi
  public val timeout: TimeDelta? = null
)

/**
 * Response type for the [DebuggerDomain.executeWasmEvaluator] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ExecuteWasmEvaluatorResponse(
  /**
   * Object wrapper for the evaluation result.
   */
  public val result: RemoteObject,
  /**
   * Exception details.
   */
  public val exceptionDetails: ExceptionDetails? = null
)

/**
 * Request object containing input parameters for the [DebuggerDomain.getPossibleBreakpoints]
 * command.
 */
@Serializable
public data class GetPossibleBreakpointsRequest(
  /**
   * Start of range to search possible breakpoint locations in.
   */
  public val start: Location,
  /**
   * End of range to search possible breakpoint locations in (excluding). When not specified, end
   * of scripts is used as end of range.
   */
  public val end: Location? = null,
  /**
   * Only consider locations which are in the same (non-nested) function as start.
   */
  public val restrictToFunction: Boolean? = null
)

/**
 * Response type for the [DebuggerDomain.getPossibleBreakpoints] command.
 */
@Serializable
public data class GetPossibleBreakpointsResponse(
  /**
   * List of the possible breakpoint locations.
   */
  public val locations: List<BreakLocation>
)

/**
 * Request object containing input parameters for the [DebuggerDomain.getScriptSource] command.
 */
@Serializable
public data class GetScriptSourceRequest(
  /**
   * Id of the script to get source for.
   */
  public val scriptId: ScriptId
)

/**
 * Response type for the [DebuggerDomain.getScriptSource] command.
 */
@Serializable
public data class GetScriptSourceResponse(
  /**
   * Script source (empty in case of Wasm bytecode).
   */
  public val scriptSource: String,
  /**
   * Wasm bytecode.
   */
  public val bytecode: String? = null
)

/**
 * Request object containing input parameters for the [DebuggerDomain.getWasmBytecode] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class GetWasmBytecodeRequest(
  /**
   * Id of the Wasm script to get source for.
   */
  public val scriptId: ScriptId
)

/**
 * Response type for the [DebuggerDomain.getWasmBytecode] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class GetWasmBytecodeResponse(
  /**
   * Script source.
   */
  public val bytecode: String
)

/**
 * Request object containing input parameters for the [DebuggerDomain.getStackTrace] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetStackTraceRequest(
  public val stackTraceId: StackTraceId
)

/**
 * Response type for the [DebuggerDomain.getStackTrace] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetStackTraceResponse(
  public val stackTrace: StackTrace
)

/**
 * Request object containing input parameters for the [DebuggerDomain.pauseOnAsyncCall] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@ExperimentalChromeApi
public data class PauseOnAsyncCallRequest(
  /**
   * Debugger will pause when async call with given stack trace is started.
   */
  public val parentStackTraceId: StackTraceId
)

/**
 * Request object containing input parameters for the [DebuggerDomain.removeBreakpoint] command.
 */
@Serializable
public data class RemoveBreakpointRequest(
  public val breakpointId: BreakpointId
)

/**
 * Request object containing input parameters for the [DebuggerDomain.restartFrame] command.
 */
@Serializable
public data class RestartFrameRequest(
  /**
   * Call frame identifier to evaluate on.
   */
  public val callFrameId: CallFrameId
)

/**
 * Response type for the [DebuggerDomain.restartFrame] command.
 */
@Serializable
public data class RestartFrameResponse(
  /**
   * New stack trace.
   */
  public val callFrames: List<CallFrame>,
  /**
   * Async stack trace, if any.
   */
  public val asyncStackTrace: StackTrace? = null,
  /**
   * Async stack trace, if any.
   */
  @ExperimentalChromeApi
  public val asyncStackTraceId: StackTraceId? = null
)

/**
 * Request object containing input parameters for the [DebuggerDomain.resume] command.
 */
@Serializable
public data class ResumeRequest(
  /**
   * Set to true to terminate execution upon resuming execution. In contrast
   * to Runtime.terminateExecution, this will allows to execute further
   * JavaScript (i.e. via evaluation) until execution of the paused code
   * is actually resumed, at which point termination is triggered.
   * If execution is currently not paused, this parameter has no effect.
   */
  public val terminateOnResume: Boolean? = null
)

/**
 * Request object containing input parameters for the [DebuggerDomain.searchInContent] command.
 */
@Serializable
public data class SearchInContentRequest(
  /**
   * Id of the script to search in.
   */
  public val scriptId: ScriptId,
  /**
   * String to search for.
   */
  public val query: String,
  /**
   * If true, search is case sensitive.
   */
  public val caseSensitive: Boolean? = null,
  /**
   * If true, treats string parameter as regex.
   */
  public val isRegex: Boolean? = null
)

/**
 * Response type for the [DebuggerDomain.searchInContent] command.
 */
@Serializable
public data class SearchInContentResponse(
  /**
   * List of search matches.
   */
  public val result: List<SearchMatch>
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setAsyncCallStackDepth]
 * command.
 */
@Serializable
public data class SetAsyncCallStackDepthRequest(
  /**
   * Maximum depth of async call stacks. Setting to `0` will effectively disable collecting async
   * call stacks (default).
   */
  public val maxDepth: Int
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setBlackboxPatterns] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBlackboxPatternsRequest(
  /**
   * Array of regexps that will be used to check script url for blackbox state.
   */
  public val patterns: List<String>
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setBlackboxedRanges] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBlackboxedRangesRequest(
  /**
   * Id of the script.
   */
  public val scriptId: ScriptId,
  public val positions: List<ScriptPosition>
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setBreakpoint] command.
 */
@Serializable
public data class SetBreakpointRequest(
  /**
   * Location to set breakpoint in.
   */
  public val location: Location,
  /**
   * Expression to use as a breakpoint condition. When specified, debugger will only stop on the
   * breakpoint if this expression evaluates to true.
   */
  public val condition: String? = null
)

/**
 * Response type for the [DebuggerDomain.setBreakpoint] command.
 */
@Serializable
public data class SetBreakpointResponse(
  /**
   * Id of the created breakpoint for further reference.
   */
  public val breakpointId: BreakpointId,
  /**
   * Location this breakpoint resolved into.
   */
  public val actualLocation: Location
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setInstrumentationBreakpoint]
 * command.
 */
@Serializable
public data class SetInstrumentationBreakpointRequest(
  /**
   * Instrumentation name.
   */
  public val instrumentation: String
)

/**
 * Response type for the [DebuggerDomain.setInstrumentationBreakpoint] command.
 */
@Serializable
public data class SetInstrumentationBreakpointResponse(
  /**
   * Id of the created breakpoint for further reference.
   */
  public val breakpointId: BreakpointId
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setBreakpointByUrl] command.
 */
@Serializable
public data class SetBreakpointByUrlRequest(
  /**
   * Line number to set breakpoint at.
   */
  public val lineNumber: Int,
  /**
   * URL of the resources to set breakpoint on.
   */
  public val url: String? = null,
  /**
   * Regex pattern for the URLs of the resources to set breakpoints on. Either `url` or
   * `urlRegex` must be specified.
   */
  public val urlRegex: String? = null,
  /**
   * Script hash of the resources to set breakpoint on.
   */
  public val scriptHash: String? = null,
  /**
   * Offset in the line to set breakpoint at.
   */
  public val columnNumber: Int? = null,
  /**
   * Expression to use as a breakpoint condition. When specified, debugger will only stop on the
   * breakpoint if this expression evaluates to true.
   */
  public val condition: String? = null
)

/**
 * Response type for the [DebuggerDomain.setBreakpointByUrl] command.
 */
@Serializable
public data class SetBreakpointByUrlResponse(
  /**
   * Id of the created breakpoint for further reference.
   */
  public val breakpointId: BreakpointId,
  /**
   * List of the locations this breakpoint resolved into upon addition.
   */
  public val locations: List<Location>
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setBreakpointOnFunctionCall]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBreakpointOnFunctionCallRequest(
  /**
   * Function object id.
   */
  public val objectId: RemoteObjectId,
  /**
   * Expression to use as a breakpoint condition. When specified, debugger will
   * stop on the breakpoint if this expression evaluates to true.
   */
  public val condition: String? = null
)

/**
 * Response type for the [DebuggerDomain.setBreakpointOnFunctionCall] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetBreakpointOnFunctionCallResponse(
  /**
   * Id of the created breakpoint for further reference.
   */
  public val breakpointId: BreakpointId
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setBreakpointsActive] command.
 */
@Serializable
public data class SetBreakpointsActiveRequest(
  /**
   * New value for breakpoints active state.
   */
  public val active: Boolean
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setPauseOnExceptions] command.
 */
@Serializable
public data class SetPauseOnExceptionsRequest(
  /**
   * Pause on exceptions mode.
   */
  public val state: String
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setReturnValue] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetReturnValueRequest(
  /**
   * New return value.
   */
  public val newValue: CallArgument
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setScriptSource] command.
 */
@Serializable
public data class SetScriptSourceRequest(
  /**
   * Id of the script to edit.
   */
  public val scriptId: ScriptId,
  /**
   * New content of the script.
   */
  public val scriptSource: String,
  /**
   * If true the change will not actually be applied. Dry run may be used to get result
   * description without actually modifying the code.
   */
  public val dryRun: Boolean? = null
)

/**
 * Response type for the [DebuggerDomain.setScriptSource] command.
 */
@Serializable
public data class SetScriptSourceResponse(
  /**
   * New stack trace in case editing has happened while VM was stopped.
   */
  public val callFrames: List<CallFrame>? = null,
  /**
   * Whether current call stack  was modified after applying the changes.
   */
  public val stackChanged: Boolean? = null,
  /**
   * Async stack trace, if any.
   */
  public val asyncStackTrace: StackTrace? = null,
  /**
   * Async stack trace, if any.
   */
  @ExperimentalChromeApi
  public val asyncStackTraceId: StackTraceId? = null,
  /**
   * Exception details if any.
   */
  public val exceptionDetails: ExceptionDetails? = null
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setSkipAllPauses] command.
 */
@Serializable
public data class SetSkipAllPausesRequest(
  /**
   * New value for skip pauses state.
   */
  public val skip: Boolean
)

/**
 * Request object containing input parameters for the [DebuggerDomain.setVariableValue] command.
 */
@Serializable
public data class SetVariableValueRequest(
  /**
   * 0-based number of scope as was listed in scope chain. Only 'local', 'closure' and 'catch'
   * scope types are allowed. Other scopes could be manipulated manually.
   */
  public val scopeNumber: Int,
  /**
   * Variable name.
   */
  public val variableName: String,
  /**
   * New variable value.
   */
  public val newValue: CallArgument,
  /**
   * Id of callframe that holds variable.
   */
  public val callFrameId: CallFrameId
)

/**
 * Request object containing input parameters for the [DebuggerDomain.stepInto] command.
 */
@Serializable
public data class StepIntoRequest(
  /**
   * Debugger will pause on the execution of the first async task which was scheduled
   * before next pause.
   */
  @ExperimentalChromeApi
  public val breakOnAsyncCall: Boolean? = null,
  /**
   * The skipList specifies location ranges that should be skipped on step into.
   */
  @ExperimentalChromeApi
  public val skipList: List<LocationRange>? = null
)

/**
 * Request object containing input parameters for the [DebuggerDomain.stepOver] command.
 */
@Serializable
public data class StepOverRequest(
  /**
   * The skipList specifies location ranges that should be skipped on step over.
   */
  @ExperimentalChromeApi
  public val skipList: List<LocationRange>? = null
)

/**
 * Debugger domain exposes JavaScript debugging capabilities. It allows setting and removing
 * breakpoints, stepping through execution, exploring stack traces, etc.
 */
public class DebuggerDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out DebuggerEvent>> =
      mapOf(
      "Debugger.breakpointResolved" to serializer<DebuggerEvent.BreakpointResolvedEvent>(),
      "Debugger.paused" to serializer<DebuggerEvent.PausedEvent>(),
      "Debugger.resumed" to serializer<DebuggerEvent.ResumedEvent>(),
      "Debugger.scriptFailedToParse" to serializer<DebuggerEvent.ScriptFailedToParseEvent>(),
      "Debugger.scriptParsed" to serializer<DebuggerEvent.ScriptParsedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<DebuggerEvent> = session.events(deserializersByEventName)

  /**
   * Fired when breakpoint is resolved to an actual script and location.
   */
  public fun breakpointResolved(): Flow<DebuggerEvent.BreakpointResolvedEvent> =
      session.events("Debugger.breakpointResolved")

  /**
   * Fired when the virtual machine stopped on breakpoint or exception or any other stop criteria.
   */
  public fun paused(): Flow<DebuggerEvent.PausedEvent> = session.events("Debugger.paused")

  /**
   * Fired when the virtual machine resumed execution.
   */
  public fun resumed(): Flow<DebuggerEvent.ResumedEvent> = session.events("Debugger.resumed")

  /**
   * Fired when virtual machine fails to parse the script.
   */
  public fun scriptFailedToParse(): Flow<DebuggerEvent.ScriptFailedToParseEvent> =
      session.events("Debugger.scriptFailedToParse")

  /**
   * Fired when virtual machine parses script. This event is also fired for all known and
   * uncollected
   * scripts upon enabling debugger.
   */
  public fun scriptParsed(): Flow<DebuggerEvent.ScriptParsedEvent> =
      session.events("Debugger.scriptParsed")

  /**
   * Continues execution until specific location is reached.
   */
  public suspend fun continueToLocation(input: ContinueToLocationRequest): Unit =
      session.request("Debugger.continueToLocation", input)

  /**
   * Disables debugger for given page.
   */
  public suspend fun disable(): Unit = session.request("Debugger.disable", Unit)

  /**
   * Enables debugger for the given page. Clients should not assume that the debugging has been
   * enabled until the result for this command is received.
   */
  public suspend fun enable(input: EnableRequest): EnableResponse =
      session.request("Debugger.enable", input)

  /**
   * Evaluates expression on a given call frame.
   */
  public suspend fun evaluateOnCallFrame(input: EvaluateOnCallFrameRequest):
      EvaluateOnCallFrameResponse = session.request("Debugger.evaluateOnCallFrame", input)

  /**
   * Execute a Wasm Evaluator module on a given call frame.
   */
  @ExperimentalChromeApi
  public suspend fun executeWasmEvaluator(input: ExecuteWasmEvaluatorRequest):
      ExecuteWasmEvaluatorResponse = session.request("Debugger.executeWasmEvaluator", input)

  /**
   * Returns possible locations for breakpoint. scriptId in start and end range locations should be
   * the same.
   */
  public suspend fun getPossibleBreakpoints(input: GetPossibleBreakpointsRequest):
      GetPossibleBreakpointsResponse = session.request("Debugger.getPossibleBreakpoints", input)

  /**
   * Returns source for the script with given id.
   */
  public suspend fun getScriptSource(input: GetScriptSourceRequest): GetScriptSourceResponse =
      session.request("Debugger.getScriptSource", input)

  /**
   * This command is deprecated. Use getScriptSource instead.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun getWasmBytecode(input: GetWasmBytecodeRequest): GetWasmBytecodeResponse =
      session.request("Debugger.getWasmBytecode", input)

  /**
   * Returns stack trace with given `stackTraceId`.
   */
  @ExperimentalChromeApi
  public suspend fun getStackTrace(input: GetStackTraceRequest): GetStackTraceResponse =
      session.request("Debugger.getStackTrace", input)

  /**
   * Stops on the next JavaScript statement.
   */
  public suspend fun pause(): Unit = session.request("Debugger.pause", Unit)

  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @ExperimentalChromeApi
  public suspend fun pauseOnAsyncCall(input: PauseOnAsyncCallRequest): Unit =
      session.request("Debugger.pauseOnAsyncCall", input)

  /**
   * Removes JavaScript breakpoint.
   */
  public suspend fun removeBreakpoint(input: RemoveBreakpointRequest): Unit =
      session.request("Debugger.removeBreakpoint", input)

  /**
   * Restarts particular call frame from the beginning.
   */
  public suspend fun restartFrame(input: RestartFrameRequest): RestartFrameResponse =
      session.request("Debugger.restartFrame", input)

  /**
   * Resumes JavaScript execution.
   */
  public suspend fun resume(input: ResumeRequest): Unit = session.request("Debugger.resume", input)

  /**
   * Searches for given string in script content.
   */
  public suspend fun searchInContent(input: SearchInContentRequest): SearchInContentResponse =
      session.request("Debugger.searchInContent", input)

  /**
   * Enables or disables async call stacks tracking.
   */
  public suspend fun setAsyncCallStackDepth(input: SetAsyncCallStackDepthRequest): Unit =
      session.request("Debugger.setAsyncCallStackDepth", input)

  /**
   * Replace previous blackbox patterns with passed ones. Forces backend to skip stepping/pausing in
   * scripts with url matching one of the patterns. VM will try to leave blackboxed script by
   * performing 'step in' several times, finally resorting to 'step out' if unsuccessful.
   */
  @ExperimentalChromeApi
  public suspend fun setBlackboxPatterns(input: SetBlackboxPatternsRequest): Unit =
      session.request("Debugger.setBlackboxPatterns", input)

  /**
   * Makes backend skip steps in the script in blackboxed ranges. VM will try leave blacklisted
   * scripts by performing 'step in' several times, finally resorting to 'step out' if unsuccessful.
   * Positions array contains positions where blackbox state is changed. First interval isn't
   * blackboxed. Array should be sorted.
   */
  @ExperimentalChromeApi
  public suspend fun setBlackboxedRanges(input: SetBlackboxedRangesRequest): Unit =
      session.request("Debugger.setBlackboxedRanges", input)

  /**
   * Sets JavaScript breakpoint at a given location.
   */
  public suspend fun setBreakpoint(input: SetBreakpointRequest): SetBreakpointResponse =
      session.request("Debugger.setBreakpoint", input)

  /**
   * Sets instrumentation breakpoint.
   */
  public suspend fun setInstrumentationBreakpoint(input: SetInstrumentationBreakpointRequest):
      SetInstrumentationBreakpointResponse =
      session.request("Debugger.setInstrumentationBreakpoint", input)

  /**
   * Sets JavaScript breakpoint at given location specified either by URL or URL regex. Once this
   * command is issued, all existing parsed scripts will have breakpoints resolved and returned in
   * `locations` property. Further matching script parsing will result in subsequent
   * `breakpointResolved` events issued. This logical breakpoint will survive page reloads.
   */
  public suspend fun setBreakpointByUrl(input: SetBreakpointByUrlRequest):
      SetBreakpointByUrlResponse = session.request("Debugger.setBreakpointByUrl", input)

  /**
   * Sets JavaScript breakpoint before each call to the given function.
   * If another function was created from the same source as a given one,
   * calling it will also trigger the breakpoint.
   */
  @ExperimentalChromeApi
  public suspend fun setBreakpointOnFunctionCall(input: SetBreakpointOnFunctionCallRequest):
      SetBreakpointOnFunctionCallResponse = session.request("Debugger.setBreakpointOnFunctionCall",
      input)

  /**
   * Activates / deactivates all breakpoints on the page.
   */
  public suspend fun setBreakpointsActive(input: SetBreakpointsActiveRequest): Unit =
      session.request("Debugger.setBreakpointsActive", input)

  /**
   * Defines pause on exceptions state. Can be set to stop on all exceptions, uncaught exceptions or
   * no exceptions. Initial pause on exceptions state is `none`.
   */
  public suspend fun setPauseOnExceptions(input: SetPauseOnExceptionsRequest): Unit =
      session.request("Debugger.setPauseOnExceptions", input)

  /**
   * Changes return value in top frame. Available only at return break position.
   */
  @ExperimentalChromeApi
  public suspend fun setReturnValue(input: SetReturnValueRequest): Unit =
      session.request("Debugger.setReturnValue", input)

  /**
   * Edits JavaScript source live.
   */
  public suspend fun setScriptSource(input: SetScriptSourceRequest): SetScriptSourceResponse =
      session.request("Debugger.setScriptSource", input)

  /**
   * Makes page not interrupt on any pauses (breakpoint, exception, dom exception etc).
   */
  public suspend fun setSkipAllPauses(input: SetSkipAllPausesRequest): Unit =
      session.request("Debugger.setSkipAllPauses", input)

  /**
   * Changes value of variable in a callframe. Object-based scopes are not supported and must be
   * mutated manually.
   */
  public suspend fun setVariableValue(input: SetVariableValueRequest): Unit =
      session.request("Debugger.setVariableValue", input)

  /**
   * Steps into the function call.
   */
  public suspend fun stepInto(input: StepIntoRequest): Unit = session.request("Debugger.stepInto",
      input)

  /**
   * Steps out of the function call.
   */
  public suspend fun stepOut(): Unit = session.request("Debugger.stepOut", Unit)

  /**
   * Steps over the statement.
   */
  public suspend fun stepOver(input: StepOverRequest): Unit = session.request("Debugger.stepOver",
      input)
}
