package org.hildan.chrome.devtools.domains.debugger

import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.runtime.RemoteObject
import org.hildan.chrome.devtools.domains.runtime.ScriptId
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Breakpoint identifier.
 */
public typealias BreakpointId = String

/**
 * Call frame identifier.
 */
public typealias CallFrameId = String

/**
 * Location in the source code.
 */
@Serializable
public data class Location(
  /**
   * Script identifier as reported in the `Debugger.scriptParsed`.
   */
  public val scriptId: ScriptId,
  /**
   * Line number in the script (0-based).
   */
  public val lineNumber: Int,
  /**
   * Column number in the script (0-based).
   */
  public val columnNumber: Int? = null
)

/**
 * Location in the source code.
 */
@ExperimentalChromeApi
@Serializable
public data class ScriptPosition(
  public val lineNumber: Int,
  public val columnNumber: Int
)

/**
 * Location range within one script.
 */
@ExperimentalChromeApi
@Serializable
public data class LocationRange(
  public val scriptId: ScriptId,
  public val start: ScriptPosition,
  public val end: ScriptPosition
)

/**
 * JavaScript call frame. Array of call frames form the call stack.
 */
@Serializable
public data class CallFrame(
  /**
   * Call frame identifier. This identifier is only valid while the virtual machine is paused.
   */
  public val callFrameId: CallFrameId,
  /**
   * Name of the JavaScript function called on this call frame.
   */
  public val functionName: String,
  /**
   * Location in the source code.
   */
  public val functionLocation: Location? = null,
  /**
   * Location in the source code.
   */
  public val location: Location,
  /**
   * JavaScript script name or url.
   */
  public val url: String,
  /**
   * Scope chain for this call frame.
   */
  public val scopeChain: List<Scope>,
  /**
   * `this` object for this call frame.
   */
  public val `this`: RemoteObject,
  /**
   * The value being returned, if the function is at return point.
   */
  public val returnValue: RemoteObject? = null
)

/**
 * Scope description.
 */
@Serializable
public data class Scope(
  /**
   * Scope type.
   */
  public val type: String,
  /**
   * Object representing the scope. For `global` and `with` scopes it represents the actual
   * object; for the rest of the scopes, it is artificial transient object enumerating scope
   * variables as its properties.
   */
  public val `object`: RemoteObject,
  public val name: String? = null,
  /**
   * Location in the source code where scope starts
   */
  public val startLocation: Location? = null,
  /**
   * Location in the source code where scope ends
   */
  public val endLocation: Location? = null
)

/**
 * Search match for resource.
 */
@Serializable
public data class SearchMatch(
  /**
   * Line number in resource content.
   */
  public val lineNumber: Double,
  /**
   * Line with match content.
   */
  public val lineContent: String
)

@Serializable
public data class BreakLocation(
  /**
   * Script identifier as reported in the `Debugger.scriptParsed`.
   */
  public val scriptId: ScriptId,
  /**
   * Line number in the script (0-based).
   */
  public val lineNumber: Int,
  /**
   * Column number in the script (0-based).
   */
  public val columnNumber: Int? = null,
  public val type: String? = null
)

/**
 * Enum of possible script languages.
 */
@Serializable
public enum class ScriptLanguage {
  JavaScript,
  WebAssembly,
}

/**
 * Debug symbols available for a wasm script.
 */
@Serializable
public data class DebugSymbols(
  /**
   * Type of the debug symbols.
   */
  public val type: String,
  /**
   * URL of the external symbol source.
   */
  public val externalURL: String? = null
)
