package org.hildan.chrome.devtools.domains.debugger.events

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import org.hildan.chrome.devtools.domains.debugger.BreakpointId
import org.hildan.chrome.devtools.domains.debugger.CallFrame
import org.hildan.chrome.devtools.domains.debugger.DebugSymbols
import org.hildan.chrome.devtools.domains.debugger.Location
import org.hildan.chrome.devtools.domains.debugger.ScriptLanguage
import org.hildan.chrome.devtools.domains.runtime.ExecutionContextId
import org.hildan.chrome.devtools.domains.runtime.ScriptId
import org.hildan.chrome.devtools.domains.runtime.StackTrace
import org.hildan.chrome.devtools.domains.runtime.StackTraceId
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

@Serializable
public sealed class DebuggerEvent {
  @Serializable
  public data class BreakpointResolvedEvent(
    /**
     * Breakpoint unique identifier.
     */
    public val breakpointId: BreakpointId,
    /**
     * Actual breakpoint location.
     */
    public val location: Location
  ) : DebuggerEvent()

  @Serializable
  public data class PausedEvent(
    /**
     * Call stack the virtual machine stopped on.
     */
    public val callFrames: List<CallFrame>,
    /**
     * Pause reason.
     */
    public val reason: String,
    /**
     * Object containing break-specific auxiliary properties.
     */
    public val `data`: JsonElement? = null,
    /**
     * Hit breakpoints IDs
     */
    public val hitBreakpoints: List<String>? = null,
    /**
     * Async stack trace, if any.
     */
    public val asyncStackTrace: StackTrace? = null,
    /**
     * Async stack trace, if any.
     */
    @ExperimentalChromeApi
    public val asyncStackTraceId: StackTraceId? = null,
    /**
     * Never present, will be removed.
     */
    @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
    @ExperimentalChromeApi
    public val asyncCallStackTraceId: StackTraceId? = null
  ) : DebuggerEvent()

  @Serializable
  public object ResumedEvent : DebuggerEvent()

  @Serializable
  public data class ScriptFailedToParseEvent(
    /**
     * Identifier of the script parsed.
     */
    public val scriptId: ScriptId,
    /**
     * URL or name of the script parsed (if any).
     */
    public val url: String,
    /**
     * Line offset of the script within the resource with given URL (for script tags).
     */
    public val startLine: Int,
    /**
     * Column offset of the script within the resource with given URL.
     */
    public val startColumn: Int,
    /**
     * Last line of the script.
     */
    public val endLine: Int,
    /**
     * Length of the last line of the script.
     */
    public val endColumn: Int,
    /**
     * Specifies script creation context.
     */
    public val executionContextId: ExecutionContextId,
    /**
     * Content hash of the script.
     */
    public val hash: String,
    /**
     * Embedder-specific auxiliary data.
     */
    public val executionContextAuxData: JsonElement? = null,
    /**
     * URL of source map associated with script (if any).
     */
    public val sourceMapURL: String? = null,
    /**
     * True, if this script has sourceURL.
     */
    public val hasSourceURL: Boolean? = null,
    /**
     * True, if this script is ES6 module.
     */
    public val isModule: Boolean? = null,
    /**
     * This script length.
     */
    public val length: Int? = null,
    /**
     * JavaScript top stack frame of where the script parsed event was triggered if available.
     */
    @ExperimentalChromeApi
    public val stackTrace: StackTrace? = null,
    /**
     * If the scriptLanguage is WebAssembly, the code section offset in the module.
     */
    @ExperimentalChromeApi
    public val codeOffset: Int? = null,
    /**
     * The language of the script.
     */
    @ExperimentalChromeApi
    public val scriptLanguage: ScriptLanguage? = null,
    /**
     * The name the embedder supplied for this script.
     */
    @ExperimentalChromeApi
    public val embedderName: String? = null
  ) : DebuggerEvent()

  @Serializable
  public data class ScriptParsedEvent(
    /**
     * Identifier of the script parsed.
     */
    public val scriptId: ScriptId,
    /**
     * URL or name of the script parsed (if any).
     */
    public val url: String,
    /**
     * Line offset of the script within the resource with given URL (for script tags).
     */
    public val startLine: Int,
    /**
     * Column offset of the script within the resource with given URL.
     */
    public val startColumn: Int,
    /**
     * Last line of the script.
     */
    public val endLine: Int,
    /**
     * Length of the last line of the script.
     */
    public val endColumn: Int,
    /**
     * Specifies script creation context.
     */
    public val executionContextId: ExecutionContextId,
    /**
     * Content hash of the script.
     */
    public val hash: String,
    /**
     * Embedder-specific auxiliary data.
     */
    public val executionContextAuxData: JsonElement? = null,
    /**
     * True, if this script is generated as a result of the live edit operation.
     */
    @ExperimentalChromeApi
    public val isLiveEdit: Boolean? = null,
    /**
     * URL of source map associated with script (if any).
     */
    public val sourceMapURL: String? = null,
    /**
     * True, if this script has sourceURL.
     */
    public val hasSourceURL: Boolean? = null,
    /**
     * True, if this script is ES6 module.
     */
    public val isModule: Boolean? = null,
    /**
     * This script length.
     */
    public val length: Int? = null,
    /**
     * JavaScript top stack frame of where the script parsed event was triggered if available.
     */
    @ExperimentalChromeApi
    public val stackTrace: StackTrace? = null,
    /**
     * If the scriptLanguage is WebAssembly, the code section offset in the module.
     */
    @ExperimentalChromeApi
    public val codeOffset: Int? = null,
    /**
     * The language of the script.
     */
    @ExperimentalChromeApi
    public val scriptLanguage: ScriptLanguage? = null,
    /**
     * If the scriptLanguage is WebASsembly, the source of debug symbols for the module.
     */
    @ExperimentalChromeApi
    public val debugSymbols: DebugSymbols? = null,
    /**
     * The name the embedder supplied for this script.
     */
    @ExperimentalChromeApi
    public val embedderName: String? = null
  ) : DebuggerEvent()
}
