package org.hildan.chrome.devtools.domains.domstorage

import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.domstorage.events.DOMStorageEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [DOMStorageDomain.clear] command.
 */
@Serializable
public data class ClearRequest(
  public val storageId: StorageId
)

/**
 * Request object containing input parameters for the [DOMStorageDomain.getDOMStorageItems] command.
 */
@Serializable
public data class GetDOMStorageItemsRequest(
  public val storageId: StorageId
)

/**
 * Response type for the [DOMStorageDomain.getDOMStorageItems] command.
 */
@Serializable
public data class GetDOMStorageItemsResponse(
  public val entries: List<Item>
)

/**
 * Request object containing input parameters for the [DOMStorageDomain.removeDOMStorageItem]
 * command.
 */
@Serializable
public data class RemoveDOMStorageItemRequest(
  public val storageId: StorageId,
  public val key: String
)

/**
 * Request object containing input parameters for the [DOMStorageDomain.setDOMStorageItem] command.
 */
@Serializable
public data class SetDOMStorageItemRequest(
  public val storageId: StorageId,
  public val key: String,
  public val value: String
)

/**
 * Query and modify DOM storage.
 */
@ExperimentalChromeApi
public class DOMStorageDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out DOMStorageEvent>> =
      mapOf(
      "DOMStorage.domStorageItemAdded" to serializer<DOMStorageEvent.DomStorageItemAddedEvent>(),
      "DOMStorage.domStorageItemRemoved" to
          serializer<DOMStorageEvent.DomStorageItemRemovedEvent>(),
      "DOMStorage.domStorageItemUpdated" to
          serializer<DOMStorageEvent.DomStorageItemUpdatedEvent>(),
      "DOMStorage.domStorageItemsCleared" to
          serializer<DOMStorageEvent.DomStorageItemsClearedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<DOMStorageEvent> = session.events(deserializersByEventName)

  public fun domStorageItemAdded(): Flow<DOMStorageEvent.DomStorageItemAddedEvent> =
      session.events("DOMStorage.domStorageItemAdded")

  public fun domStorageItemRemoved(): Flow<DOMStorageEvent.DomStorageItemRemovedEvent> =
      session.events("DOMStorage.domStorageItemRemoved")

  public fun domStorageItemUpdated(): Flow<DOMStorageEvent.DomStorageItemUpdatedEvent> =
      session.events("DOMStorage.domStorageItemUpdated")

  public fun domStorageItemsCleared(): Flow<DOMStorageEvent.DomStorageItemsClearedEvent> =
      session.events("DOMStorage.domStorageItemsCleared")

  public suspend fun clear(input: ClearRequest): Unit = session.request("DOMStorage.clear", input)

  /**
   * Disables storage tracking, prevents storage events from being sent to the client.
   */
  public suspend fun disable(): Unit = session.request("DOMStorage.disable", Unit)

  /**
   * Enables storage tracking, storage events will now be delivered to the client.
   */
  public suspend fun enable(): Unit = session.request("DOMStorage.enable", Unit)

  public suspend fun getDOMStorageItems(input: GetDOMStorageItemsRequest):
      GetDOMStorageItemsResponse = session.request("DOMStorage.getDOMStorageItems", input)

  public suspend fun removeDOMStorageItem(input: RemoveDOMStorageItemRequest): Unit =
      session.request("DOMStorage.removeDOMStorageItem", input)

  public suspend fun setDOMStorageItem(input: SetDOMStorageItemRequest): Unit =
      session.request("DOMStorage.setDOMStorageItem", input)
}
