package org.hildan.chrome.devtools.domains.heapprofiler

import kotlin.Boolean
import kotlin.Double
import kotlin.String
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.heapprofiler.events.HeapProfilerEvent
import org.hildan.chrome.devtools.domains.runtime.RemoteObject
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [HeapProfilerDomain.addInspectedHeapObject]
 * command.
 */
@Serializable
public data class AddInspectedHeapObjectRequest(
  /**
   * Heap snapshot object id to be accessible by means of $x command line API.
   */
  public val heapObjectId: HeapSnapshotObjectId
)

/**
 * Request object containing input parameters for the [HeapProfilerDomain.getHeapObjectId] command.
 */
@Serializable
public data class GetHeapObjectIdRequest(
  /**
   * Identifier of the object to get heap object id for.
   */
  public val objectId: RemoteObjectId
)

/**
 * Response type for the [HeapProfilerDomain.getHeapObjectId] command.
 */
@Serializable
public data class GetHeapObjectIdResponse(
  /**
   * Id of the heap snapshot object corresponding to the passed remote object id.
   */
  public val heapSnapshotObjectId: HeapSnapshotObjectId
)

/**
 * Request object containing input parameters for the [HeapProfilerDomain.getObjectByHeapObjectId]
 * command.
 */
@Serializable
public data class GetObjectByHeapObjectIdRequest(
  public val objectId: HeapSnapshotObjectId,
  /**
   * Symbolic group name that can be used to release multiple objects.
   */
  public val objectGroup: String? = null
)

/**
 * Response type for the [HeapProfilerDomain.getObjectByHeapObjectId] command.
 */
@Serializable
public data class GetObjectByHeapObjectIdResponse(
  /**
   * Evaluation result.
   */
  public val result: RemoteObject
)

/**
 * Response type for the [HeapProfilerDomain.getSamplingProfile] command.
 */
@Serializable
public data class GetSamplingProfileResponse(
  /**
   * Return the sampling profile being collected.
   */
  public val profile: SamplingHeapProfile
)

/**
 * Request object containing input parameters for the [HeapProfilerDomain.startSampling] command.
 */
@Serializable
public data class StartSamplingRequest(
  /**
   * Average sample interval in bytes. Poisson distribution is used for the intervals. The
   * default value is 32768 bytes.
   */
  public val samplingInterval: Double? = null
)

/**
 * Request object containing input parameters for the [HeapProfilerDomain.startTrackingHeapObjects]
 * command.
 */
@Serializable
public data class StartTrackingHeapObjectsRequest(
  public val trackAllocations: Boolean? = null
)

/**
 * Response type for the [HeapProfilerDomain.stopSampling] command.
 */
@Serializable
public data class StopSamplingResponse(
  /**
   * Recorded sampling heap profile.
   */
  public val profile: SamplingHeapProfile
)

/**
 * Request object containing input parameters for the [HeapProfilerDomain.stopTrackingHeapObjects]
 * command.
 */
@Serializable
public data class StopTrackingHeapObjectsRequest(
  /**
   * If true 'reportHeapSnapshotProgress' events will be generated while snapshot is being taken
   * when the tracking is stopped.
   */
  public val reportProgress: Boolean? = null,
  public val treatGlobalObjectsAsRoots: Boolean? = null
)

/**
 * Request object containing input parameters for the [HeapProfilerDomain.takeHeapSnapshot] command.
 */
@Serializable
public data class TakeHeapSnapshotRequest(
  /**
   * If true 'reportHeapSnapshotProgress' events will be generated while snapshot is being taken.
   */
  public val reportProgress: Boolean? = null,
  /**
   * If true, a raw snapshot without artifical roots will be generated
   */
  public val treatGlobalObjectsAsRoots: Boolean? = null
)

@ExperimentalChromeApi
public class HeapProfilerDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out HeapProfilerEvent>>
      = mapOf(
      "HeapProfiler.addHeapSnapshotChunk" to
          serializer<HeapProfilerEvent.AddHeapSnapshotChunkEvent>(),
      "HeapProfiler.heapStatsUpdate" to serializer<HeapProfilerEvent.HeapStatsUpdateEvent>(),
      "HeapProfiler.lastSeenObjectId" to serializer<HeapProfilerEvent.LastSeenObjectIdEvent>(),
      "HeapProfiler.reportHeapSnapshotProgress" to
          serializer<HeapProfilerEvent.ReportHeapSnapshotProgressEvent>(),
      "HeapProfiler.resetProfiles" to serializer<HeapProfilerEvent.ResetProfilesEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<HeapProfilerEvent> = session.events(deserializersByEventName)

  public fun addHeapSnapshotChunk(): Flow<HeapProfilerEvent.AddHeapSnapshotChunkEvent> =
      session.events("HeapProfiler.addHeapSnapshotChunk")

  /**
   * If heap objects tracking has been started then backend may send update for one or more
   * fragments
   */
  public fun heapStatsUpdate(): Flow<HeapProfilerEvent.HeapStatsUpdateEvent> =
      session.events("HeapProfiler.heapStatsUpdate")

  /**
   * If heap objects tracking has been started then backend regularly sends a current value for last
   * seen object id and corresponding timestamp. If the were changes in the heap since last event
   * then one or more heapStatsUpdate events will be sent before a new lastSeenObjectId event.
   */
  public fun lastSeenObjectId(): Flow<HeapProfilerEvent.LastSeenObjectIdEvent> =
      session.events("HeapProfiler.lastSeenObjectId")

  public fun reportHeapSnapshotProgress(): Flow<HeapProfilerEvent.ReportHeapSnapshotProgressEvent> =
      session.events("HeapProfiler.reportHeapSnapshotProgress")

  public fun resetProfiles(): Flow<HeapProfilerEvent.ResetProfilesEvent> =
      session.events("HeapProfiler.resetProfiles")

  /**
   * Enables console to refer to the node with given id via $x (see Command Line API for more
   * details
   * $x functions).
   */
  public suspend fun addInspectedHeapObject(input: AddInspectedHeapObjectRequest): Unit =
      session.request("HeapProfiler.addInspectedHeapObject", input)

  public suspend fun collectGarbage(): Unit = session.request("HeapProfiler.collectGarbage", Unit)

  public suspend fun disable(): Unit = session.request("HeapProfiler.disable", Unit)

  public suspend fun enable(): Unit = session.request("HeapProfiler.enable", Unit)

  public suspend fun getHeapObjectId(input: GetHeapObjectIdRequest): GetHeapObjectIdResponse =
      session.request("HeapProfiler.getHeapObjectId", input)

  public suspend fun getObjectByHeapObjectId(input: GetObjectByHeapObjectIdRequest):
      GetObjectByHeapObjectIdResponse = session.request("HeapProfiler.getObjectByHeapObjectId",
      input)

  public suspend fun getSamplingProfile(): GetSamplingProfileResponse =
      session.request("HeapProfiler.getSamplingProfile", Unit)

  public suspend fun startSampling(input: StartSamplingRequest): Unit =
      session.request("HeapProfiler.startSampling", input)

  public suspend fun startTrackingHeapObjects(input: StartTrackingHeapObjectsRequest): Unit =
      session.request("HeapProfiler.startTrackingHeapObjects", input)

  public suspend fun stopSampling(): StopSamplingResponse =
      session.request("HeapProfiler.stopSampling", Unit)

  public suspend fun stopTrackingHeapObjects(input: StopTrackingHeapObjectsRequest): Unit =
      session.request("HeapProfiler.stopTrackingHeapObjects", input)

  public suspend fun takeHeapSnapshot(input: TakeHeapSnapshotRequest): Unit =
      session.request("HeapProfiler.takeHeapSnapshot", input)
}
