package org.hildan.chrome.devtools.domains.indexeddb

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [IndexedDBDomain.clearObjectStore] command.
 */
@Serializable
public data class ClearObjectStoreRequest(
  /**
   * Security origin.
   */
  public val securityOrigin: String,
  /**
   * Database name.
   */
  public val databaseName: String,
  /**
   * Object store name.
   */
  public val objectStoreName: String
)

/**
 * Request object containing input parameters for the [IndexedDBDomain.deleteDatabase] command.
 */
@Serializable
public data class DeleteDatabaseRequest(
  /**
   * Security origin.
   */
  public val securityOrigin: String,
  /**
   * Database name.
   */
  public val databaseName: String
)

/**
 * Request object containing input parameters for the [IndexedDBDomain.deleteObjectStoreEntries]
 * command.
 */
@Serializable
public data class DeleteObjectStoreEntriesRequest(
  public val securityOrigin: String,
  public val databaseName: String,
  public val objectStoreName: String,
  /**
   * Range of entry keys to delete
   */
  public val keyRange: KeyRange
)

/**
 * Request object containing input parameters for the [IndexedDBDomain.requestData] command.
 */
@Serializable
public data class RequestDataRequest(
  /**
   * Security origin.
   */
  public val securityOrigin: String,
  /**
   * Database name.
   */
  public val databaseName: String,
  /**
   * Object store name.
   */
  public val objectStoreName: String,
  /**
   * Index name, empty string for object store data requests.
   */
  public val indexName: String,
  /**
   * Number of records to skip.
   */
  public val skipCount: Int,
  /**
   * Number of records to fetch.
   */
  public val pageSize: Int,
  /**
   * Key range.
   */
  public val keyRange: KeyRange? = null
)

/**
 * Response type for the [IndexedDBDomain.requestData] command.
 */
@Serializable
public data class RequestDataResponse(
  /**
   * Array of object store data entries.
   */
  public val objectStoreDataEntries: List<DataEntry>,
  /**
   * If true, there are more entries to fetch in the given range.
   */
  public val hasMore: Boolean
)

/**
 * Request object containing input parameters for the [IndexedDBDomain.getMetadata] command.
 */
@Serializable
public data class GetMetadataRequest(
  /**
   * Security origin.
   */
  public val securityOrigin: String,
  /**
   * Database name.
   */
  public val databaseName: String,
  /**
   * Object store name.
   */
  public val objectStoreName: String
)

/**
 * Response type for the [IndexedDBDomain.getMetadata] command.
 */
@Serializable
public data class GetMetadataResponse(
  /**
   * the entries count
   */
  public val entriesCount: Double,
  /**
   * the current value of key generator, to become the next inserted
   * key into the object store. Valid if objectStore.autoIncrement
   * is true.
   */
  public val keyGeneratorValue: Double
)

/**
 * Request object containing input parameters for the [IndexedDBDomain.requestDatabase] command.
 */
@Serializable
public data class RequestDatabaseRequest(
  /**
   * Security origin.
   */
  public val securityOrigin: String,
  /**
   * Database name.
   */
  public val databaseName: String
)

/**
 * Response type for the [IndexedDBDomain.requestDatabase] command.
 */
@Serializable
public data class RequestDatabaseResponse(
  /**
   * Database with an array of object stores.
   */
  public val databaseWithObjectStores: DatabaseWithObjectStores
)

/**
 * Request object containing input parameters for the [IndexedDBDomain.requestDatabaseNames]
 * command.
 */
@Serializable
public data class RequestDatabaseNamesRequest(
  /**
   * Security origin.
   */
  public val securityOrigin: String
)

/**
 * Response type for the [IndexedDBDomain.requestDatabaseNames] command.
 */
@Serializable
public data class RequestDatabaseNamesResponse(
  /**
   * Database names for origin.
   */
  public val databaseNames: List<String>
)

@ExperimentalChromeApi
public class IndexedDBDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Clears all entries from an object store.
   */
  public suspend fun clearObjectStore(input: ClearObjectStoreRequest): Unit =
      session.request("IndexedDB.clearObjectStore", input)

  /**
   * Deletes a database.
   */
  public suspend fun deleteDatabase(input: DeleteDatabaseRequest): Unit =
      session.request("IndexedDB.deleteDatabase", input)

  /**
   * Delete a range of entries from an object store
   */
  public suspend fun deleteObjectStoreEntries(input: DeleteObjectStoreEntriesRequest): Unit =
      session.request("IndexedDB.deleteObjectStoreEntries", input)

  /**
   * Disables events from backend.
   */
  public suspend fun disable(): Unit = session.request("IndexedDB.disable", Unit)

  /**
   * Enables events from backend.
   */
  public suspend fun enable(): Unit = session.request("IndexedDB.enable", Unit)

  /**
   * Requests data from object store or index.
   */
  public suspend fun requestData(input: RequestDataRequest): RequestDataResponse =
      session.request("IndexedDB.requestData", input)

  /**
   * Gets metadata of an object store
   */
  public suspend fun getMetadata(input: GetMetadataRequest): GetMetadataResponse =
      session.request("IndexedDB.getMetadata", input)

  /**
   * Requests database with given name in given frame.
   */
  public suspend fun requestDatabase(input: RequestDatabaseRequest): RequestDatabaseResponse =
      session.request("IndexedDB.requestDatabase", input)

  /**
   * Requests database names for given security origin.
   */
  public suspend fun requestDatabaseNames(input: RequestDatabaseNamesRequest):
      RequestDatabaseNamesResponse = session.request("IndexedDB.requestDatabaseNames", input)
}
