package org.hildan.chrome.devtools.domains.log

import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.log.events.LogEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession

/**
 * Request object containing input parameters for the [LogDomain.startViolationsReport] command.
 */
@Serializable
public data class StartViolationsReportRequest(
  /**
   * Configuration for violations.
   */
  public val config: List<ViolationSetting>
)

/**
 * Provides access to log entries.
 */
public class LogDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out LogEvent>> = mapOf(
      "Log.entryAdded" to serializer<LogEvent.EntryAddedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<LogEvent> = session.events(deserializersByEventName)

  /**
   * Issued when new message was logged.
   */
  public fun entryAdded(): Flow<LogEvent.EntryAddedEvent> = session.events("Log.entryAdded")

  /**
   * Clears the log.
   */
  public suspend fun clear(): Unit = session.request("Log.clear", Unit)

  /**
   * Disables log domain, prevents further log entries from being reported to the client.
   */
  public suspend fun disable(): Unit = session.request("Log.disable", Unit)

  /**
   * Enables log domain, sends the entries collected so far to the client by means of the
   * `entryAdded` notification.
   */
  public suspend fun enable(): Unit = session.request("Log.enable", Unit)

  /**
   * start violation reporting.
   */
  public suspend fun startViolationsReport(input: StartViolationsReportRequest): Unit =
      session.request("Log.startViolationsReport", input)

  /**
   * Stop violation reporting.
   */
  public suspend fun stopViolationsReport(): Unit = session.request("Log.stopViolationsReport",
      Unit)
}
