package org.hildan.chrome.devtools.domains.network.events

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.network.AuthChallenge
import org.hildan.chrome.devtools.domains.network.BlockedCookieWithReason
import org.hildan.chrome.devtools.domains.network.BlockedReason
import org.hildan.chrome.devtools.domains.network.BlockedSetCookieWithReason
import org.hildan.chrome.devtools.domains.network.ErrorReason
import org.hildan.chrome.devtools.domains.network.Headers
import org.hildan.chrome.devtools.domains.network.Initiator
import org.hildan.chrome.devtools.domains.network.InterceptionId
import org.hildan.chrome.devtools.domains.network.LoaderId
import org.hildan.chrome.devtools.domains.network.MonotonicTime
import org.hildan.chrome.devtools.domains.network.Request
import org.hildan.chrome.devtools.domains.network.RequestId
import org.hildan.chrome.devtools.domains.network.ResourcePriority
import org.hildan.chrome.devtools.domains.network.ResourceType
import org.hildan.chrome.devtools.domains.network.Response
import org.hildan.chrome.devtools.domains.network.SignedExchangeInfo
import org.hildan.chrome.devtools.domains.network.TimeSinceEpoch
import org.hildan.chrome.devtools.domains.network.WebSocketFrame
import org.hildan.chrome.devtools.domains.network.WebSocketRequest
import org.hildan.chrome.devtools.domains.network.WebSocketResponse
import org.hildan.chrome.devtools.domains.page.FrameId

@Serializable
public sealed class NetworkEvent {
  @Serializable
  public data class DataReceivedEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Data chunk length.
     */
    public val dataLength: Int,
    /**
     * Actual bytes received (might be less than dataLength for compressed encodings).
     */
    public val encodedDataLength: Int
  ) : NetworkEvent()

  @Serializable
  public data class EventSourceMessageReceivedEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Message type.
     */
    public val eventName: String,
    /**
     * Message identifier.
     */
    public val eventId: String,
    /**
     * Message content.
     */
    public val `data`: String
  ) : NetworkEvent()

  @Serializable
  public data class LoadingFailedEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Resource type.
     */
    public val type: ResourceType,
    /**
     * User friendly error message.
     */
    public val errorText: String,
    /**
     * True if loading was canceled.
     */
    public val canceled: Boolean? = null,
    /**
     * The reason why loading was blocked, if any.
     */
    public val blockedReason: BlockedReason? = null
  ) : NetworkEvent()

  @Serializable
  public data class LoadingFinishedEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Total number of bytes received for this request.
     */
    public val encodedDataLength: Double,
    /**
     * Set when 1) response was blocked by Cross-Origin Read Blocking and also
     * 2) this needs to be reported to the DevTools console.
     */
    public val shouldReportCorbBlocking: Boolean? = null
  ) : NetworkEvent()

  @Serializable
  public data class RequestInterceptedEvent(
    /**
     * Each request the page makes will have a unique id, however if any redirects are encountered
     * while processing that fetch, they will be reported with the same id as the original fetch.
     * Likewise if HTTP authentication is needed then the same fetch id will be used.
     */
    public val interceptionId: InterceptionId,
    public val request: Request,
    /**
     * The id of the frame that initiated the request.
     */
    public val frameId: FrameId,
    /**
     * How the requested resource will be used.
     */
    public val resourceType: ResourceType,
    /**
     * Whether this is a navigation request, which can abort the navigation completely.
     */
    public val isNavigationRequest: Boolean,
    /**
     * Set if the request is a navigation that will result in a download.
     * Only present after response is received from the server (i.e. HeadersReceived stage).
     */
    public val isDownload: Boolean? = null,
    /**
     * Redirect location, only sent if a redirect was intercepted.
     */
    public val redirectUrl: String? = null,
    /**
     * Details of the Authorization Challenge encountered. If this is set then
     * continueInterceptedRequest must contain an authChallengeResponse.
     */
    public val authChallenge: AuthChallenge? = null,
    /**
     * Response error if intercepted at response stage or if redirect occurred while intercepting
     * request.
     */
    public val responseErrorReason: ErrorReason? = null,
    /**
     * Response code if intercepted at response stage or if redirect occurred while intercepting
     * request or auth retry occurred.
     */
    public val responseStatusCode: Int? = null,
    /**
     * Response headers if intercepted at the response stage or if redirect occurred while
     * intercepting request or auth retry occurred.
     */
    public val responseHeaders: Headers? = null,
    /**
     * If the intercepted request had a corresponding requestWillBeSent event fired for it, then
     * this requestId will be the same as the requestId present in the requestWillBeSent event.
     */
    public val requestId: RequestId? = null
  ) : NetworkEvent()

  @Serializable
  public data class RequestServedFromCacheEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId
  ) : NetworkEvent()

  @Serializable
  public data class RequestWillBeSentEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Loader identifier. Empty string if the request is fetched from worker.
     */
    public val loaderId: LoaderId,
    /**
     * URL of the document this request is loaded for.
     */
    public val documentURL: String,
    /**
     * Request data.
     */
    public val request: Request,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Timestamp.
     */
    public val wallTime: TimeSinceEpoch,
    /**
     * Request initiator.
     */
    public val initiator: Initiator,
    /**
     * Redirect response data.
     */
    public val redirectResponse: Response? = null,
    /**
     * Type of this resource.
     */
    public val type: ResourceType? = null,
    /**
     * Frame identifier.
     */
    public val frameId: FrameId? = null,
    /**
     * Whether the request is initiated by a user gesture. Defaults to false.
     */
    public val hasUserGesture: Boolean? = null
  ) : NetworkEvent()

  @Serializable
  public data class ResourceChangedPriorityEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * New priority
     */
    public val newPriority: ResourcePriority,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime
  ) : NetworkEvent()

  @Serializable
  public data class SignedExchangeReceivedEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Information about the signed exchange response.
     */
    public val info: SignedExchangeInfo
  ) : NetworkEvent()

  @Serializable
  public data class ResponseReceivedEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Loader identifier. Empty string if the request is fetched from worker.
     */
    public val loaderId: LoaderId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * Resource type.
     */
    public val type: ResourceType,
    /**
     * Response data.
     */
    public val response: Response,
    /**
     * Frame identifier.
     */
    public val frameId: FrameId? = null
  ) : NetworkEvent()

  @Serializable
  public data class WebSocketClosedEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime
  ) : NetworkEvent()

  @Serializable
  public data class WebSocketCreatedEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * WebSocket request URL.
     */
    public val url: String,
    /**
     * Request initiator.
     */
    public val initiator: Initiator? = null
  ) : NetworkEvent()

  @Serializable
  public data class WebSocketFrameErrorEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * WebSocket error message.
     */
    public val errorMessage: String
  ) : NetworkEvent()

  @Serializable
  public data class WebSocketFrameReceivedEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * WebSocket response data.
     */
    public val response: WebSocketFrame
  ) : NetworkEvent()

  @Serializable
  public data class WebSocketFrameSentEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * WebSocket response data.
     */
    public val response: WebSocketFrame
  ) : NetworkEvent()

  @Serializable
  public data class WebSocketHandshakeResponseReceivedEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * WebSocket response data.
     */
    public val response: WebSocketResponse
  ) : NetworkEvent()

  @Serializable
  public data class WebSocketWillSendHandshakeRequestEvent(
    /**
     * Request identifier.
     */
    public val requestId: RequestId,
    /**
     * Timestamp.
     */
    public val timestamp: MonotonicTime,
    /**
     * UTC Timestamp.
     */
    public val wallTime: TimeSinceEpoch,
    /**
     * WebSocket request data.
     */
    public val request: WebSocketRequest
  ) : NetworkEvent()

  @Serializable
  public data class RequestWillBeSentExtraInfoEvent(
    /**
     * Request identifier. Used to match this information to an existing requestWillBeSent event.
     */
    public val requestId: RequestId,
    /**
     * A list of cookies potentially associated to the requested URL. This includes both cookies
     * sent with
     * the request and the ones not sent; the latter are distinguished by having blockedReason field
     * set.
     */
    public val associatedCookies: List<BlockedCookieWithReason>,
    /**
     * Raw request headers as they will be sent over the wire.
     */
    public val headers: Headers
  ) : NetworkEvent()

  @Serializable
  public data class ResponseReceivedExtraInfoEvent(
    /**
     * Request identifier. Used to match this information to another responseReceived event.
     */
    public val requestId: RequestId,
    /**
     * A list of cookies which were not stored from the response along with the corresponding
     * reasons for blocking. The cookies here may not be valid due to syntax errors, which
     * are represented by the invalid cookie line string instead of a proper cookie.
     */
    public val blockedCookies: List<BlockedSetCookieWithReason>,
    /**
     * Raw response headers as they were received over the wire.
     */
    public val headers: Headers,
    /**
     * Raw response header text as it was received over the wire. The raw text may not always be
     * available, such as in the case of HTTP/2 or QUIC.
     */
    public val headersText: String? = null
  ) : NetworkEvent()
}
