package org.hildan.chrome.devtools.domains.security

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.security.events.SecurityEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [SecurityDomain.setIgnoreCertificateErrors]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetIgnoreCertificateErrorsRequest(
  /**
   * If true, all certificate errors will be ignored.
   */
  public val ignore: Boolean
)

/**
 * Request object containing input parameters for the [SecurityDomain.handleCertificateError]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class HandleCertificateErrorRequest(
  /**
   * The ID of the event.
   */
  public val eventId: Int,
  /**
   * The action to take on the certificate error.
   */
  public val action: CertificateErrorAction
)

/**
 * Request object containing input parameters for the [SecurityDomain.setOverrideCertificateErrors]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class SetOverrideCertificateErrorsRequest(
  /**
   * If true, certificate errors will be overridden.
   */
  public val `override`: Boolean
)

/**
 * Security
 */
public class SecurityDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out SecurityEvent>> =
      mapOf(
      "Security.certificateError" to serializer<SecurityEvent.CertificateErrorEvent>(),
      "Security.visibleSecurityStateChanged" to
          serializer<SecurityEvent.VisibleSecurityStateChangedEvent>(),
      "Security.securityStateChanged" to serializer<SecurityEvent.SecurityStateChangedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<SecurityEvent> = session.events(deserializersByEventName)

  /**
   * There is a certificate error. If overriding certificate errors is enabled, then it should be
   * handled with the `handleCertificateError` command. Note: this event does not fire if the
   * certificate error has been allowed internally. Only one client per target should override
   * certificate errors at the same time.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public fun certificateError(): Flow<SecurityEvent.CertificateErrorEvent> =
      session.events("Security.certificateError")

  /**
   * The security state of the page changed.
   */
  @ExperimentalChromeApi
  public fun visibleSecurityStateChanged(): Flow<SecurityEvent.VisibleSecurityStateChangedEvent> =
      session.events("Security.visibleSecurityStateChanged")

  /**
   * The security state of the page changed.
   */
  public fun securityStateChanged(): Flow<SecurityEvent.SecurityStateChangedEvent> =
      session.events("Security.securityStateChanged")

  /**
   * Disables tracking security state changes.
   */
  public suspend fun disable(): Unit = session.request("Security.disable", Unit)

  /**
   * Enables tracking security state changes.
   */
  public suspend fun enable(): Unit = session.request("Security.enable", Unit)

  /**
   * Enable/disable whether all certificate errors should be ignored.
   */
  @ExperimentalChromeApi
  public suspend fun setIgnoreCertificateErrors(input: SetIgnoreCertificateErrorsRequest): Unit =
      session.request("Security.setIgnoreCertificateErrors", input)

  /**
   * Handles a certificate error that fired a certificateError event.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun handleCertificateError(input: HandleCertificateErrorRequest): Unit =
      session.request("Security.handleCertificateError", input)

  /**
   * Enable/disable overriding certificate errors. If enabled, all certificate error events need to
   * be handled by the DevTools client and should be answered with `handleCertificateError`
   * commands.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun setOverrideCertificateErrors(input: SetOverrideCertificateErrorsRequest): Unit
      = session.request("Security.setOverrideCertificateErrors", input)
}
