package org.hildan.chrome.devtools.domains.webaudio

import kotlin.Double
import kotlin.String
import kotlinx.serialization.Serializable

/**
 * An unique ID for a graph object (AudioContext, AudioNode, AudioParam) in Web Audio API
 */
public typealias GraphObjectId = String

/**
 * Enum of BaseAudioContext types
 */
@Serializable
public enum class ContextType {
  realtime,
  offline,
}

/**
 * Enum of AudioContextState from the spec
 */
@Serializable
public enum class ContextState {
  suspended,
  running,
  closed,
}

/**
 * Enum of AudioNode types
 */
public typealias NodeType = String

/**
 * Enum of AudioNode::ChannelCountMode from the spec
 */
@Serializable
public enum class ChannelCountMode {
  `clamped-max`,
  explicit,
  max,
}

/**
 * Enum of AudioNode::ChannelInterpretation from the spec
 */
@Serializable
public enum class ChannelInterpretation {
  discrete,
  speakers,
}

/**
 * Enum of AudioParam types
 */
public typealias ParamType = String

/**
 * Enum of AudioParam::AutomationRate from the spec
 */
@Serializable
public enum class AutomationRate {
  `a-rate`,
  `k-rate`,
}

/**
 * Fields in AudioContext that change in real-time.
 */
@Serializable
public data class ContextRealtimeData(
  /**
   * The current context time in second in BaseAudioContext.
   */
  public val currentTime: Double,
  /**
   * The time spent on rendering graph divided by render qunatum duration,
   * and multiplied by 100. 100 means the audio renderer reached the full
   * capacity and glitch may occur.
   */
  public val renderCapacity: Double,
  /**
   * A running mean of callback interval.
   */
  public val callbackIntervalMean: Double,
  /**
   * A running variance of callback interval.
   */
  public val callbackIntervalVariance: Double
)

/**
 * Protocol object for BaseAudioContext
 */
@Serializable
public data class BaseAudioContext(
  public val contextId: GraphObjectId,
  public val contextType: ContextType,
  public val contextState: ContextState,
  public val realtimeData: ContextRealtimeData? = null,
  /**
   * Platform-dependent callback buffer size.
   */
  public val callbackBufferSize: Double,
  /**
   * Number of output channels supported by audio hardware in use.
   */
  public val maxOutputChannelCount: Double,
  /**
   * Context sample rate.
   */
  public val sampleRate: Double
)

/**
 * Protocol object for AudioListener
 */
@Serializable
public data class AudioListener(
  public val listenerId: GraphObjectId,
  public val contextId: GraphObjectId
)

/**
 * Protocol object for AudioNode
 */
@Serializable
public data class AudioNode(
  public val nodeId: GraphObjectId,
  public val contextId: GraphObjectId,
  public val nodeType: NodeType,
  public val numberOfInputs: Double,
  public val numberOfOutputs: Double,
  public val channelCount: Double,
  public val channelCountMode: ChannelCountMode,
  public val channelInterpretation: ChannelInterpretation
)

/**
 * Protocol object for AudioParam
 */
@Serializable
public data class AudioParam(
  public val paramId: GraphObjectId,
  public val nodeId: GraphObjectId,
  public val contextId: GraphObjectId,
  public val paramType: ParamType,
  public val rate: AutomationRate,
  public val defaultValue: Double,
  public val minValue: Double,
  public val maxValue: Double
)
