package org.hildan.chrome.devtools.domains.domsnapshot

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.dom.PseudoType
import org.hildan.chrome.devtools.domains.dom.Rect
import org.hildan.chrome.devtools.domains.dom.ShadowRootType
import org.hildan.chrome.devtools.domains.domdebugger.EventListener
import org.hildan.chrome.devtools.domains.page.FrameId

/**
 * A Node in the DOM tree.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-DOMNode)
 */
@Serializable
public data class DOMNode(
  /**
   * `Node`'s nodeType.
   */
  public val nodeType: Int,
  /**
   * `Node`'s nodeName.
   */
  public val nodeName: String,
  /**
   * `Node`'s nodeValue.
   */
  public val nodeValue: String,
  /**
   * Only set for textarea elements, contains the text value.
   */
  public val textValue: String? = null,
  /**
   * Only set for input elements, contains the input's associated text value.
   */
  public val inputValue: String? = null,
  /**
   * Only set for radio and checkbox input elements, indicates if the element has been checked
   */
  public val inputChecked: Boolean? = null,
  /**
   * Only set for option elements, indicates if the element has been selected
   */
  public val optionSelected: Boolean? = null,
  /**
   * `Node`'s id, corresponds to DOM.Node.backendNodeId.
   */
  public val backendNodeId: BackendNodeId,
  /**
   * The indexes of the node's child nodes in the `domNodes` array returned by `getSnapshot`, if
   * any.
   */
  public val childNodeIndexes: List<Int>? = null,
  /**
   * Attributes of an `Element` node.
   */
  public val attributes: List<NameValue>? = null,
  /**
   * Indexes of pseudo elements associated with this node in the `domNodes` array returned by
   * `getSnapshot`, if any.
   */
  public val pseudoElementIndexes: List<Int>? = null,
  /**
   * The index of the node's related layout tree node in the `layoutTreeNodes` array returned by
   * `getSnapshot`, if any.
   */
  public val layoutNodeIndex: Int? = null,
  /**
   * Document URL that `Document` or `FrameOwner` node points to.
   */
  public val documentURL: String? = null,
  /**
   * Base URL that `Document` or `FrameOwner` node uses for URL completion.
   */
  public val baseURL: String? = null,
  /**
   * Only set for documents, contains the document's content language.
   */
  public val contentLanguage: String? = null,
  /**
   * Only set for documents, contains the document's character set encoding.
   */
  public val documentEncoding: String? = null,
  /**
   * `DocumentType` node's publicId.
   */
  public val publicId: String? = null,
  /**
   * `DocumentType` node's systemId.
   */
  public val systemId: String? = null,
  /**
   * Frame ID for frame owner elements and also for the document node.
   */
  public val frameId: FrameId? = null,
  /**
   * The index of a frame owner element's content document in the `domNodes` array returned by
   * `getSnapshot`, if any.
   */
  public val contentDocumentIndex: Int? = null,
  /**
   * Type of a pseudo element node.
   */
  public val pseudoType: PseudoType? = null,
  /**
   * Shadow root type.
   */
  public val shadowRootType: ShadowRootType? = null,
  /**
   * Whether this DOM node responds to mouse clicks. This includes nodes that have had click
   * event listeners attached via JavaScript as well as anchor tags that naturally navigate when
   * clicked.
   */
  public val isClickable: Boolean? = null,
  /**
   * Details of the node's event listeners, if any.
   */
  public val eventListeners: List<EventListener>? = null,
  /**
   * The selected url for nodes with a srcset attribute.
   */
  public val currentSourceURL: String? = null,
  /**
   * The url of the script (if any) that generates this node.
   */
  public val originURL: String? = null,
  /**
   * Scroll offsets, set when this node is a Document.
   */
  public val scrollOffsetX: Double? = null,
  public val scrollOffsetY: Double? = null
)

/**
 * Details of post layout rendered text positions. The exact layout should not be regarded as
 * stable and may change between versions.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-InlineTextBox)
 */
@Serializable
public data class InlineTextBox(
  /**
   * The bounding box in document coordinates. Note that scroll offset of the document is ignored.
   */
  public val boundingBox: Rect,
  /**
   * The starting index in characters, for this post layout textbox substring. Characters that
   * would be represented as a surrogate pair in UTF-16 have length 2.
   */
  public val startCharacterIndex: Int,
  /**
   * The number of characters in this post layout textbox substring. Characters that would be
   * represented as a surrogate pair in UTF-16 have length 2.
   */
  public val numCharacters: Int
)

/**
 * Details of an element in the DOM tree with a LayoutObject.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-LayoutTreeNode)
 */
@Serializable
public data class LayoutTreeNode(
  /**
   * The index of the related DOM node in the `domNodes` array returned by `getSnapshot`.
   */
  public val domNodeIndex: Int,
  /**
   * The bounding box in document coordinates. Note that scroll offset of the document is ignored.
   */
  public val boundingBox: Rect,
  /**
   * Contents of the LayoutText, if any.
   */
  public val layoutText: String? = null,
  /**
   * The post-layout inline text nodes, if any.
   */
  public val inlineTextNodes: List<InlineTextBox>? = null,
  /**
   * Index into the `computedStyles` array returned by `getSnapshot`.
   */
  public val styleIndex: Int? = null,
  /**
   * Global paint order index, which is determined by the stacking order of the nodes. Nodes
   * that are painted together will have the same index. Only provided if includePaintOrder in
   * getSnapshot was true.
   */
  public val paintOrder: Int? = null,
  /**
   * Set to true to indicate the element begins a new stacking context.
   */
  public val isStackingContext: Boolean? = null
)

/**
 * A subset of the full ComputedStyle as defined by the request whitelist.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-ComputedStyle)
 */
@Serializable
public data class ComputedStyle(
  /**
   * Name/value pairs of computed style properties.
   */
  public val properties: List<NameValue>
)

/**
 * A name/value pair.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-NameValue)
 */
@Serializable
public data class NameValue(
  /**
   * Attribute/property name.
   */
  public val name: String,
  /**
   * Attribute/property value.
   */
  public val value: String
)

/**
 * Index of the string in the strings table.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-StringIndex)
 */
public typealias StringIndex = Int

/**
 * Index of the string in the strings table.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-ArrayOfStrings)
 */
public typealias ArrayOfStrings = List<StringIndex>

/**
 * Data that is only present on rare nodes.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-RareStringData)
 */
@Serializable
public data class RareStringData(
  public val index: List<Int>,
  public val value: List<StringIndex>
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-RareBooleanData)
 */
@Serializable
public data class RareBooleanData(
  public val index: List<Int>
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-RareIntegerData)
 */
@Serializable
public data class RareIntegerData(
  public val index: List<Int>,
  public val value: List<Int>
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-Rectangle)
 */
public typealias Rectangle = List<Double>

/**
 * Document snapshot.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-DocumentSnapshot)
 */
@Serializable
public data class DocumentSnapshot(
  /**
   * Document URL that `Document` or `FrameOwner` node points to.
   */
  public val documentURL: StringIndex,
  /**
   * Document title.
   */
  public val title: StringIndex,
  /**
   * Base URL that `Document` or `FrameOwner` node uses for URL completion.
   */
  public val baseURL: StringIndex,
  /**
   * Contains the document's content language.
   */
  public val contentLanguage: StringIndex,
  /**
   * Contains the document's character set encoding.
   */
  public val encodingName: StringIndex,
  /**
   * `DocumentType` node's publicId.
   */
  public val publicId: StringIndex,
  /**
   * `DocumentType` node's systemId.
   */
  public val systemId: StringIndex,
  /**
   * Frame ID for frame owner elements and also for the document node.
   */
  public val frameId: StringIndex,
  /**
   * A table with dom nodes.
   */
  public val nodes: NodeTreeSnapshot,
  /**
   * The nodes in the layout tree.
   */
  public val layout: LayoutTreeSnapshot,
  /**
   * The post-layout inline text nodes.
   */
  public val textBoxes: TextBoxSnapshot,
  /**
   * Horizontal scroll offset.
   */
  public val scrollOffsetX: Double? = null,
  /**
   * Vertical scroll offset.
   */
  public val scrollOffsetY: Double? = null,
  /**
   * Document content width.
   */
  public val contentWidth: Double? = null,
  /**
   * Document content height.
   */
  public val contentHeight: Double? = null
)

/**
 * Table containing nodes.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-NodeTreeSnapshot)
 */
@Serializable
public data class NodeTreeSnapshot(
  /**
   * Parent node index.
   */
  public val parentIndex: List<Int>? = null,
  /**
   * `Node`'s nodeType.
   */
  public val nodeType: List<Int>? = null,
  /**
   * `Node`'s nodeName.
   */
  public val nodeName: List<StringIndex>? = null,
  /**
   * `Node`'s nodeValue.
   */
  public val nodeValue: List<StringIndex>? = null,
  /**
   * `Node`'s id, corresponds to DOM.Node.backendNodeId.
   */
  public val backendNodeId: List<BackendNodeId>? = null,
  /**
   * Attributes of an `Element` node. Flatten name, value pairs.
   */
  public val attributes: List<ArrayOfStrings>? = null,
  /**
   * Only set for textarea elements, contains the text value.
   */
  public val textValue: RareStringData? = null,
  /**
   * Only set for input elements, contains the input's associated text value.
   */
  public val inputValue: RareStringData? = null,
  /**
   * Only set for radio and checkbox input elements, indicates if the element has been checked
   */
  public val inputChecked: RareBooleanData? = null,
  /**
   * Only set for option elements, indicates if the element has been selected
   */
  public val optionSelected: RareBooleanData? = null,
  /**
   * The index of the document in the list of the snapshot documents.
   */
  public val contentDocumentIndex: RareIntegerData? = null,
  /**
   * Type of a pseudo element node.
   */
  public val pseudoType: RareStringData? = null,
  /**
   * Whether this DOM node responds to mouse clicks. This includes nodes that have had click
   * event listeners attached via JavaScript as well as anchor tags that naturally navigate when
   * clicked.
   */
  public val isClickable: RareBooleanData? = null,
  /**
   * The selected url for nodes with a srcset attribute.
   */
  public val currentSourceURL: RareStringData? = null,
  /**
   * The url of the script (if any) that generates this node.
   */
  public val originURL: RareStringData? = null
)

/**
 * Table of details of an element in the DOM tree with a LayoutObject.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-LayoutTreeSnapshot)
 */
@Serializable
public data class LayoutTreeSnapshot(
  /**
   * Index of the corresponding node in the `NodeTreeSnapshot` array returned by `captureSnapshot`.
   */
  public val nodeIndex: List<Int>,
  /**
   * Array of indexes specifying computed style strings, filtered according to the `computedStyles`
   * parameter passed to `captureSnapshot`.
   */
  public val styles: List<ArrayOfStrings>,
  /**
   * The absolute position bounding box.
   */
  public val bounds: List<Rectangle>,
  /**
   * Contents of the LayoutText, if any.
   */
  public val text: List<StringIndex>,
  /**
   * Stacking context information.
   */
  public val stackingContexts: RareBooleanData,
  /**
   * Global paint order index, which is determined by the stacking order of the nodes. Nodes
   * that are painted together will have the same index. Only provided if includePaintOrder in
   * captureSnapshot was true.
   */
  public val paintOrders: List<Int>? = null,
  /**
   * The offset rect of nodes. Only available when includeDOMRects is set to true
   */
  public val offsetRects: List<Rectangle>? = null,
  /**
   * The scroll rect of nodes. Only available when includeDOMRects is set to true
   */
  public val scrollRects: List<Rectangle>? = null,
  /**
   * The client rect of nodes. Only available when includeDOMRects is set to true
   */
  public val clientRects: List<Rectangle>? = null
)

/**
 * Table of details of the post layout rendered text positions. The exact layout should not be
 * regarded as
 * stable and may change between versions.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOMSnapshot/#type-TextBoxSnapshot)
 */
@Serializable
public data class TextBoxSnapshot(
  /**
   * Index of the layout tree node that owns this box collection.
   */
  public val layoutIndex: List<Int>,
  /**
   * The absolute position bounding box.
   */
  public val bounds: List<Rectangle>,
  /**
   * The starting index in characters, for this post layout textbox substring. Characters that
   * would be represented as a surrogate pair in UTF-16 have length 2.
   */
  public val start: List<Int>,
  /**
   * The number of characters in this post layout textbox substring. Characters that would be
   * represented as a surrogate pair in UTF-16 have length 2.
   */
  public val length: List<Int>
)
