package org.hildan.chrome.devtools.domains.page

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.network.LoaderId
import org.hildan.chrome.devtools.domains.network.ResourceType
import org.hildan.chrome.devtools.domains.network.TimeSinceEpoch
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Unique frame identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FrameId)
 */
public typealias FrameId = String

/**
 * Indicates whether a frame has been identified as an ad.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-AdFrameType)
 */
@ExperimentalChromeApi
@Serializable
public enum class AdFrameType {
  none,
  child,
  root,
}

/**
 * Indicates whether the frame is a secure context and why it is the case.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-SecureContextType)
 */
@ExperimentalChromeApi
@Serializable
public enum class SecureContextType {
  Secure,
  SecureLocalhost,
  InsecureScheme,
  InsecureAncestor,
}

/**
 * Indicates whether the frame is cross-origin isolated and why it is the case.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-CrossOriginIsolatedContextType)
 */
@ExperimentalChromeApi
@Serializable
public enum class CrossOriginIsolatedContextType {
  Isolated,
  NotIsolated,
  NotIsolatedFeatureDisabled,
}

/**
 * Information about the Frame on the page.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-Frame)
 */
@Serializable
public data class Frame(
  /**
   * Frame unique identifier.
   */
  public val id: FrameId,
  /**
   * Parent frame identifier.
   */
  public val parentId: String? = null,
  /**
   * Identifier of the loader associated with this frame.
   */
  public val loaderId: LoaderId,
  /**
   * Frame's name as specified in the tag.
   */
  public val name: String? = null,
  /**
   * Frame document's URL without fragment.
   */
  public val url: String,
  /**
   * Frame document's URL fragment including the '#'.
   */
  @ExperimentalChromeApi
  public val urlFragment: String? = null,
  /**
   * Frame document's registered domain, taking the public suffixes list into account.
   * Extracted from the Frame's url.
   * Example URLs: http://www.google.com/file.html -> "google.com"
   *               http://a.b.co.uk/file.html      -> "b.co.uk"
   */
  @ExperimentalChromeApi
  public val domainAndRegistry: String? = null,
  /**
   * Frame document's security origin.
   */
  public val securityOrigin: String,
  /**
   * Frame document's mimeType as determined by the browser.
   */
  public val mimeType: String,
  /**
   * If the frame failed to load, this contains the URL that could not be loaded. Note that unlike
   * url above, this URL may contain a fragment.
   */
  @ExperimentalChromeApi
  public val unreachableUrl: String? = null,
  /**
   * Indicates whether this frame was tagged as an ad.
   */
  @ExperimentalChromeApi
  public val adFrameType: AdFrameType? = null,
  /**
   * Indicates whether the main document is a secure context and explains why that is the case.
   */
  @ExperimentalChromeApi
  public val secureContextType: SecureContextType? = null,
  /**
   * Indicates whether this is a cross origin isolated context.
   */
  @ExperimentalChromeApi
  public val crossOriginIsolatedContextType: CrossOriginIsolatedContextType? = null
)

/**
 * Information about the Resource on the page.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FrameResource)
 */
@ExperimentalChromeApi
@Serializable
public data class FrameResource(
  /**
   * Resource URL.
   */
  public val url: String,
  /**
   * Type of this resource.
   */
  public val type: ResourceType,
  /**
   * Resource mimeType as determined by the browser.
   */
  public val mimeType: String,
  /**
   * last-modified timestamp as reported by server.
   */
  public val lastModified: TimeSinceEpoch? = null,
  /**
   * Resource content size.
   */
  public val contentSize: Double? = null,
  /**
   * True if the resource failed to load.
   */
  public val failed: Boolean? = null,
  /**
   * True if the resource was canceled during loading.
   */
  public val canceled: Boolean? = null
)

/**
 * Information about the Frame hierarchy along with their cached resources.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FrameResourceTree)
 */
@ExperimentalChromeApi
@Serializable
public data class FrameResourceTree(
  /**
   * Frame information for this tree item.
   */
  public val frame: Frame,
  /**
   * Child frames.
   */
  public val childFrames: List<FrameResourceTree>? = null,
  /**
   * Information about frame resources.
   */
  public val resources: List<FrameResource>
)

/**
 * Information about the Frame hierarchy.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FrameTree)
 */
@Serializable
public data class FrameTree(
  /**
   * Frame information for this tree item.
   */
  public val frame: Frame,
  /**
   * Child frames.
   */
  public val childFrames: List<FrameTree>? = null
)

/**
 * Unique script identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ScriptIdentifier)
 */
public typealias ScriptIdentifier = String

/**
 * Transition type.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-TransitionType)
 */
@Serializable
public enum class TransitionType {
  link,
  typed,
  address_bar,
  auto_bookmark,
  auto_subframe,
  manual_subframe,
  generated,
  auto_toplevel,
  form_submit,
  reload,
  keyword,
  keyword_generated,
  other,
}

/**
 * Navigation history entry.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-NavigationEntry)
 */
@Serializable
public data class NavigationEntry(
  /**
   * Unique id of the navigation history entry.
   */
  public val id: Int,
  /**
   * URL of the navigation history entry.
   */
  public val url: String,
  /**
   * URL that the user typed in the url bar.
   */
  public val userTypedURL: String,
  /**
   * Title of the navigation history entry.
   */
  public val title: String,
  /**
   * Transition type.
   */
  public val transitionType: TransitionType
)

/**
 * Screencast frame metadata.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ScreencastFrameMetadata)
 */
@ExperimentalChromeApi
@Serializable
public data class ScreencastFrameMetadata(
  /**
   * Top offset in DIP.
   */
  public val offsetTop: Double,
  /**
   * Page scale factor.
   */
  public val pageScaleFactor: Double,
  /**
   * Device screen width in DIP.
   */
  public val deviceWidth: Double,
  /**
   * Device screen height in DIP.
   */
  public val deviceHeight: Double,
  /**
   * Position of horizontal scroll in CSS pixels.
   */
  public val scrollOffsetX: Double,
  /**
   * Position of vertical scroll in CSS pixels.
   */
  public val scrollOffsetY: Double,
  /**
   * Frame swap timestamp.
   */
  public val timestamp: TimeSinceEpoch? = null
)

/**
 * Javascript dialog type.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-DialogType)
 */
@Serializable
public enum class DialogType {
  alert,
  confirm,
  prompt,
  beforeunload,
}

/**
 * Error while paring app manifest.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-AppManifestError)
 */
@Serializable
public data class AppManifestError(
  /**
   * Error message.
   */
  public val message: String,
  /**
   * If criticial, this is a non-recoverable parse error.
   */
  public val critical: Int,
  /**
   * Error line.
   */
  public val line: Int,
  /**
   * Error column.
   */
  public val column: Int
)

/**
 * Parsed app manifest properties.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-AppManifestParsedProperties)
 */
@ExperimentalChromeApi
@Serializable
public data class AppManifestParsedProperties(
  /**
   * Computed scope value
   */
  public val scope: String
)

/**
 * Layout viewport position and dimensions.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-LayoutViewport)
 */
@Serializable
public data class LayoutViewport(
  /**
   * Horizontal offset relative to the document (CSS pixels).
   */
  public val pageX: Int,
  /**
   * Vertical offset relative to the document (CSS pixels).
   */
  public val pageY: Int,
  /**
   * Width (CSS pixels), excludes scrollbar if present.
   */
  public val clientWidth: Int,
  /**
   * Height (CSS pixels), excludes scrollbar if present.
   */
  public val clientHeight: Int
)

/**
 * Visual viewport position, dimensions, and scale.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-VisualViewport)
 */
@Serializable
public data class VisualViewport(
  /**
   * Horizontal offset relative to the layout viewport (CSS pixels).
   */
  public val offsetX: Double,
  /**
   * Vertical offset relative to the layout viewport (CSS pixels).
   */
  public val offsetY: Double,
  /**
   * Horizontal offset relative to the document (CSS pixels).
   */
  public val pageX: Double,
  /**
   * Vertical offset relative to the document (CSS pixels).
   */
  public val pageY: Double,
  /**
   * Width (CSS pixels), excludes scrollbar if present.
   */
  public val clientWidth: Double,
  /**
   * Height (CSS pixels), excludes scrollbar if present.
   */
  public val clientHeight: Double,
  /**
   * Scale relative to the ideal viewport (size at width=device-width).
   */
  public val scale: Double,
  /**
   * Page zoom factor (CSS to device independent pixels ratio).
   */
  public val zoom: Double? = null
)

/**
 * Viewport for capturing screenshot.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-Viewport)
 */
@Serializable
public data class Viewport(
  /**
   * X offset in device independent pixels (dip).
   */
  public val x: Double,
  /**
   * Y offset in device independent pixels (dip).
   */
  public val y: Double,
  /**
   * Rectangle width in device independent pixels (dip).
   */
  public val width: Double,
  /**
   * Rectangle height in device independent pixels (dip).
   */
  public val height: Double,
  /**
   * Page scale factor.
   */
  public val scale: Double
)

/**
 * Generic font families collection.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FontFamilies)
 */
@ExperimentalChromeApi
@Serializable
public data class FontFamilies(
  /**
   * The standard font-family.
   */
  public val standard: String? = null,
  /**
   * The fixed font-family.
   */
  public val fixed: String? = null,
  /**
   * The serif font-family.
   */
  public val serif: String? = null,
  /**
   * The sansSerif font-family.
   */
  public val sansSerif: String? = null,
  /**
   * The cursive font-family.
   */
  public val cursive: String? = null,
  /**
   * The fantasy font-family.
   */
  public val fantasy: String? = null,
  /**
   * The pictograph font-family.
   */
  public val pictograph: String? = null
)

/**
 * Default font sizes.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-FontSizes)
 */
@ExperimentalChromeApi
@Serializable
public data class FontSizes(
  /**
   * Default standard font size.
   */
  public val standard: Int? = null,
  /**
   * Default fixed font size.
   */
  public val fixed: Int? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ClientNavigationReason)
 */
@ExperimentalChromeApi
@Serializable
public enum class ClientNavigationReason {
  formSubmissionGet,
  formSubmissionPost,
  httpHeaderRefresh,
  scriptInitiated,
  metaTagRefresh,
  pageBlockInterstitial,
  reload,
  anchorClick,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ClientNavigationDisposition)
 */
@ExperimentalChromeApi
@Serializable
public enum class ClientNavigationDisposition {
  currentTab,
  newTab,
  newWindow,
  download,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-InstallabilityErrorArgument)
 */
@ExperimentalChromeApi
@Serializable
public data class InstallabilityErrorArgument(
  /**
   * Argument name (e.g. name:'minimum-icon-size-in-pixels').
   */
  public val name: String,
  /**
   * Argument value (e.g. value:'64').
   */
  public val value: String
)

/**
 * The installability error
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-InstallabilityError)
 */
@ExperimentalChromeApi
@Serializable
public data class InstallabilityError(
  /**
   * The error id (e.g. 'manifest-missing-suitable-icon').
   */
  public val errorId: String,
  /**
   * The list of error arguments (e.g. {name:'minimum-icon-size-in-pixels', value:'64'}).
   */
  public val errorArguments: List<InstallabilityErrorArgument>
)

/**
 * The referring-policy used for the navigation.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#type-ReferrerPolicy)
 */
@ExperimentalChromeApi
@Serializable
public enum class ReferrerPolicy {
  noReferrer,
  noReferrerWhenDowngrade,
  origin,
  originWhenCrossOrigin,
  sameOrigin,
  strictOrigin,
  strictOriginWhenCrossOrigin,
  unsafeUrl,
}
