package org.hildan.chrome.devtools.domains.page.events

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.network.LoaderId
import org.hildan.chrome.devtools.domains.network.MonotonicTime
import org.hildan.chrome.devtools.domains.page.ClientNavigationDisposition
import org.hildan.chrome.devtools.domains.page.ClientNavigationReason
import org.hildan.chrome.devtools.domains.page.DialogType
import org.hildan.chrome.devtools.domains.page.Frame
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.domains.page.ScreencastFrameMetadata
import org.hildan.chrome.devtools.domains.runtime.StackTrace
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

@Serializable
public sealed class PageEvent {
  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-domContentEventFired)
   */
  @Serializable
  public data class DomContentEventFiredEvent(
    public val timestamp: MonotonicTime
  ) : PageEvent()

  /**
   * Emitted only when `page.interceptFileChooser` is enabled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-fileChooserOpened)
   */
  @Serializable
  public data class FileChooserOpenedEvent(
    /**
     * Id of the frame containing input node.
     */
    @ExperimentalChromeApi
    public val frameId: FrameId? = null,
    /**
     * Input node id.
     */
    @ExperimentalChromeApi
    public val backendNodeId: BackendNodeId? = null,
    /**
     * Input mode.
     */
    public val mode: String
  ) : PageEvent()

  /**
   * Fired when frame has been attached to its parent.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameAttached)
   */
  @Serializable
  public data class FrameAttachedEvent(
    /**
     * Id of the frame that has been attached.
     */
    public val frameId: FrameId,
    /**
     * Parent frame identifier.
     */
    public val parentFrameId: FrameId,
    /**
     * JavaScript stack trace of when frame was attached, only set if frame initiated from script.
     */
    public val stack: StackTrace? = null
  ) : PageEvent()

  /**
   * Fired when frame no longer has a scheduled navigation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameClearedScheduledNavigation)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @Serializable
  public data class FrameClearedScheduledNavigationEvent(
    /**
     * Id of the frame that has cleared its scheduled navigation.
     */
    public val frameId: FrameId
  ) : PageEvent()

  /**
   * Fired when frame has been detached from its parent.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameDetached)
   */
  @Serializable
  public data class FrameDetachedEvent(
    /**
     * Id of the frame that has been detached.
     */
    public val frameId: FrameId
  ) : PageEvent()

  /**
   * Fired once navigation of the frame has completed. Frame is now associated with the new loader.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameNavigated)
   */
  @Serializable
  public data class FrameNavigatedEvent(
    /**
     * Frame object.
     */
    public val frame: Frame
  ) : PageEvent()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameResized)
   */
  @ExperimentalChromeApi
  @Serializable
  public object FrameResizedEvent : PageEvent()

  /**
   * Fired when a renderer-initiated navigation is requested.
   * Navigation may still be cancelled after the event is issued.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameRequestedNavigation)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class FrameRequestedNavigationEvent(
    /**
     * Id of the frame that is being navigated.
     */
    public val frameId: FrameId,
    /**
     * The reason for the navigation.
     */
    public val reason: ClientNavigationReason,
    /**
     * The destination URL for the requested navigation.
     */
    public val url: String,
    /**
     * The disposition for the navigation.
     */
    public val disposition: ClientNavigationDisposition
  ) : PageEvent()

  /**
   * Fired when frame schedules a potential navigation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameScheduledNavigation)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @Serializable
  public data class FrameScheduledNavigationEvent(
    /**
     * Id of the frame that has scheduled a navigation.
     */
    public val frameId: FrameId,
    /**
     * Delay (in seconds) until the navigation is scheduled to begin. The navigation is not
     * guaranteed to start.
     */
    public val delay: Double,
    /**
     * The reason for the navigation.
     */
    public val reason: ClientNavigationReason,
    /**
     * The destination URL for the scheduled navigation.
     */
    public val url: String
  ) : PageEvent()

  /**
   * Fired when frame has started loading.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameStartedLoading)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class FrameStartedLoadingEvent(
    /**
     * Id of the frame that has started loading.
     */
    public val frameId: FrameId
  ) : PageEvent()

  /**
   * Fired when frame has stopped loading.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-frameStoppedLoading)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class FrameStoppedLoadingEvent(
    /**
     * Id of the frame that has stopped loading.
     */
    public val frameId: FrameId
  ) : PageEvent()

  /**
   * Fired when page is about to start a download.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-downloadWillBegin)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class DownloadWillBeginEvent(
    /**
     * Id of the frame that caused download to begin.
     */
    public val frameId: FrameId,
    /**
     * Global unique identifier of the download.
     */
    public val guid: String,
    /**
     * URL of the resource being downloaded.
     */
    public val url: String,
    /**
     * Suggested file name of the resource (the actual name of the file saved on disk may differ).
     */
    public val suggestedFilename: String
  ) : PageEvent()

  /**
   * Fired when download makes progress. Last call has |done| == true.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-downloadProgress)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class DownloadProgressEvent(
    /**
     * Global unique identifier of the download.
     */
    public val guid: String,
    /**
     * Total expected bytes to download.
     */
    public val totalBytes: Double,
    /**
     * Total bytes received.
     */
    public val receivedBytes: Double,
    /**
     * Download status.
     */
    public val state: String
  ) : PageEvent()

  /**
   * Fired when interstitial page was hidden
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-interstitialHidden)
   */
  @Serializable
  public object InterstitialHiddenEvent : PageEvent()

  /**
   * Fired when interstitial page was shown
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-interstitialShown)
   */
  @Serializable
  public object InterstitialShownEvent : PageEvent()

  /**
   * Fired when a JavaScript initiated dialog (alert, confirm, prompt, or onbeforeunload) has been
   * closed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-javascriptDialogClosed)
   */
  @Serializable
  public data class JavascriptDialogClosedEvent(
    /**
     * Whether dialog was confirmed.
     */
    public val result: Boolean,
    /**
     * User input in case of prompt.
     */
    public val userInput: String
  ) : PageEvent()

  /**
   * Fired when a JavaScript initiated dialog (alert, confirm, prompt, or onbeforeunload) is about
   * to
   * open.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-javascriptDialogOpening)
   */
  @Serializable
  public data class JavascriptDialogOpeningEvent(
    /**
     * Frame url.
     */
    public val url: String,
    /**
     * Message that will be displayed by the dialog.
     */
    public val message: String,
    /**
     * Dialog type.
     */
    public val type: DialogType,
    /**
     * True iff browser is capable showing or acting on the given dialog. When browser has no
     * dialog handler for given target, calling alert while Page domain is engaged will stall
     * the page execution. Execution can be resumed via calling Page.handleJavaScriptDialog.
     */
    public val hasBrowserHandler: Boolean,
    /**
     * Default dialog prompt.
     */
    public val defaultPrompt: String? = null
  ) : PageEvent()

  /**
   * Fired for top level page lifecycle events such as navigation, load, paint, etc.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-lifecycleEvent)
   */
  @Serializable
  public data class LifecycleEventEvent(
    /**
     * Id of the frame.
     */
    public val frameId: FrameId,
    /**
     * Loader identifier. Empty string if the request is fetched from worker.
     */
    public val loaderId: LoaderId,
    public val name: String,
    public val timestamp: MonotonicTime
  ) : PageEvent()

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-loadEventFired)
   */
  @Serializable
  public data class LoadEventFiredEvent(
    public val timestamp: MonotonicTime
  ) : PageEvent()

  /**
   * Fired when same-document navigation happens, e.g. due to history API usage or anchor
   * navigation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-navigatedWithinDocument)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class NavigatedWithinDocumentEvent(
    /**
     * Id of the frame.
     */
    public val frameId: FrameId,
    /**
     * Frame's new url.
     */
    public val url: String
  ) : PageEvent()

  /**
   * Compressed image data requested by the `startScreencast`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-screencastFrame)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class ScreencastFrameEvent(
    /**
     * Base64-encoded compressed image.
     */
    public val `data`: String,
    /**
     * Screencast frame metadata.
     */
    public val metadata: ScreencastFrameMetadata,
    /**
     * Frame number.
     */
    public val sessionId: Int
  ) : PageEvent()

  /**
   * Fired when the page with currently enabled screencast was shown or hidden `.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-screencastVisibilityChanged)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class ScreencastVisibilityChangedEvent(
    /**
     * True if the page is visible.
     */
    public val visible: Boolean
  ) : PageEvent()

  /**
   * Fired when a new window is going to be opened, via window.open(), link click, form submission,
   * etc.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-windowOpen)
   */
  @Serializable
  public data class WindowOpenEvent(
    /**
     * The URL for the new window.
     */
    public val url: String,
    /**
     * Window name.
     */
    public val windowName: String,
    /**
     * An array of enabled window features.
     */
    public val windowFeatures: List<String>,
    /**
     * Whether or not it was triggered by user gesture.
     */
    public val userGesture: Boolean
  ) : PageEvent()

  /**
   * Issued for every compilation cache generated. Is only available
   * if Page.setGenerateCompilationCache is enabled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Page/#event-compilationCacheProduced)
   */
  @ExperimentalChromeApi
  @Serializable
  public data class CompilationCacheProducedEvent(
    public val url: String,
    /**
     * Base64-encoded data
     */
    public val `data`: String
  ) : PageEvent()
}
