package org.hildan.chrome.devtools.domains.tethering

import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.tethering.events.TetheringEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [TetheringDomain.bind] command.
 */
@Serializable
public data class BindRequest(
  /**
   * Port number to bind.
   */
  public val port: Int
)

/**
 * Request object containing input parameters for the [TetheringDomain.unbind] command.
 */
@Serializable
public data class UnbindRequest(
  /**
   * Port number to unbind.
   */
  public val port: Int
)

/**
 * The Tethering domain defines methods and events for browser port binding.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tethering)
 */
@ExperimentalChromeApi
public class TetheringDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out TetheringEvent>> =
      mapOf(
      "Tethering.accepted" to serializer<TetheringEvent.AcceptedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<TetheringEvent> = session.events(deserializersByEventName)

  /**
   * Informs that port was successfully bound and got a specified connection id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tethering/#event-accepted)
   */
  public fun accepted(): Flow<TetheringEvent.AcceptedEvent> = session.events("Tethering.accepted")

  /**
   * Request browser port binding.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tethering/#method-bind)
   */
  public suspend fun bind(input: BindRequest): Unit = session.request("Tethering.bind", input)

  /**
   * Request browser port unbinding.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tethering/#method-unbind)
   */
  public suspend fun unbind(input: UnbindRequest): Unit = session.request("Tethering.unbind", input)
}
