package org.hildan.chrome.devtools.domains.browser

import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#type-BrowserContextID)
 */
@ExperimentalChromeApi
public typealias BrowserContextID = String

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#type-WindowID)
 */
@ExperimentalChromeApi
public typealias WindowID = Int

/**
 * The state of the browser window.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#type-WindowState)
 */
@ExperimentalChromeApi
@Serializable
public enum class WindowState {
  normal,
  minimized,
  maximized,
  fullscreen,
}

/**
 * Browser window bounds information
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#type-Bounds)
 */
@ExperimentalChromeApi
@Serializable
public data class Bounds(
  /**
   * The offset from the left edge of the screen to the window in pixels.
   */
  public val left: Int? = null,
  /**
   * The offset from the top edge of the screen to the window in pixels.
   */
  public val top: Int? = null,
  /**
   * The window width in pixels.
   */
  public val width: Int? = null,
  /**
   * The window height in pixels.
   */
  public val height: Int? = null,
  /**
   * The window state. Default to normal.
   */
  public val windowState: WindowState? = null
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#type-PermissionType)
 */
@ExperimentalChromeApi
@Serializable
public enum class PermissionType {
  accessibilityEvents,
  audioCapture,
  backgroundSync,
  backgroundFetch,
  clipboardReadWrite,
  clipboardSanitizedWrite,
  durableStorage,
  flash,
  geolocation,
  midi,
  midiSysex,
  nfc,
  notifications,
  paymentHandler,
  periodicBackgroundSync,
  protectedMediaIdentifier,
  sensors,
  videoCapture,
  videoCapturePanTiltZoom,
  idleDetection,
  wakeLockScreen,
  wakeLockSystem,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#type-PermissionSetting)
 */
@ExperimentalChromeApi
@Serializable
public enum class PermissionSetting {
  granted,
  denied,
  prompt,
}

/**
 * Definition of PermissionDescriptor defined in the Permissions API:
 * https://w3c.github.io/permissions/#dictdef-permissiondescriptor.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#type-PermissionDescriptor)
 */
@ExperimentalChromeApi
@Serializable
public data class PermissionDescriptor(
  /**
   * Name of permission.
   * See
   * https://cs.chromium.org/chromium/src/third_party/blink/renderer/modules/permissions/permission_descriptor.idl
   * for valid permission names.
   */
  public val name: String,
  /**
   * For "midi" permission, may also specify sysex control.
   */
  public val sysex: Boolean? = null,
  /**
   * For "push" permission, may specify userVisibleOnly.
   * Note that userVisibleOnly = true is the only currently supported type.
   */
  public val userVisibleOnly: Boolean? = null,
  /**
   * For "clipboard" permission, may specify allowWithoutSanitization.
   */
  public val allowWithoutSanitization: Boolean? = null,
  /**
   * For "camera" permission, may specify panTiltZoom.
   */
  public val panTiltZoom: Boolean? = null
)

/**
 * Browser command ids used by executeBrowserCommand.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#type-BrowserCommandId)
 */
@ExperimentalChromeApi
@Serializable
public enum class BrowserCommandId {
  openTabSearch,
  closeTabSearch,
}

/**
 * Chrome histogram bucket.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#type-Bucket)
 */
@ExperimentalChromeApi
@Serializable
public data class Bucket(
  /**
   * Minimum value (inclusive).
   */
  public val low: Int,
  /**
   * Maximum value (exclusive).
   */
  public val high: Int,
  /**
   * Number of samples.
   */
  public val count: Int
)

/**
 * Chrome histogram.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Browser/#type-Histogram)
 */
@ExperimentalChromeApi
@Serializable
public data class Histogram(
  /**
   * Name.
   */
  public val name: String,
  /**
   * Sum of sample values.
   */
  public val sum: Int,
  /**
   * Total number of samples.
   */
  public val count: Int,
  /**
   * Buckets.
   */
  public val buckets: List<Bucket>
)
