package org.hildan.chrome.devtools.domains.dom

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.dom.events.DOMEvent
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.domains.runtime.ExecutionContextId
import org.hildan.chrome.devtools.domains.runtime.RemoteObject
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.domains.runtime.StackTrace
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [DOMDomain.collectClassNamesFromSubtree]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class CollectClassNamesFromSubtreeRequest(
  /**
   * Id of the node to collect class names.
   */
  public val nodeId: NodeId
)

/**
 * Response type for the [DOMDomain.collectClassNamesFromSubtree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CollectClassNamesFromSubtreeResponse(
  /**
   * Class name list.
   */
  public val classNames: List<String>
)

/**
 * Request object containing input parameters for the [DOMDomain.copyTo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CopyToRequest(
  /**
   * Id of the node to copy.
   */
  public val nodeId: NodeId,
  /**
   * Id of the element to drop the copy into.
   */
  public val targetNodeId: NodeId,
  /**
   * Drop the copy before this node (if absent, the copy becomes the last child of
   * `targetNodeId`).
   */
  public val insertBeforeNodeId: NodeId? = null
)

/**
 * Response type for the [DOMDomain.copyTo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CopyToResponse(
  /**
   * Id of the node clone.
   */
  public val nodeId: NodeId
)

/**
 * Request object containing input parameters for the [DOMDomain.describeNode] command.
 */
@Serializable
public data class DescribeNodeRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
   * entire subtree or provide an integer larger than 0.
   */
  public val depth: Int? = null,
  /**
   * Whether or not iframes and shadow roots should be traversed when returning the subtree
   * (default is false).
   */
  public val pierce: Boolean? = null
)

/**
 * Response type for the [DOMDomain.describeNode] command.
 */
@Serializable
public data class DescribeNodeResponse(
  /**
   * Node description.
   */
  public val node: Node
)

/**
 * Request object containing input parameters for the [DOMDomain.scrollIntoViewIfNeeded] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ScrollIntoViewIfNeededRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * The rect to be scrolled into view, relative to the node's border box, in CSS pixels.
   * When omitted, center of the node will be used, similar to Element.scrollIntoView.
   */
  public val rect: Rect? = null
)

/**
 * Request object containing input parameters for the [DOMDomain.discardSearchResults] command.
 */
@Serializable
@ExperimentalChromeApi
public data class DiscardSearchResultsRequest(
  /**
   * Unique search session identifier.
   */
  public val searchId: String
)

/**
 * Request object containing input parameters for the [DOMDomain.focus] command.
 */
@Serializable
public data class FocusRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null
)

/**
 * Request object containing input parameters for the [DOMDomain.getAttributes] command.
 */
@Serializable
public data class GetAttributesRequest(
  /**
   * Id of the node to retrieve attibutes for.
   */
  public val nodeId: NodeId
)

/**
 * Response type for the [DOMDomain.getAttributes] command.
 */
@Serializable
public data class GetAttributesResponse(
  /**
   * An interleaved array of node attribute names and values.
   */
  public val attributes: List<String>
)

/**
 * Request object containing input parameters for the [DOMDomain.getBoxModel] command.
 */
@Serializable
public data class GetBoxModelRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null
)

/**
 * Response type for the [DOMDomain.getBoxModel] command.
 */
@Serializable
public data class GetBoxModelResponse(
  /**
   * Box model for the node.
   */
  public val model: BoxModel
)

/**
 * Request object containing input parameters for the [DOMDomain.getContentQuads] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetContentQuadsRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null
)

/**
 * Response type for the [DOMDomain.getContentQuads] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetContentQuadsResponse(
  /**
   * Quads that describe node layout relative to viewport.
   */
  public val quads: List<Quad>
)

/**
 * Request object containing input parameters for the [DOMDomain.getDocument] command.
 */
@Serializable
public data class GetDocumentRequest(
  /**
   * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
   * entire subtree or provide an integer larger than 0.
   */
  public val depth: Int? = null,
  /**
   * Whether or not iframes and shadow roots should be traversed when returning the subtree
   * (default is false).
   */
  public val pierce: Boolean? = null
)

/**
 * Response type for the [DOMDomain.getDocument] command.
 */
@Serializable
public data class GetDocumentResponse(
  /**
   * Resulting node.
   */
  public val root: Node
)

/**
 * Request object containing input parameters for the [DOMDomain.getFlattenedDocument] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class GetFlattenedDocumentRequest(
  /**
   * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
   * entire subtree or provide an integer larger than 0.
   */
  public val depth: Int? = null,
  /**
   * Whether or not iframes and shadow roots should be traversed when returning the subtree
   * (default is false).
   */
  public val pierce: Boolean? = null
)

/**
 * Response type for the [DOMDomain.getFlattenedDocument] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class GetFlattenedDocumentResponse(
  /**
   * Resulting node.
   */
  public val nodes: List<Node>
)

/**
 * Request object containing input parameters for the [DOMDomain.getNodesForSubtreeByStyle] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetNodesForSubtreeByStyleRequest(
  /**
   * Node ID pointing to the root of a subtree.
   */
  public val nodeId: NodeId,
  /**
   * The style to filter nodes by (includes nodes if any of properties matches).
   */
  public val computedStyles: List<CSSComputedStyleProperty>,
  /**
   * Whether or not iframes and shadow roots in the same target should be traversed when returning
   * the
   * results (default is false).
   */
  public val pierce: Boolean? = null
)

/**
 * Response type for the [DOMDomain.getNodesForSubtreeByStyle] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetNodesForSubtreeByStyleResponse(
  /**
   * Resulting nodes.
   */
  public val nodeIds: List<NodeId>
)

/**
 * Request object containing input parameters for the [DOMDomain.getNodeForLocation] command.
 */
@Serializable
public data class GetNodeForLocationRequest(
  /**
   * X coordinate.
   */
  public val x: Int,
  /**
   * Y coordinate.
   */
  public val y: Int,
  /**
   * False to skip to the nearest non-UA shadow root ancestor (default: false).
   */
  public val includeUserAgentShadowDOM: Boolean? = null,
  /**
   * Whether to ignore pointer-events: none on elements and hit test them.
   */
  public val ignorePointerEventsNone: Boolean? = null
)

/**
 * Response type for the [DOMDomain.getNodeForLocation] command.
 */
@Serializable
public data class GetNodeForLocationResponse(
  /**
   * Resulting node.
   */
  public val backendNodeId: BackendNodeId,
  /**
   * Frame this node belongs to.
   */
  public val frameId: FrameId,
  /**
   * Id of the node at given coordinates, only when enabled and requested document.
   */
  public val nodeId: NodeId? = null
)

/**
 * Request object containing input parameters for the [DOMDomain.getOuterHTML] command.
 */
@Serializable
public data class GetOuterHTMLRequest(
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null
)

/**
 * Response type for the [DOMDomain.getOuterHTML] command.
 */
@Serializable
public data class GetOuterHTMLResponse(
  /**
   * Outer HTML markup.
   */
  public val outerHTML: String
)

/**
 * Request object containing input parameters for the [DOMDomain.getRelayoutBoundary] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetRelayoutBoundaryRequest(
  /**
   * Id of the node.
   */
  public val nodeId: NodeId
)

/**
 * Response type for the [DOMDomain.getRelayoutBoundary] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetRelayoutBoundaryResponse(
  /**
   * Relayout boundary node id for the given node.
   */
  public val nodeId: NodeId
)

/**
 * Request object containing input parameters for the [DOMDomain.getSearchResults] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSearchResultsRequest(
  /**
   * Unique search session identifier.
   */
  public val searchId: String,
  /**
   * Start index of the search result to be returned.
   */
  public val fromIndex: Int,
  /**
   * End index of the search result to be returned.
   */
  public val toIndex: Int
)

/**
 * Response type for the [DOMDomain.getSearchResults] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSearchResultsResponse(
  /**
   * Ids of the search result nodes.
   */
  public val nodeIds: List<NodeId>
)

/**
 * Request object containing input parameters for the [DOMDomain.moveTo] command.
 */
@Serializable
public data class MoveToRequest(
  /**
   * Id of the node to move.
   */
  public val nodeId: NodeId,
  /**
   * Id of the element to drop the moved node into.
   */
  public val targetNodeId: NodeId,
  /**
   * Drop node before this one (if absent, the moved node becomes the last child of
   * `targetNodeId`).
   */
  public val insertBeforeNodeId: NodeId? = null
)

/**
 * Response type for the [DOMDomain.moveTo] command.
 */
@Serializable
public data class MoveToResponse(
  /**
   * New id of the moved node.
   */
  public val nodeId: NodeId
)

/**
 * Request object containing input parameters for the [DOMDomain.performSearch] command.
 */
@Serializable
@ExperimentalChromeApi
public data class PerformSearchRequest(
  /**
   * Plain text or query selector or XPath search query.
   */
  public val query: String,
  /**
   * True to search in user agent shadow DOM.
   */
  public val includeUserAgentShadowDOM: Boolean? = null
)

/**
 * Response type for the [DOMDomain.performSearch] command.
 */
@Serializable
@ExperimentalChromeApi
public data class PerformSearchResponse(
  /**
   * Unique search session identifier.
   */
  public val searchId: String,
  /**
   * Number of search results.
   */
  public val resultCount: Int
)

/**
 * Request object containing input parameters for the [DOMDomain.pushNodeByPathToFrontend] command.
 */
@Serializable
@ExperimentalChromeApi
public data class PushNodeByPathToFrontendRequest(
  /**
   * Path to node in the proprietary format.
   */
  public val path: String
)

/**
 * Response type for the [DOMDomain.pushNodeByPathToFrontend] command.
 */
@Serializable
@ExperimentalChromeApi
public data class PushNodeByPathToFrontendResponse(
  /**
   * Id of the node for given path.
   */
  public val nodeId: NodeId
)

/**
 * Request object containing input parameters for the [DOMDomain.pushNodesByBackendIdsToFrontend]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class PushNodesByBackendIdsToFrontendRequest(
  /**
   * The array of backend node ids.
   */
  public val backendNodeIds: List<BackendNodeId>
)

/**
 * Response type for the [DOMDomain.pushNodesByBackendIdsToFrontend] command.
 */
@Serializable
@ExperimentalChromeApi
public data class PushNodesByBackendIdsToFrontendResponse(
  /**
   * The array of ids of pushed nodes that correspond to the backend ids specified in
   * backendNodeIds.
   */
  public val nodeIds: List<NodeId>
)

/**
 * Request object containing input parameters for the [DOMDomain.querySelector] command.
 */
@Serializable
public data class QuerySelectorRequest(
  /**
   * Id of the node to query upon.
   */
  public val nodeId: NodeId,
  /**
   * Selector string.
   */
  public val selector: String
)

/**
 * Response type for the [DOMDomain.querySelector] command.
 */
@Serializable
public data class QuerySelectorResponse(
  /**
   * Query selector result.
   */
  public val nodeId: NodeId
)

/**
 * Request object containing input parameters for the [DOMDomain.querySelectorAll] command.
 */
@Serializable
public data class QuerySelectorAllRequest(
  /**
   * Id of the node to query upon.
   */
  public val nodeId: NodeId,
  /**
   * Selector string.
   */
  public val selector: String
)

/**
 * Response type for the [DOMDomain.querySelectorAll] command.
 */
@Serializable
public data class QuerySelectorAllResponse(
  /**
   * Query selector result.
   */
  public val nodeIds: List<NodeId>
)

/**
 * Request object containing input parameters for the [DOMDomain.removeAttribute] command.
 */
@Serializable
public data class RemoveAttributeRequest(
  /**
   * Id of the element to remove attribute from.
   */
  public val nodeId: NodeId,
  /**
   * Name of the attribute to remove.
   */
  public val name: String
)

/**
 * Request object containing input parameters for the [DOMDomain.removeNode] command.
 */
@Serializable
public data class RemoveNodeRequest(
  /**
   * Id of the node to remove.
   */
  public val nodeId: NodeId
)

/**
 * Request object containing input parameters for the [DOMDomain.requestChildNodes] command.
 */
@Serializable
public data class RequestChildNodesRequest(
  /**
   * Id of the node to get children for.
   */
  public val nodeId: NodeId,
  /**
   * The maximum depth at which children should be retrieved, defaults to 1. Use -1 for the
   * entire subtree or provide an integer larger than 0.
   */
  public val depth: Int? = null,
  /**
   * Whether or not iframes and shadow roots should be traversed when returning the sub-tree
   * (default is false).
   */
  public val pierce: Boolean? = null
)

/**
 * Request object containing input parameters for the [DOMDomain.requestNode] command.
 */
@Serializable
public data class RequestNodeRequest(
  /**
   * JavaScript object id to convert into node.
   */
  public val objectId: RemoteObjectId
)

/**
 * Response type for the [DOMDomain.requestNode] command.
 */
@Serializable
public data class RequestNodeResponse(
  /**
   * Node id for given object.
   */
  public val nodeId: NodeId
)

/**
 * Request object containing input parameters for the [DOMDomain.resolveNode] command.
 */
@Serializable
public data class ResolveNodeRequest(
  /**
   * Id of the node to resolve.
   */
  public val nodeId: NodeId? = null,
  /**
   * Backend identifier of the node to resolve.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * Symbolic group name that can be used to release multiple objects.
   */
  public val objectGroup: String? = null,
  /**
   * Execution context in which to resolve the node.
   */
  public val executionContextId: ExecutionContextId? = null
)

/**
 * Response type for the [DOMDomain.resolveNode] command.
 */
@Serializable
public data class ResolveNodeResponse(
  /**
   * JavaScript object wrapper for given node.
   */
  public val `object`: RemoteObject
)

/**
 * Request object containing input parameters for the [DOMDomain.setAttributeValue] command.
 */
@Serializable
public data class SetAttributeValueRequest(
  /**
   * Id of the element to set attribute for.
   */
  public val nodeId: NodeId,
  /**
   * Attribute name.
   */
  public val name: String,
  /**
   * Attribute value.
   */
  public val value: String
)

/**
 * Request object containing input parameters for the [DOMDomain.setAttributesAsText] command.
 */
@Serializable
public data class SetAttributesAsTextRequest(
  /**
   * Id of the element to set attributes for.
   */
  public val nodeId: NodeId,
  /**
   * Text with a number of attributes. Will parse this text using HTML parser.
   */
  public val text: String,
  /**
   * Attribute name to replace with new attributes derived from text in case text parsed
   * successfully.
   */
  public val name: String? = null
)

/**
 * Request object containing input parameters for the [DOMDomain.setFileInputFiles] command.
 */
@Serializable
public data class SetFileInputFilesRequest(
  /**
   * Array of file paths to set.
   */
  public val files: List<String>,
  /**
   * Identifier of the node.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId? = null
)

/**
 * Request object containing input parameters for the [DOMDomain.setNodeStackTracesEnabled] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetNodeStackTracesEnabledRequest(
  /**
   * Enable or disable.
   */
  public val enable: Boolean
)

/**
 * Request object containing input parameters for the [DOMDomain.getNodeStackTraces] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetNodeStackTracesRequest(
  /**
   * Id of the node to get stack traces for.
   */
  public val nodeId: NodeId
)

/**
 * Response type for the [DOMDomain.getNodeStackTraces] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetNodeStackTracesResponse(
  /**
   * Creation stack trace, if available.
   */
  public val creation: StackTrace? = null
)

/**
 * Request object containing input parameters for the [DOMDomain.getFileInfo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFileInfoRequest(
  /**
   * JavaScript object id of the node wrapper.
   */
  public val objectId: RemoteObjectId
)

/**
 * Response type for the [DOMDomain.getFileInfo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFileInfoResponse(
  public val path: String
)

/**
 * Request object containing input parameters for the [DOMDomain.setInspectedNode] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetInspectedNodeRequest(
  /**
   * DOM node id to be accessible by means of $x command line API.
   */
  public val nodeId: NodeId
)

/**
 * Request object containing input parameters for the [DOMDomain.setNodeName] command.
 */
@Serializable
public data class SetNodeNameRequest(
  /**
   * Id of the node to set name for.
   */
  public val nodeId: NodeId,
  /**
   * New node's name.
   */
  public val name: String
)

/**
 * Response type for the [DOMDomain.setNodeName] command.
 */
@Serializable
public data class SetNodeNameResponse(
  /**
   * New node's id.
   */
  public val nodeId: NodeId
)

/**
 * Request object containing input parameters for the [DOMDomain.setNodeValue] command.
 */
@Serializable
public data class SetNodeValueRequest(
  /**
   * Id of the node to set value for.
   */
  public val nodeId: NodeId,
  /**
   * New node's value.
   */
  public val value: String
)

/**
 * Request object containing input parameters for the [DOMDomain.setOuterHTML] command.
 */
@Serializable
public data class SetOuterHTMLRequest(
  /**
   * Id of the node to set markup for.
   */
  public val nodeId: NodeId,
  /**
   * Outer HTML markup to set.
   */
  public val outerHTML: String
)

/**
 * Request object containing input parameters for the [DOMDomain.getFrameOwner] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFrameOwnerRequest(
  public val frameId: FrameId
)

/**
 * Response type for the [DOMDomain.getFrameOwner] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFrameOwnerResponse(
  /**
   * Resulting node.
   */
  public val backendNodeId: BackendNodeId,
  /**
   * Id of the node at given coordinates, only when enabled and requested document.
   */
  public val nodeId: NodeId? = null
)

/**
 * This domain exposes DOM read/write operations. Each DOM Node is represented with its mirror
 * object
 * that has an `id`. This `id` can be used to get additional information on the Node, resolve it
 * into
 * the JavaScript object wrapper, etc. It is important that client receives DOM events only for the
 * nodes that are known to the client. Backend keeps track of the nodes that were sent to the client
 * and never sends the same node twice. It is client's responsibility to collect information about
 * the nodes that were sent to the client.<p>Note that `iframe` owner elements will return
 * corresponding document elements as their child nodes.</p>
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM)
 */
public class DOMDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out DOMEvent>> = mapOf(
      "DOM.attributeModified" to serializer<DOMEvent.AttributeModifiedEvent>(),
      "DOM.attributeRemoved" to serializer<DOMEvent.AttributeRemovedEvent>(),
      "DOM.characterDataModified" to serializer<DOMEvent.CharacterDataModifiedEvent>(),
      "DOM.childNodeCountUpdated" to serializer<DOMEvent.ChildNodeCountUpdatedEvent>(),
      "DOM.childNodeInserted" to serializer<DOMEvent.ChildNodeInsertedEvent>(),
      "DOM.childNodeRemoved" to serializer<DOMEvent.ChildNodeRemovedEvent>(),
      "DOM.distributedNodesUpdated" to serializer<DOMEvent.DistributedNodesUpdatedEvent>(),
      "DOM.documentUpdated" to serializer<DOMEvent.DocumentUpdatedEvent>(),
      "DOM.inlineStyleInvalidated" to serializer<DOMEvent.InlineStyleInvalidatedEvent>(),
      "DOM.pseudoElementAdded" to serializer<DOMEvent.PseudoElementAddedEvent>(),
      "DOM.pseudoElementRemoved" to serializer<DOMEvent.PseudoElementRemovedEvent>(),
      "DOM.setChildNodes" to serializer<DOMEvent.SetChildNodesEvent>(),
      "DOM.shadowRootPopped" to serializer<DOMEvent.ShadowRootPoppedEvent>(),
      "DOM.shadowRootPushed" to serializer<DOMEvent.ShadowRootPushedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<DOMEvent> = session.events(deserializersByEventName)

  /**
   * Fired when `Element`'s attribute is modified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-attributeModified)
   */
  public fun attributeModified(): Flow<DOMEvent.AttributeModifiedEvent> =
      session.events("DOM.attributeModified")

  /**
   * Fired when `Element`'s attribute is removed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-attributeRemoved)
   */
  public fun attributeRemoved(): Flow<DOMEvent.AttributeRemovedEvent> =
      session.events("DOM.attributeRemoved")

  /**
   * Mirrors `DOMCharacterDataModified` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-characterDataModified)
   */
  public fun characterDataModified(): Flow<DOMEvent.CharacterDataModifiedEvent> =
      session.events("DOM.characterDataModified")

  /**
   * Fired when `Container`'s child node count has changed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-childNodeCountUpdated)
   */
  public fun childNodeCountUpdated(): Flow<DOMEvent.ChildNodeCountUpdatedEvent> =
      session.events("DOM.childNodeCountUpdated")

  /**
   * Mirrors `DOMNodeInserted` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-childNodeInserted)
   */
  public fun childNodeInserted(): Flow<DOMEvent.ChildNodeInsertedEvent> =
      session.events("DOM.childNodeInserted")

  /**
   * Mirrors `DOMNodeRemoved` event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-childNodeRemoved)
   */
  public fun childNodeRemoved(): Flow<DOMEvent.ChildNodeRemovedEvent> =
      session.events("DOM.childNodeRemoved")

  /**
   * Called when distrubution is changed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-distributedNodesUpdated)
   */
  @ExperimentalChromeApi
  public fun distributedNodesUpdated(): Flow<DOMEvent.DistributedNodesUpdatedEvent> =
      session.events("DOM.distributedNodesUpdated")

  /**
   * Fired when `Document` has been totally updated. Node ids are no longer valid.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-documentUpdated)
   */
  public fun documentUpdated(): Flow<DOMEvent.DocumentUpdatedEvent> =
      session.events("DOM.documentUpdated")

  /**
   * Fired when `Element`'s inline style is modified via a CSS property modification.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-inlineStyleInvalidated)
   */
  @ExperimentalChromeApi
  public fun inlineStyleInvalidated(): Flow<DOMEvent.InlineStyleInvalidatedEvent> =
      session.events("DOM.inlineStyleInvalidated")

  /**
   * Called when a pseudo element is added to an element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-pseudoElementAdded)
   */
  @ExperimentalChromeApi
  public fun pseudoElementAdded(): Flow<DOMEvent.PseudoElementAddedEvent> =
      session.events("DOM.pseudoElementAdded")

  /**
   * Called when a pseudo element is removed from an element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-pseudoElementRemoved)
   */
  @ExperimentalChromeApi
  public fun pseudoElementRemoved(): Flow<DOMEvent.PseudoElementRemovedEvent> =
      session.events("DOM.pseudoElementRemoved")

  /**
   * Fired when backend wants to provide client with the missing DOM structure. This happens upon
   * most of the calls requesting node ids.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-setChildNodes)
   */
  public fun setChildNodes(): Flow<DOMEvent.SetChildNodesEvent> =
      session.events("DOM.setChildNodes")

  /**
   * Called when shadow root is popped from the element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-shadowRootPopped)
   */
  @ExperimentalChromeApi
  public fun shadowRootPopped(): Flow<DOMEvent.ShadowRootPoppedEvent> =
      session.events("DOM.shadowRootPopped")

  /**
   * Called when shadow root is pushed into the element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#event-shadowRootPushed)
   */
  @ExperimentalChromeApi
  public fun shadowRootPushed(): Flow<DOMEvent.ShadowRootPushedEvent> =
      session.events("DOM.shadowRootPushed")

  /**
   * Collects class names for the node with given id and all of it's child nodes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-collectClassNamesFromSubtree)
   */
  @ExperimentalChromeApi
  public suspend fun collectClassNamesFromSubtree(input: CollectClassNamesFromSubtreeRequest):
      CollectClassNamesFromSubtreeResponse = session.request("DOM.collectClassNamesFromSubtree",
      input)

  /**
   * Creates a deep copy of the specified node and places it into the target container before the
   * given anchor.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-copyTo)
   */
  @ExperimentalChromeApi
  public suspend fun copyTo(input: CopyToRequest): CopyToResponse = session.request("DOM.copyTo",
      input)

  /**
   * Describes node given its id, does not require domain to be enabled. Does not start tracking any
   * objects, can be used for automation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-describeNode)
   */
  public suspend fun describeNode(input: DescribeNodeRequest): DescribeNodeResponse =
      session.request("DOM.describeNode", input)

  /**
   * Scrolls the specified rect of the given node into view if not already visible.
   * Note: exactly one between nodeId, backendNodeId and objectId should be passed
   * to identify the node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-scrollIntoViewIfNeeded)
   */
  @ExperimentalChromeApi
  public suspend fun scrollIntoViewIfNeeded(input: ScrollIntoViewIfNeededRequest): Unit =
      session.request("DOM.scrollIntoViewIfNeeded", input)

  /**
   * Disables DOM agent for the given page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("DOM.disable", Unit)

  /**
   * Discards search results from the session with the given id. `getSearchResults` should no longer
   * be called for that search.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-discardSearchResults)
   */
  @ExperimentalChromeApi
  public suspend fun discardSearchResults(input: DiscardSearchResultsRequest): Unit =
      session.request("DOM.discardSearchResults", input)

  /**
   * Enables DOM agent for the given page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("DOM.enable", Unit)

  /**
   * Focuses the given element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-focus)
   */
  public suspend fun focus(input: FocusRequest): Unit = session.request("DOM.focus", input)

  /**
   * Returns attributes for the specified node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getAttributes)
   */
  public suspend fun getAttributes(input: GetAttributesRequest): GetAttributesResponse =
      session.request("DOM.getAttributes", input)

  /**
   * Returns boxes for the given node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getBoxModel)
   */
  public suspend fun getBoxModel(input: GetBoxModelRequest): GetBoxModelResponse =
      session.request("DOM.getBoxModel", input)

  /**
   * Returns quads that describe node position on the page. This method
   * might return multiple quads for inline nodes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getContentQuads)
   */
  @ExperimentalChromeApi
  public suspend fun getContentQuads(input: GetContentQuadsRequest): GetContentQuadsResponse =
      session.request("DOM.getContentQuads", input)

  /**
   * Returns the root DOM node (and optionally the subtree) to the caller.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getDocument)
   */
  public suspend fun getDocument(input: GetDocumentRequest): GetDocumentResponse =
      session.request("DOM.getDocument", input)

  /**
   * Returns the root DOM node (and optionally the subtree) to the caller.
   * Deprecated, as it is not designed to work well with the rest of the DOM agent.
   * Use DOMSnapshot.captureSnapshot instead.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getFlattenedDocument)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun getFlattenedDocument(input: GetFlattenedDocumentRequest):
      GetFlattenedDocumentResponse = session.request("DOM.getFlattenedDocument", input)

  /**
   * Finds nodes with a given computed style in a subtree.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getNodesForSubtreeByStyle)
   */
  @ExperimentalChromeApi
  public suspend fun getNodesForSubtreeByStyle(input: GetNodesForSubtreeByStyleRequest):
      GetNodesForSubtreeByStyleResponse = session.request("DOM.getNodesForSubtreeByStyle", input)

  /**
   * Returns node id at given location. Depending on whether DOM domain is enabled, nodeId is
   * either returned or not.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getNodeForLocation)
   */
  public suspend fun getNodeForLocation(input: GetNodeForLocationRequest):
      GetNodeForLocationResponse = session.request("DOM.getNodeForLocation", input)

  /**
   * Returns node's HTML markup.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getOuterHTML)
   */
  public suspend fun getOuterHTML(input: GetOuterHTMLRequest): GetOuterHTMLResponse =
      session.request("DOM.getOuterHTML", input)

  /**
   * Returns the id of the nearest ancestor that is a relayout boundary.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getRelayoutBoundary)
   */
  @ExperimentalChromeApi
  public suspend fun getRelayoutBoundary(input: GetRelayoutBoundaryRequest):
      GetRelayoutBoundaryResponse = session.request("DOM.getRelayoutBoundary", input)

  /**
   * Returns search results from given `fromIndex` to given `toIndex` from the search with the given
   * identifier.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getSearchResults)
   */
  @ExperimentalChromeApi
  public suspend fun getSearchResults(input: GetSearchResultsRequest): GetSearchResultsResponse =
      session.request("DOM.getSearchResults", input)

  /**
   * Hides any highlight.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-hideHighlight)
   */
  public suspend fun hideHighlight(): Unit = session.request("DOM.hideHighlight", Unit)

  /**
   * Highlights DOM node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-highlightNode)
   */
  public suspend fun highlightNode(): Unit = session.request("DOM.highlightNode", Unit)

  /**
   * Highlights given rectangle.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-highlightRect)
   */
  public suspend fun highlightRect(): Unit = session.request("DOM.highlightRect", Unit)

  /**
   * Marks last undoable state.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-markUndoableState)
   */
  @ExperimentalChromeApi
  public suspend fun markUndoableState(): Unit = session.request("DOM.markUndoableState", Unit)

  /**
   * Moves node into the new container, places it before the given anchor.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-moveTo)
   */
  public suspend fun moveTo(input: MoveToRequest): MoveToResponse = session.request("DOM.moveTo",
      input)

  /**
   * Searches for a given string in the DOM tree. Use `getSearchResults` to access search results or
   * `cancelSearch` to end this search session.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-performSearch)
   */
  @ExperimentalChromeApi
  public suspend fun performSearch(input: PerformSearchRequest): PerformSearchResponse =
      session.request("DOM.performSearch", input)

  /**
   * Requests that the node is sent to the caller given its path. // FIXME, use XPath
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-pushNodeByPathToFrontend)
   */
  @ExperimentalChromeApi
  public suspend fun pushNodeByPathToFrontend(input: PushNodeByPathToFrontendRequest):
      PushNodeByPathToFrontendResponse = session.request("DOM.pushNodeByPathToFrontend", input)

  /**
   * Requests that a batch of nodes is sent to the caller given their backend node ids.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-pushNodesByBackendIdsToFrontend)
   */
  @ExperimentalChromeApi
  public suspend fun pushNodesByBackendIdsToFrontend(input: PushNodesByBackendIdsToFrontendRequest):
      PushNodesByBackendIdsToFrontendResponse =
      session.request("DOM.pushNodesByBackendIdsToFrontend", input)

  /**
   * Executes `querySelector` on a given node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-querySelector)
   */
  public suspend fun querySelector(input: QuerySelectorRequest): QuerySelectorResponse =
      session.request("DOM.querySelector", input)

  /**
   * Executes `querySelectorAll` on a given node.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-querySelectorAll)
   */
  public suspend fun querySelectorAll(input: QuerySelectorAllRequest): QuerySelectorAllResponse =
      session.request("DOM.querySelectorAll", input)

  /**
   * Re-does the last undone action.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-redo)
   */
  @ExperimentalChromeApi
  public suspend fun redo(): Unit = session.request("DOM.redo", Unit)

  /**
   * Removes attribute with given name from an element with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-removeAttribute)
   */
  public suspend fun removeAttribute(input: RemoveAttributeRequest): Unit =
      session.request("DOM.removeAttribute", input)

  /**
   * Removes node with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-removeNode)
   */
  public suspend fun removeNode(input: RemoveNodeRequest): Unit = session.request("DOM.removeNode",
      input)

  /**
   * Requests that children of the node with given id are returned to the caller in form of
   * `setChildNodes` events where not only immediate children are retrieved, but all children down
   * to
   * the specified depth.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-requestChildNodes)
   */
  public suspend fun requestChildNodes(input: RequestChildNodesRequest): Unit =
      session.request("DOM.requestChildNodes", input)

  /**
   * Requests that the node is sent to the caller given the JavaScript node object reference. All
   * nodes that form the path from the node to the root are also sent to the client as a series of
   * `setChildNodes` notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-requestNode)
   */
  public suspend fun requestNode(input: RequestNodeRequest): RequestNodeResponse =
      session.request("DOM.requestNode", input)

  /**
   * Resolves the JavaScript node object for a given NodeId or BackendNodeId.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-resolveNode)
   */
  public suspend fun resolveNode(input: ResolveNodeRequest): ResolveNodeResponse =
      session.request("DOM.resolveNode", input)

  /**
   * Sets attribute for an element with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setAttributeValue)
   */
  public suspend fun setAttributeValue(input: SetAttributeValueRequest): Unit =
      session.request("DOM.setAttributeValue", input)

  /**
   * Sets attributes on element with given id. This method is useful when user edits some existing
   * attribute value and types in several attribute name/value pairs.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setAttributesAsText)
   */
  public suspend fun setAttributesAsText(input: SetAttributesAsTextRequest): Unit =
      session.request("DOM.setAttributesAsText", input)

  /**
   * Sets files for the given file input element.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setFileInputFiles)
   */
  public suspend fun setFileInputFiles(input: SetFileInputFilesRequest): Unit =
      session.request("DOM.setFileInputFiles", input)

  /**
   * Sets if stack traces should be captured for Nodes. See `Node.getNodeStackTraces`. Default is
   * disabled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setNodeStackTracesEnabled)
   */
  @ExperimentalChromeApi
  public suspend fun setNodeStackTracesEnabled(input: SetNodeStackTracesEnabledRequest): Unit =
      session.request("DOM.setNodeStackTracesEnabled", input)

  /**
   * Gets stack traces associated with a Node. As of now, only provides stack trace for Node
   * creation.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getNodeStackTraces)
   */
  @ExperimentalChromeApi
  public suspend fun getNodeStackTraces(input: GetNodeStackTracesRequest):
      GetNodeStackTracesResponse = session.request("DOM.getNodeStackTraces", input)

  /**
   * Returns file information for the given
   * File wrapper.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getFileInfo)
   */
  @ExperimentalChromeApi
  public suspend fun getFileInfo(input: GetFileInfoRequest): GetFileInfoResponse =
      session.request("DOM.getFileInfo", input)

  /**
   * Enables console to refer to the node with given id via $x (see Command Line API for more
   * details
   * $x functions).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setInspectedNode)
   */
  @ExperimentalChromeApi
  public suspend fun setInspectedNode(input: SetInspectedNodeRequest): Unit =
      session.request("DOM.setInspectedNode", input)

  /**
   * Sets node name for a node with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setNodeName)
   */
  public suspend fun setNodeName(input: SetNodeNameRequest): SetNodeNameResponse =
      session.request("DOM.setNodeName", input)

  /**
   * Sets node value for a node with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setNodeValue)
   */
  public suspend fun setNodeValue(input: SetNodeValueRequest): Unit =
      session.request("DOM.setNodeValue", input)

  /**
   * Sets node HTML markup, returns new node id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-setOuterHTML)
   */
  public suspend fun setOuterHTML(input: SetOuterHTMLRequest): Unit =
      session.request("DOM.setOuterHTML", input)

  /**
   * Undoes the last performed action.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-undo)
   */
  @ExperimentalChromeApi
  public suspend fun undo(): Unit = session.request("DOM.undo", Unit)

  /**
   * Returns iframe node that owns iframe with the given domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/DOM/#method-getFrameOwner)
   */
  @ExperimentalChromeApi
  public suspend fun getFrameOwner(input: GetFrameOwnerRequest): GetFrameOwnerResponse =
      session.request("DOM.getFrameOwner", input)
}
