package org.hildan.chrome.devtools.domains.headlessexperimental

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.String
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.headlessexperimental.events.HeadlessExperimentalEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [HeadlessExperimentalDomain.beginFrame]
 * command.
 */
@Serializable
public data class BeginFrameRequest(
  /**
   * Timestamp of this BeginFrame in Renderer TimeTicks (milliseconds of uptime). If not set,
   * the current time will be used.
   */
  public val frameTimeTicks: Double? = null,
  /**
   * The interval between BeginFrames that is reported to the compositor, in milliseconds.
   * Defaults to a 60 frames/second interval, i.e. about 16.666 milliseconds.
   */
  public val interval: Double? = null,
  /**
   * Whether updates should not be committed and drawn onto the display. False by default. If
   * true, only side effects of the BeginFrame will be run, such as layout and animations, but
   * any visual updates may not be visible on the display or in screenshots.
   */
  public val noDisplayUpdates: Boolean? = null,
  /**
   * If set, a screenshot of the frame will be captured and returned in the response. Otherwise,
   * no screenshot will be captured. Note that capturing a screenshot can fail, for example,
   * during renderer initialization. In such a case, no screenshot data will be returned.
   */
  public val screenshot: ScreenshotParams? = null
)

/**
 * Response type for the [HeadlessExperimentalDomain.beginFrame] command.
 */
@Serializable
public data class BeginFrameResponse(
  /**
   * Whether the BeginFrame resulted in damage and, thus, a new frame was committed to the
   * display. Reported for diagnostic uses, may be removed in the future.
   */
  public val hasDamage: Boolean,
  /**
   * Base64-encoded image data of the screenshot, if one was requested and successfully taken.
   */
  public val screenshotData: String? = null
)

/**
 * This domain provides experimental commands only supported in headless mode.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeadlessExperimental)
 */
@ExperimentalChromeApi
public class HeadlessExperimentalDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out
      HeadlessExperimentalEvent>> = mapOf(
      "HeadlessExperimental.needsBeginFramesChanged" to
          serializer<HeadlessExperimentalEvent.NeedsBeginFramesChangedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<HeadlessExperimentalEvent> = session.events(deserializersByEventName)

  /**
   * Issued when the target starts or stops needing BeginFrames.
   * Deprecated. Issue beginFrame unconditionally instead and use result from
   * beginFrame to detect whether the frames were suppressed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeadlessExperimental/#event-needsBeginFramesChanged)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public fun needsBeginFramesChanged(): Flow<HeadlessExperimentalEvent.NeedsBeginFramesChangedEvent>
      = session.events("HeadlessExperimental.needsBeginFramesChanged")

  /**
   * Sends a BeginFrame to the target and returns when the frame was completed. Optionally captures
   * a
   * screenshot from the resulting frame. Requires that the target was created with enabled
   * BeginFrameControl. Designed for use with --run-all-compositor-stages-before-draw, see also
   * https://goo.gl/3zHXhB for more background.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeadlessExperimental/#method-beginFrame)
   */
  public suspend fun beginFrame(input: BeginFrameRequest): BeginFrameResponse =
      session.request("HeadlessExperimental.beginFrame", input)

  /**
   * Disables headless events for the target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeadlessExperimental/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("HeadlessExperimental.disable", Unit)

  /**
   * Enables headless events for the target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/HeadlessExperimental/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("HeadlessExperimental.enable", Unit)
}
