package org.hildan.chrome.devtools.domains.inspector

import kotlin.String
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.inspector.events.InspectorEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector)
 */
@ExperimentalChromeApi
public class InspectorDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out InspectorEvent>> =
      mapOf(
      "Inspector.detached" to serializer<InspectorEvent.DetachedEvent>(),
      "Inspector.targetCrashed" to serializer<InspectorEvent.TargetCrashedEvent>(),
      "Inspector.targetReloadedAfterCrash" to
          serializer<InspectorEvent.TargetReloadedAfterCrashEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<InspectorEvent> = session.events(deserializersByEventName)

  /**
   * Fired when remote debugging connection is about to be terminated. Contains detach reason.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector/#event-detached)
   */
  public fun detached(): Flow<InspectorEvent.DetachedEvent> = session.events("Inspector.detached")

  /**
   * Fired when debugging target has crashed
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector/#event-targetCrashed)
   */
  public fun targetCrashed(): Flow<InspectorEvent.TargetCrashedEvent> =
      session.events("Inspector.targetCrashed")

  /**
   * Fired when debugging target has reloaded after crash
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector/#event-targetReloadedAfterCrash)
   */
  public fun targetReloadedAfterCrash(): Flow<InspectorEvent.TargetReloadedAfterCrashEvent> =
      session.events("Inspector.targetReloadedAfterCrash")

  /**
   * Disables inspector domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Inspector.disable", Unit)

  /**
   * Enables inspector domain notifications.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Inspector/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("Inspector.enable", Unit)
}
